/**
  ******************************************************************************
  * @file    lcm32f06x_tim.c
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   This file provides all the tim firmware functions.
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
/* Includes ------------------------------------------------------------------*/
#include "lcm32f06x_tim.h"

/***************** TIM Function Definition **************/

static void TI1_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                       uint32_t TIM_ICFilter);
static void TI2_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                uint32_t TIM_ICFilter);
static void TI3_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                       uint32_t TIM_ICFilter);
static void TI4_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                       uint32_t TIM_ICFilter);
static void TI5_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                       uint32_t TIM_ICFilter);

/**
  * @name   TIM_DTS_Cfg
  * @brief  EN: DTS clock enable for digital filters.
  *         CN: ˲ DTS ʱʹܡ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval None
  */
void TIM_DTS_Cfg(TIM_TypeDef *TIMx)
{
  TIMx->CR1 |= TIM_CR1_FTE_Msk; // 1<<16
}

/**
  * @name   TIM_ETRE_Enable
  * @brief  EN: ETR input detection/filtering enable.
  *         CN: ETR /˲ʹܡ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval None
  */
void TIM_ETRE_Enable(TIM_TypeDef *TIMx)
{
  TIMx->CR1 |= TIM_CR1_ETRE_Msk; // 1<<17
}

/**
  * @name   TIM_TI1E_Set
  * @brief  EN: TI1 input detection/filtering enable.
  *         CN: TI1 /˲ʹܡ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval None
  */
void TIM_TI1E_Set(TIM_TypeDef *TIMx)
{
  TIMx->CR1 |= TIM_CR1_TI1E_Msk; // 1<<18
}

/**
  * @name   TIM_TI2E_Set
  * @brief  EN: TI2 input detection/filtering enable.
  *         CN: TI2 /˲ʹܡ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @retval None
  */
void TIM_TI2E_Set(TIM_TypeDef *TIMx)
{
  TIMx->CR1 |= TIM_CR1_TI2E_Msk; // 1<<19
}

/**
  * @name   TIM_TI3E_Set
  * @brief  EN: TI3 input detection/filtering enable.
  *         CN: TI3 /˲ʹܡ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @retval None
  */
void TIM_TI3E_Set(TIM_TypeDef *TIMx)
{
  TIMx->CR1 |= TIM_CR1_TI3E_Msk; // 1<<20
}

/**
  * @name   TIM_TI4E_Set
  * @brief  EN: TI4 input detection/filtering enable.
  *         CN: TI4 /˲ʹܡ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @retval None
  */
void TIM_TI4E_Set(TIM_TypeDef *TIMx)
{
  TIMx->CR1 |= TIM_CR1_TI4E_Msk; // 1<<21
}

/**
  * @name   TIM_TimeBaseInit
  * @brief  EN: Initializes the TIM Basic configuration according to the specified
  *             parameters in the TIM_TimeBaseInitTypeDef.
  *         CN: TIM_TimeBaseInitTypeDefָĲʼTIMá
  * @param  TIMx: EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *               CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_TimeBaseInitStruct: 
  *         EN: pointer to a TIM_TimeBaseInitTypeDef structure that contains the
  *             configuration information for the specified TIM peripheral.
  *         CN:ָTIM_TimeBaseInitTypeDefṹָ룬ýṹָTIMΧ豸Ϣ
  * @retval None
  */
void TIM_TimeBaseInit(TIM_TypeDef *TIMx, TIM_TimeBaseInitTypeDef *TIM_TimeBaseInitStruct)
{
  uint32_t tmpcr1 = 0;

  /* Check the parameters */

  tmpcr1 = TIMx->CR1;

  /* Select the Counter Mode */
  tmpcr1 &= (uint32_t)(~((uint32_t)(TIM_CR1_DIR_Msk | TIM_CR1_CMS_Msk)));
  tmpcr1 |= (uint32_t)TIM_TimeBaseInitStruct->TIM_CounterMode;

  /* Set the clock division */
  tmpcr1 &= (uint32_t)(~((uint32_t)TIM_CR1_CKD_Msk));
  tmpcr1 |= (uint32_t)TIM_TimeBaseInitStruct->TIM_ClockDivision;

  TIMx->CR1 = tmpcr1;

  /* Set the Autoreload value */
  TIMx->ARR = TIM_TimeBaseInitStruct->TIM_Period;

  /* Set the Prescaler value */
  TIMx->PSC = TIM_TimeBaseInitStruct->TIM_Prescaler;

  if ((TIMx == TIM1) || (TIMx == TIM15))
  {
    /* Set the Repetition Counter value */
    TIMx->RCR = TIM_TimeBaseInitStruct->TIM_RepetitionCounter; // RCR
  }

  /* Generate an update event to reload the Prescaler and the Repetition counter
     values immediately */
  if (TIMx->CR1 & TIM_CR1_URS_Msk)
  {
    TIMx->EGR = TIM_PSCReloadMode_Immediate;
  }
  else
  {
    TIM_UpdateRequestConfig(TIMx,TIM_UpdateSource_Regular);
    TIMx->EGR = TIM_PSCReloadMode_Immediate;
    if ((TIMx->CR1 & TIM_CR1_DIR_Msk) != TIM_CR1_DIR_Msk)
    {
      while(TIMx->CNT != 0)
      {
        __NOP();__NOP();__NOP();__NOP();__NOP();
      }
    }
    else
    {
      while(TIMx->CNT != TIMx->ARR)
      {
        __NOP();__NOP();__NOP();__NOP();__NOP();
      }
    }
    TIM_UpdateRequestConfig(TIMx,TIM_UpdateSource_Global);
  }
}

/**
  * @name   TIM_TimeBaseStructInit
  * @brief  EN: Fills each TIM_TimeBaseInitStruct member with its default value.
  *         CN: Ĭֵÿ TIM_TimeBaseInitStruct Ա
  * @param  TIM_TimeBaseInitStruct:
  *         EN: pointer to a TIM_TimeBaseInitTypeDef structure that contains the 
  *             configuration information for the specified TIM Base peripheral.
  *         CN: ָ TIM_TimeBaseInitTypeDef ṹָ룬ýṹָ TIM Ϣ
  * @retval None
  */
void TIM_TimeBaseStructInit(TIM_TimeBaseInitTypeDef *TIM_TimeBaseInitStruct)
{
  /* Set the default configuration */
  TIM_TimeBaseInitStruct->TIM_Period = 0xFFFFFFFF;
  TIM_TimeBaseInitStruct->TIM_Prescaler = 0x0000;
  TIM_TimeBaseInitStruct->TIM_ClockDivision = TIM_CKD_DIV1;
  TIM_TimeBaseInitStruct->TIM_CounterMode = TIM_CounterMode_Up;
  TIM_TimeBaseInitStruct->TIM_RepetitionCounter = 0x0000;
}

/**
  * @name   TIM_PrescalerConfig
  * @brief  EN: Prescaler configuration.
  *         CN: ԤƵá
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  Prescaler: 
  *         EN: The value of the prescaler.
  *         CN: ԤƵֵ
  *         @range 0x0 - 0xFFFF
  * @param  TIM_PSCReloadMode: 
  *         EN: Specify how to generate update events
  *         CN: ָ¼ʽ
  *             @arg TIM_PSCReloadMode_Update
  *             @arg TIM_PSCReloadMode_Immediate
  * @retval None
  */
void TIM_PrescalerConfig(TIM_TypeDef *TIMx, uint32_t Prescaler, uint32_t TIM_PSCReloadMode)
{
  /* Check the parameters */

  /* Set the Prescaler value */
  TIMx->PSC = Prescaler;
  /* Set or reset the UG Bit */
  TIMx->EGR = TIM_PSCReloadMode;
}

/**
  * @name   TIM_CounterModeConfig
  * @brief  EN: Set counter mode.
  *         CN: üģʽ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_CounterMode: 
  *         EN: Specify counter mode.
  *         CN: ָģʽ
  *             @arg TIM_CounterMode_Up
  *             @arg TIM_CounterMode_Down
  *             @arg TIM_CounterMode_CenterAligned1
  *             @arg TIM_CounterMode_CenterAligned2
  *             @arg TIM_CounterMode_CenterAligned3
  * @retval None
  */
void TIM_CounterModeConfig(TIM_TypeDef *TIMx, uint32_t TIM_CounterMode)
{
  uint32_t tmpcr1 = 0;

  /* Check the parameters */

  tmpcr1 = TIMx->CR1;
  /* Reset the CMS and DIR Bits */
  tmpcr1 &= (uint32_t)(~((uint32_t)(TIM_CR1_DIR_Msk | TIM_CR1_CMS_Msk)));
  /* Set the Counter Mode */
  tmpcr1 |= TIM_CounterMode;
  /* Write to TIMx CR1 register */
  TIMx->CR1 = tmpcr1;
}

/**
  * @name   TIM_SetCounter
  * @brief  EN: Set the initial value of the counter.
  *         CN: üֵ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval None
  */
void TIM_SetCounter(TIM_TypeDef *TIMx, uint32_t Counter)
{
  /* Check the parameters */

  /* Set the Counter Register value */
  TIMx->CNT = Counter;
}

/**
  * @name   TIM_SetAutoreload
  * @brief  EN: Set Automatic reloading value.
  *         CN: Զװֵ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval None
  */
void TIM_SetAutoreload(TIM_TypeDef *TIMx, uint32_t Autoreload)
{
  /* Check the parameters */

  /* Set the Autoreload Register value */
  TIMx->ARR = Autoreload;
}

/**
  * @name   TIM_GetCounter
  * @brief  EN: Get the counter value.
  *         CN: ȡֵ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval EN: Counter Register value.
  *         CN: Ĵֵ
  */
uint32_t TIM_GetCounter(TIM_TypeDef *TIMx)
{
  /* Check the parameters */

  /* Get the Counter Register value */
  return TIMx->CNT;
}

/**
  * @name   TIM_GetPrescaler
  * @brief  EN: Get the predivider value.
  *         CN: ȡԤƵֵ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval EN: The value of the prescaler.
  *         CN: ԤƵֵ
  */
uint32_t TIM_GetPrescaler(TIM_TypeDef *TIMx)
{
  /* Check the parameters */
  /* Get the Prescaler Register value */
  return TIMx->PSC;
}

/**
  * @name   TIM_UpdateDisableConfig
  * @brief  EN: Disable update.
  *         CN: ֹʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: Enable or disable Automatic reload.
  *         CN: ûֹʹܡ
  * @retval None
  */
void TIM_UpdateDisableConfig(TIM_TypeDef *TIMx, FunctionalState NewState)
{
  /* Check the parameters */

  if (NewState != DISABLE)
  {
    /* Set the Update Disable Bit */
    TIMx->CR1 |= TIM_CR1_UDIS_Msk;
  }
  else
  {
    /* Reset the Update Disable Bit */
    TIMx->CR1 &= (uint32_t) ~((uint32_t)TIM_CR1_UDIS_Msk);
  }
}

/**
  * @name   TIM_UpdateRequestConfig
  * @brief  EN: Update Request source.
  *         CN: Դ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_UpdateSource: 
  *         EN: Enable or disable Automatic reload.
  *         CN: ûֹԶװء
  * @retval None
  */
void TIM_UpdateRequestConfig(TIM_TypeDef *TIMx, uint32_t TIM_UpdateSource)
{
  /* Check the parameters */

  if (TIM_UpdateSource != TIM_UpdateSource_Global)
  {
    /* Set the URS Bit */
    TIMx->CR1 |= TIM_CR1_URS_Msk;
  }
  else
  {
    /* Reset the URS Bit */
    TIMx->CR1 &= (uint32_t) ~((uint32_t)TIM_CR1_URS_Msk);
  }
}

/**
  * @name   TIM_ARRPreloadConfig
  * @brief  EN: Automatic reload enables buffering.
  *         CN: Զװʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: Enable or disable Automatic reload.
  *         CN: ûֹԶװء
  * @retval None
  */
void TIM_ARRPreloadConfig(TIM_TypeDef *TIMx, FunctionalState NewState)
{
  /* Check the parameters */

  if (NewState != DISABLE)
  {
    /* Set the ARR Preload Bit */
    TIMx->CR1 |= TIM_CR1_ARPE_Msk;
  }
  else
  {
    /* Reset the ARR Preload Bit */
    TIMx->CR1 &= (uint32_t) ~((uint32_t)TIM_CR1_ARPE_Msk);
  }
}

/**
  * @name   TIM_SelectOnePulseMode
  * @brief  EN: single pulse mode.
  *         CN: ģʽ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_OPMode: 
  *         EN: Enable or disable single pulse mode.
  *         CN: ûֹģʽ
  *             @arg TIM_OPMode_Single
  *             @arg TIM_OPMode_Repetitive
  * @retval None
  */
void TIM_SelectOnePulseMode(TIM_TypeDef *TIMx, uint32_t TIM_OPMode)
{
  /* Check the parameters */

  /* Reset the OPM Bit */
  TIMx->CR1 &= (uint32_t) ~((uint32_t)TIM_CR1_OPM_Msk);
  /* Configure the OPM Mode */
  TIMx->CR1 |= TIM_OPMode;
}

/**
  * @name   TIM_SetClockDivision
  * @brief  EN: Set the clock frequency.
  *         CN: ʱӷƵ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_ClockDivision: 
  *         EN: The clock division ratio between the timer clock (CK_INT) frequency and the dead-time/sampling clock (CK_DTS).
  *         CN: CK_INT ʱӺ/ʱӣCK_DTSķƵϵ
  *             @arg TIM_CKD_DIV1
  *             @arg TIM_CKD_DIV2
  *             @arg TIM_CKD_DIV4
  * @retval None
  */
void TIM_SetClockDivision(TIM_TypeDef *TIMx, uint32_t TIM_ClockDivision)
{
  /* Check the parameters */

  /* Reset the CKD Bits */
  TIMx->CR1 &= (uint32_t) ~((uint32_t)TIM_CR1_CKD_Msk);
  /* Set the CKD value */
  TIMx->CR1 |= TIM_ClockDivision;
}

/**
  * @name   TIM_Cmd
  * @brief  EN: Enables or disables the specified TIMx peripheral.
  *         CN: ûָ TIMx 
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx peripheral.
  *         CN: TIMx ״̬
  * @retval None
  */
void TIM_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
  /* Check the parameters */

  if (NewState != DISABLE)
  {
    /* Enable the TIM Counter */
    TIMx->CR1 |= TIM_CR1_CEN_Msk;
  }
  else
  {
    /* Disable the TIM Counter */
    TIMx->CR1 &= (uint32_t)(~((uint32_t)TIM_CR1_CEN_Msk));
    if (TIMx->CR1 & TIM_CR1_URS_Msk)
    {
      TIMx->EGR = TIM_PSCReloadMode_Immediate;
    }
    else
    {
      TIM_UpdateRequestConfig(TIMx,TIM_UpdateSource_Regular);
      TIMx->EGR = TIM_PSCReloadMode_Immediate;
      if ((TIMx->CR1 & TIM_CR1_DIR_Msk) != TIM_CR1_DIR_Msk)
      {
        while(TIMx->CNT != 0)
        {
          __NOP();__NOP();__NOP();__NOP();__NOP();
        }
      }
      else
      {
        while(TIMx->CNT != TIMx->ARR)
        {
          __NOP();__NOP();__NOP();__NOP();__NOP();
        }
      }
      TIM_UpdateRequestConfig(TIMx,TIM_UpdateSource_Global);
    }
  }
}

/**
  * @name   TIM_BDTRConfig
  * @brief  EN: Configure TIM peripheral devices according to the specified parameters
  *             in TIM-BDTRInitStructure to control brake configuration.
  *         CN: TIM_BDTRInitStructеָTIMΧ豸Կɲá
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_BDTRInitStruct: 
  *         EN: pointer to a TIM_BDTRInitTypeDef structure
  *             that contains the configuration information for the specified TIM
  *         CN:ָTIM_BDTRInitTypeDefṹָ룬ýṹָTIMϢ
  * @retval None
  */
void TIM_BDTRConfig(TIM_TypeDef *TIMx, TIM_BDTRInitTypeDef *TIM_BDTRInitStruct)
{
  /* Check the parameters */

  /* Set the Lock level, the Break enable Bit and the Ploarity, the OSSR State,
     the OSSI State, the dead time value and the Automatic Output Enable Bit */
  TIMx->BDTR = (uint32_t)TIM_BDTRInitStruct->TIM_OSSRState | TIM_BDTRInitStruct->TIM_OSSIState |
               TIM_BDTRInitStruct->TIM_LOCKLevel | TIM_BDTRInitStruct->TIM_DeadTime |
               TIM_BDTRInitStruct->TIM_BreakPolarity | TIM_BDTRInitStruct->TIM_AutomaticOutput | \
              (TIM_BDTRInitStruct->TIM_Break2Polarity << 12)  | (TIM_BDTRInitStruct->TIM_AutomaticOutput2 << 12);
  TIMx->BDTR |= TIM_BDTRInitStruct->TIM_Break | (TIM_BDTRInitStruct->TIM_Break2 << 12);
}

/**
  * @name   TIM_BDTRStructInit
  * @brief  EN: Fills each TIM_BDTRStructInit member with its default value.
  *         CN: Ĭֵÿ TIM_BDTRStructInit Ա
  * @param  TIM_BDTRInitStruct:
  *         EN: pointer to a TIM_BDTRInitTypeDef structure that contains the 
  *             configuration information for the specified TIM Base peripheral.
  *         CN: ָ TIM_BDTRInitTypeDef ṹָ룬ýṹָ TIM BDTRϢ
  * @retval None
  */
void TIM_BDTRStructInit(TIM_BDTRInitTypeDef *TIM_BDTRInitStruct)
{
  /* Set the default configuration */
  TIM_BDTRInitStruct->TIM_OSSRState = TIM_OSSRState_Disable;
  TIM_BDTRInitStruct->TIM_OSSIState = TIM_OSSIState_Disable;
  TIM_BDTRInitStruct->TIM_LOCKLevel = TIM_LOCKLevel_OFF;
  TIM_BDTRInitStruct->TIM_DeadTime = 0x00;
  TIM_BDTRInitStruct->TIM_Break = TIM_Break_Disable;
  TIM_BDTRInitStruct->TIM_BreakPolarity = TIM_BreakPolarity_Low;
  TIM_BDTRInitStruct->TIM_AutomaticOutput = TIM_AutomaticOutput_Disable;
  TIM_BDTRInitStruct->TIM_Break2 = TIM_Break_Disable;
  TIM_BDTRInitStruct->TIM_Break2Polarity = TIM_BreakPolarity_Low;
  TIM_BDTRInitStruct->TIM_AutomaticOutput2 = TIM_AutomaticOutput_Disable;
}

/**
  * @name   TIM_CtrlPWMOutputs
  * @brief  EN: Enables or disables the TIM peripheral Main Outputs.
  *         CN: ûTIMΧ豸
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: new state of the TIM peripheral Main Outputs.
  *         CN: TIMΧ豸״̬
  * @retval None
  */
void TIM_CtrlPWMOutputs(TIM_TypeDef *TIMx, FunctionalState NewState)
{
  /* Check the parameters */

  if (NewState != DISABLE)
  {
    /* Enable the TIM Main Output */
    TIMx->BDTR |= TIM_BDTR_MOE_Msk;
  }
  else
  {
    /* Disable the TIM Main Output */
    TIMx->BDTR &= (uint32_t)(~((uint32_t)TIM_BDTR_MOE_Msk));
  }
}

/**
  * @name   TIM_OC1Init
  * @brief  EN: Initializes the TIMx Channel1 according to the specified
  *             parameters in the TIM_OCInitStruct.
  *         CN: TIM_OCInitStructָĲʼTIM ͨ1
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_OCInitStruct: 
  *         EN: pointer to a TIM_OCInitTypeDef structure
  *             that contains the configuration information for the specified TIM
  *         CN:ָTIM_OCInitTypeDefṹָ룬ýṹָTIMϢ
  * @retval None
  */
void TIM_OC1Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct)
{
  uint32_t tmpccmrx = 0, tmpccer = 0, tmpcr2 = 0;

  /* Check the parameters */

  /* Disable the Channel 1: Reset the CC1E Bit */
  TIMx->CCER &= (uint32_t)(~(uint32_t)TIM_CCER_CC1E_Msk);
  /* Get the TIMx CCER register value */
  tmpccer = TIMx->CCER;
  /* Get the TIMx CR2 register value */
  tmpcr2 = TIMx->CR2;

  /* Get the TIMx CCMR1 register value */
  tmpccmrx = TIMx->CCMR1;

  /* Reset the Output Compare Mode Bits */
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR1_Output_OC1M_Msk));
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR1_Output_CC1S_Msk));

  /* Select the Output Compare Mode */
  tmpccmrx |= TIM_OCInitStruct->TIM_OCMode;

  /* Reset the Output Polarity level */
  tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC1P_Msk));
  /* Set the Output Compare Polarity */
  tmpccer |= TIM_OCInitStruct->TIM_OCPolarity;

  /* Set the Output State */
  tmpccer |= TIM_OCInitStruct->TIM_OutputState;

  /* Reset the Ouput Compare and Output Compare IDLE State */
  tmpcr2 &= (uint32_t)(~((uint32_t)TIM_CR2_OIS1_Msk));
  /* Set the Output Idle state */
  tmpcr2 |= TIM_OCInitStruct->TIM_OCIdleState;

  if ((TIMx == TIM1) || (TIMx == TIM15) || (TIMx == TIM16) || (TIMx == TIM17))
  {

    /* Reset the Output N Polarity level */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC1NP_Msk));
    /* Set the Output N Polarity */
    tmpccer |= TIM_OCInitStruct->TIM_OCNPolarity;

    /* Reset the Output N State */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC1NE_Msk));
    /* Set the Output N State */
    tmpccer |= TIM_OCInitStruct->TIM_OutputNState;

    /* Reset the Ouput Compare and Output Compare IDLE N State */
    tmpcr2 &= (uint32_t)(~((uint32_t)TIM_CR2_OIS1N_Msk));
    /* Set the Output N Idle state */
    tmpcr2 |= TIM_OCInitStruct->TIM_OCNIdleState;
  }
  /* Write to TIMx CR2 */
  TIMx->CR2 = tmpcr2;

  /* Write to TIMx CCMR1 */
  TIMx->CCMR1 = tmpccmrx;

  /* Set the Capture Compare Register value */
  TIMx->CCR1 = TIM_OCInitStruct->TIM_Pulse;

  /* Write to TIMx CCER */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC2Init
  * @brief  EN: Initializes the TIMx Channel2 according to the specified
  *             parameters in the TIM_OCInitStruct.
  *         CN: TIM_OCInitStructָĲʼTIM ͨ2
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @param  TIM_OCInitStruct: 
  *         EN: pointer to a TIM_OCInitTypeDef structure
  *             that contains the configuration information for the specified TIM
  *         CN:ָTIM_OCInitTypeDefṹָ룬ýṹָTIMϢ
  * @retval None
  */
void TIM_OC2Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct)
{
  uint32_t tmpccmrx = 0, tmpccer = 0, tmpcr2 = 0;

  /* Check the parameters */

  /* Disable the Channel 2: Reset the CC2E Bit */
  TIMx->CCER &= (uint32_t)(~((uint32_t)TIM_CCER_CC2E_Msk));

  /* Get the TIMx CCER register value */
  tmpccer = TIMx->CCER;
  /* Get the TIMx CR2 register value */
  tmpcr2 = TIMx->CR2;

  /* Get the TIMx CCMR1 register value */
  tmpccmrx = TIMx->CCMR1;

  /* Reset the Output Compare mode and Capture/Compare selection Bits */
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR1_Output_OC2M_Msk));
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR1_Output_CC2S_Msk));

  /* Select the Output Compare Mode */
  tmpccmrx |= (uint32_t)(TIM_OCInitStruct->TIM_OCMode << 8);

  /* Reset the Output Polarity level */
  tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC2P_Msk));
  /* Set the Output Compare Polarity */
  tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OCPolarity << 4);

  /* Set the Output State */
  tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OutputState << 4);
  /* Reset the Ouput Compare and Output Compare IDLE State */
  tmpcr2 &= (uint32_t)(~((uint32_t)TIM_CR2_OIS2_Msk));

  /* Set the Output Idle state */
  tmpcr2 |= (uint32_t)(TIM_OCInitStruct->TIM_OCIdleState << 2);

  if ((TIMx == TIM1) || (TIMx == TIM15))
  {
    /* Check the parameters */

    /* Reset the Output N Polarity level */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC2NP_Msk));
    /* Set the Output N Polarity */
    tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OCNPolarity << 4);

    /* Reset the Output N State */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC2NE_Msk));
    /* Set the Output N State */
    tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OutputNState << 4);

    /* Reset the Output Compare N IDLE State */
    tmpcr2 &= (uint32_t)(~((uint32_t)TIM_CR2_OIS2N_Msk));
    /* Set the Output N Idle state */
    tmpcr2 |= (uint32_t)(TIM_OCInitStruct->TIM_OCNIdleState << 2);
  }

  /* Write to TIMx CR2 */
  TIMx->CR2 = tmpcr2;

  /* Write to TIMx CCMR1 */
  TIMx->CCMR1 = tmpccmrx;

  /* Set the Capture Compare Register value */
  TIMx->CCR2 = TIM_OCInitStruct->TIM_Pulse;

  /* Write to TIMx CCER */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC3Init
  * @brief  EN: Initializes the TIMx Channel3 according to the specified
  *             parameters in the TIM_OCInitStruct.
  *         CN: TIM_OCInitStructָĲʼTIM ͨ3
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_OCInitStruct: 
  *         EN: pointer to a TIM_OCInitTypeDef structure
  *             that contains the configuration information for the specified TIM
  *         CN:ָTIM_OCInitTypeDefṹָ룬ýṹָTIMϢ
  * @retval None
  */
void TIM_OC3Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct)
{
  uint32_t tmpccmrx = 0, tmpccer = 0, tmpcr2 = 0;

  /* Check the parameters */

  /* Disable the Channel 2: Reset the CC2E Bit */
  TIMx->CCER &= (uint32_t)(~((uint32_t)TIM_CCER_CC3E_Msk));

  /* Get the TIMx CCER register value */
  tmpccer = TIMx->CCER;
  /* Get the TIMx CR2 register value */
  tmpcr2 = TIMx->CR2;

  /* Get the TIMx CCMR2 register value */
  tmpccmrx = TIMx->CCMR2;

  /* Reset the Output Compare mode and Capture/Compare selection Bits */
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR2_Output_OC3M_Msk));
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR2_Output_CC3S_Msk));
  /* Select the Output Compare Mode */
  tmpccmrx |= TIM_OCInitStruct->TIM_OCMode;

  /* Reset the Output Polarity level */
  tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC3P_Msk));
  /* Set the Output Compare Polarity */
  tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OCPolarity << 8);

  /* Set the Output State */
  tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OutputState << 8);

  /* Reset the Ouput Compare and Output Compare IDLE State */
  tmpcr2 &= (uint32_t)(~((uint32_t)TIM_CR2_OIS3_Msk));
  /* Set the Output Idle state */
  tmpcr2 |= (uint32_t)(TIM_OCInitStruct->TIM_OCIdleState << 4);

  if (TIMx == TIM1)
  {

    /* Reset the Output N Polarity level */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC3NP_Msk));
    /* Set the Output N Polarity */
    tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OCNPolarity << 8);
    
    /* Reset the Output N State */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC3NE_Msk));
    /* Set the Output N State */
    tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OutputNState << 8);

    /* Reset the Ouput Compare and Output Compare N IDLE State */
    tmpcr2 &= (uint32_t)(~((uint32_t)TIM_CR2_OIS3N_Msk));
    /* Set the Output N Idle state */
    tmpcr2 |= (uint32_t)(TIM_OCInitStruct->TIM_OCNIdleState << 4);
  }
  /* Write to TIMx CR2 */
  TIMx->CR2 = tmpcr2;

  /* Write to TIMx CCMR2 */
  TIMx->CCMR2 = tmpccmrx;

  /* Set the Capture Compare Register value */
  TIMx->CCR3 = TIM_OCInitStruct->TIM_Pulse;

  /* Write to TIMx CCER */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC4Init
  * @brief  EN: Initializes the TIMx Channel4 according to the specified
  *             parameters in the TIM_OCInitStruct.
  *         CN: TIM_OCInitStructָĲʼTIM ͨ4
  * @param  TIMx: 
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_OCInitStruct: 
  *         EN: pointer to a TIM_OCInitTypeDef structure
  *             that contains the configuration information for the specified TIM
  *         CN:ָTIM_OCInitTypeDefṹָ룬ýṹָTIMϢ
  * @retval None
  */
void TIM_OC4Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct)
{
  uint32_t tmpccmrx = 0, tmpccer = 0, tmpcr2 = 0;

  /* Check the parameters */

  /* Disable the Channel 2: Reset the CC4E Bit */
  TIMx->CCER &= (uint32_t)(~((uint32_t)TIM_CCER_CC4E_Msk));

  /* Get the TIMx CCER register value */
  tmpccer = TIMx->CCER;
  /* Get the TIMx CR2 register value */
  tmpcr2 = TIMx->CR2;

  /* Get the TIMx CCMR2 register value */
  tmpccmrx = TIMx->CCMR2;

  /* Reset the Output Compare mode and Capture/Compare selection Bits */
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR2_Output_OC4M_Msk));
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR2_Output_CC4S_Msk));

  /* Select the Output Compare Mode */
  tmpccmrx |= (uint32_t)(TIM_OCInitStruct->TIM_OCMode << 8);

  /* Reset the Output Polarity level */
  tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC4P_Msk));
  /* Set the Output Compare Polarity */
  tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OCPolarity << 12);

  /* Set the Output State */
  tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OutputState << 12);
  
  /* Reset the Ouput Compare and Output Compare IDLE State */
  tmpcr2 &= (uint32_t)(~((uint32_t)TIM_CR2_OIS4_Msk));
  /* Set the Output Idle state */
  tmpcr2 |= (uint32_t)(TIM_OCInitStruct->TIM_OCIdleState << 6);

  if (TIMx == TIM1)
  {

    /* Reset the Output N Polarity level */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC4NP_Msk));
    /* Set the Output N Polarity */
    tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OCNPolarity << 12);

    /* Reset the Output N State */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC4NE_Msk));
    /* Set the Output N State */
    tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OutputNState << 12);

    /* Reset the Ouput Compare and Output Compare N IDLE State */
    tmpcr2 &= (uint32_t)(~((uint32_t)TIM_CR2_OIS4N_Msk));
    /* Set the Output N Idle state */
    tmpcr2 |= (uint32_t)(TIM_OCInitStruct->TIM_OCNIdleState << 6);
  }
  /* Write to TIMx CR2 */
  TIMx->CR2 = tmpcr2;

  /* Write to TIMx CCMR2 */
  TIMx->CCMR2 = tmpccmrx;

  /* Set the Capture Compare Register value */
  TIMx->CCR4 = TIM_OCInitStruct->TIM_Pulse;

  /* Write to TIMx CCER */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC5Init
  * @brief  EN: Initializes the TIMx Channel5 according to the specified
  *             parameters in the TIM_OCInitStruct.
  *         CN: TIM_OCInitStructָĲʼTIM ͨ5
  * @param  TIMx: 
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  TIM_OCInitStruct: 
  *         EN: pointer to a TIM_OCInitTypeDef structure
  *             that contains the configuration information for the specified TIM
  *         CN:ָTIM_OCInitTypeDefṹָ룬ýṹָTIMϢ
  * @retval None
  */
void TIM_OC5Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct)
{
  uint32_t tmpccmrx = 0, tmpccer = 0, tmpcr2 = 0;

  /* Check the parameters */

  /* Disable the Channel 5: Reset the CC5E Bit */
  TIMx->CCER &= (uint32_t)(~(uint32_t)TIM_CCER_CC5E_Msk);
  /* Get the TIMx CCER register value */
  tmpccer = TIMx->CCER;
  /* Get the TIMx CR2 register value */
  tmpcr2 = TIMx->CR2;

  /* Get the TIMx CCMR3 register value */
  tmpccmrx = TIMx->CCMR3;

  /* Reset the Output Compare Mode Bits */
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR3_Output_OC5M_Msk));
  tmpccmrx &= (uint32_t)(~((uint32_t)TIM_CCMR3_Output_CC5S_Msk));

  /* Select the Output Compare Mode */
  tmpccmrx |= TIM_OCInitStruct->TIM_OCMode;

  /* Reset the Output Polarity level */
  tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC5P_Msk));
  /* Set the Output Compare Polarity */
  tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OCPolarity << 16);

  /* Set the Output State */
  tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OutputState << 16);

  if (TIMx == TIM1)
  {

    /* Reset the Output N Polarity level */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC5NP_Msk));
    /* Set the Output N Polarity */
    tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OCNPolarity << 16);

    /* Reset the Output N State */
    tmpccer &= (uint32_t)(~((uint32_t)TIM_CCER_CC5NE_Msk));
    /* Set the Output N State */
    tmpccer |= (uint32_t)(TIM_OCInitStruct->TIM_OutputNState << 16);

  }
  /* Write to TIMx CR2 */
  TIMx->CR2 = tmpcr2;

  /* Write to TIMx CCMR3 */
  TIMx->CCMR3 = tmpccmrx;

  /* Set the Capture Compare Register value */
  TIMx->CCR5 = TIM_OCInitStruct->TIM_Pulse;

  /* Write to TIMx CCER */
  TIMx->CCER = tmpccer;
}
/**
  * @name   TIM_OCStructInit
  * @brief  EN: Fills each TIM_OCInitStruct member with its default value.
  *         CN: Ĭֵÿ TIM_OCInitStruct Ա
  * @param  TIM_ICInitStruct:
  *         EN: pointer to a TIM_OCInitTypeDef structure that contains the 
  *             configuration information for the specified TIM Base peripheral.
  *         CN: ָ TIM_OCInitTypeDef ṹָ룬ýṹָ TIM OCϢ
  * @retval None
  */
void TIM_OCStructInit(TIM_OCInitTypeDef *TIM_OCInitStruct)
{
  /* Set the default configuration */
  TIM_OCInitStruct->TIM_OCMode = TIM_OCMode_Timing;
  TIM_OCInitStruct->TIM_OutputState = TIM_OutputState_Disable;
  TIM_OCInitStruct->TIM_OutputNState = TIM_OutputNState_Disable;
  TIM_OCInitStruct->TIM_Pulse = 0x0000000;
  TIM_OCInitStruct->TIM_OCPolarity = TIM_OCPolarity_High;
  TIM_OCInitStruct->TIM_OCNPolarity = TIM_OCPolarity_High;
  TIM_OCInitStruct->TIM_OCIdleState = TIM_OCIdleState_Reset;
  TIM_OCInitStruct->TIM_OCNIdleState = TIM_OCNIdleState_Reset;
}

/**
  * @name   TIM_SelectOCxM
  * @brief  EN: Selects the TIM Output Compare Mode.
  *         CN: ѡTIMȽģʽ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_Channel: 
  *         EN: specifies the TIM Channel
  *         CN: ָTIMͨ
  *             @arg TIM_Channel_1
  *             @arg TIM_Channel_2
  *             @arg TIM_Channel_3
  *             @arg TIM_Channel_4
  *             @arg TIM_Channel_5
  * @param  TIM_CCx: 
  *         EN: specifies the TIM Output Compare Mode.
  *         CN: ָTIMȽģʽ
  *             @arg TIM_OCMode_Timing
  *             @arg TIM_OCMode_Active
  *             @arg TIM_OCMode_Toggle
  *             @arg TIM_OCMode_PWM1
  *             @arg TIM_OCMode_PWM2
  *             @arg TIM_ForcedAction_Active
  *             @arg TIM_ForcedAction_InActive
  * @retval None
  */
void TIM_SelectOCxM(TIM_TypeDef *TIMx, uint32_t TIM_Channel, uint32_t TIM_OCMode)
{
  uint32_t tmp = 0;
  uint32_t tmp1 = 0;

  /* Check the parameters */

  tmp = (uint32_t)TIMx;
  tmp += 0x00000018;

  tmp1 = TIM_CCER_CC1E_Msk << (uint32_t)TIM_Channel;

  /* Disable the Channel: Reset the CCxE Bit */
  TIMx->CCER &= (uint32_t)~tmp1;

  if ((TIM_Channel == TIM_Channel_1) || (TIM_Channel == TIM_Channel_3))
  {
    tmp += (TIM_Channel >> 1);

    /* Reset the OCxM bits in the CCMRx register */
    *(__IO uint32_t *)tmp &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC1M_Msk);

    /* Configure the OCxM bits in the CCMRx register */
    *(__IO uint32_t *)tmp |= TIM_OCMode;
  }
  else
  {
    tmp += (uint32_t)(TIM_Channel - (uint32_t)4) >> (uint32_t)1;

    /* Reset the OCxM bits in the CCMRx register */
    *(__IO uint32_t *)tmp &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC2M_Msk);

    /* Configure the OCxM bits in the CCMRx register */
    *(__IO uint32_t *)tmp |= (uint32_t)(TIM_OCMode << 8);
  }
}

/**
  * @name   TIM_SetCompare1
  * @brief  EN: Sets the TIMx Capture Compare1 Register value.
  *         CN: TIMx Capture Compare1Ĵֵ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  Compare1: 
  *         EN: specifies the Capture Compare1 register new value.
  *         CN: ָCapture Compare1Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompare1(TIM_TypeDef *TIMx, uint32_t Compare1)
{
  /* Check the parameters */

  /* Set the Capture Compare1 Register value */
  TIMx->CCR1 = Compare1;
}

/**
  * @name   TIM_SetCompare2
  * @brief  EN: Sets the TIMx Capture Compare2 Register value.
  *         CN: TIMx Capture Compare2Ĵֵ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @param  Compare2: 
  *         EN: specifies the Capture Compare2 register new value.
  *         CN: ָCapture Compare2Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompare2(TIM_TypeDef *TIMx, uint32_t Compare2)
{
  /* Check the parameters */

  /* Set the Capture Compare2 Register value */
  TIMx->CCR2 = Compare2;
}

/**
  * @name   TIM_SetCompare3
  * @brief  EN: Sets the TIMx Capture Compare3 Register value.
  *         CN: TIMx Capture Compare3Ĵֵ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  Compare3: 
  *         EN: specifies the Capture Compare3 register new value.
  *         CN: ָCapture Compare3Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompare3(TIM_TypeDef *TIMx, uint32_t Compare3)
{
  /* Check the parameters */

  /* Set the Capture Compare3 Register value */
  TIMx->CCR3 = Compare3;
}

/**
  * @name   TIM_SetCompare4
  * @brief  EN: Sets the TIMx Capture Compare4 Register value.
  *         CN: TIMx Capture Compare4Ĵֵ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  Compare4: 
  *         EN: specifies the Capture Compare4 register new value.
  *         CN: ָCapture Compare4Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompare4(TIM_TypeDef *TIMx, uint32_t Compare4)
{
  /* Check the parameters */

  /* Set the Capture Compare4 Register value */
  TIMx->CCR4 = Compare4;
}

/**
  * @name   TIM_SetCompare4
  * @brief  EN: Sets the TIMx Capture Compare4 Register value.
  *         CN: TIMx Capture Compare4Ĵֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  Compare5: 
  *         EN: specifies the Capture Compare5 register new value.
  *         CN: ָCapture Compare5Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompare5(TIM_TypeDef *TIMx, uint32_t Compare5)
{
  /* Check the parameters */

  /* Set the Capture Compare4 Register value */
  TIMx->CCR5 = Compare5;
}

/**
  * @name   TIM_ForcedOC1Config
  * @brief  EN: Forces the TIMx output 1 waveform to active or inactive level.
  *         CN: ǿTIMx1ΪǼƽ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_ForcedAction: 
  *         EN: specifies the forced Action to be set to the output waveform.
  *         CN: ָҪΪεǿƶ
  *             @arg TIM_ForcedAction_Active
  *             @arg TIM_ForcedAction_InActive
  * @retval None
  */
void TIM_ForcedOC1Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction)
{
  uint32_t tmpccmr1 = 0;
  /* Check the parameters */
  tmpccmr1 = TIMx->CCMR1;
  /* Reset the OC1M Bits */
  tmpccmr1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC1M_Msk);
  /* Configure The Forced output Mode */
  tmpccmr1 |= TIM_ForcedAction;
  /* Write to TIMx CCMR1 register */
  TIMx->CCMR1 = tmpccmr1;
}

/**
  * @name   TIM_ForcedOC2Config
  * @brief  EN: Forces the TIMx output 2 waveform to active or inactive level.
  *         CN: ǿTIMx2ΪǼƽ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @param  TIM_ForcedAction: 
  *         EN: specifies the forced Action to be set to the output waveform.
  *         CN: ָҪΪεǿƶ
  *             @arg TIM_ForcedAction_Active
  *             @arg TIM_ForcedAction_InActive
  * @retval None
  */
void TIM_ForcedOC2Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction)
{
  uint32_t tmpccmr1 = 0;

  /* Check the parameters */

  tmpccmr1 = TIMx->CCMR1;
  /* Reset the OC2M Bits */
  tmpccmr1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC2M_Msk);
  /* Configure The Forced output Mode */
  tmpccmr1 |= (uint32_t)(TIM_ForcedAction << 8);
  /* Write to TIMx CCMR1 register */
  TIMx->CCMR1 = tmpccmr1;
}

/**
  * @name   TIM_ForcedOC3Config
  * @brief  EN: Forces the TIMx output 3 waveform to active or inactive level.
  *         CN: ǿTIMx3ΪǼƽ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_ForcedAction: 
  *         EN: specifies the forced Action to be set to the output waveform.
  *         CN: ָҪΪεǿƶ
  *             @arg TIM_ForcedAction_Active
  *             @arg TIM_ForcedAction_InActive
  * @retval None
  */
void TIM_ForcedOC3Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction)
{
  uint32_t tmpccmr2 = 0;

  /* Check the parameters */

  tmpccmr2 = TIMx->CCMR2;
  /* Reset the OC1M Bits */
  tmpccmr2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Output_OC3M_Msk);
  /* Configure The Forced output Mode */
  tmpccmr2 |= TIM_ForcedAction;
  /* Write to TIMx CCMR2 register */
  TIMx->CCMR2 = tmpccmr2;
}

/**
  * @name   TIM_ForcedOC4Config
  * @brief  EN: Forces the TIMx output 4 waveform to active or inactive level.
  *         CN: ǿTIMx4ΪǼƽ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_ForcedAction: 
  *         EN: specifies the forced Action to be set to the output waveform.
  *         CN: ָҪΪεǿƶ
  *             @arg TIM_ForcedAction_Active
  *             @arg TIM_ForcedAction_InActive
  * @retval None
  */
void TIM_ForcedOC4Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction)
{
  uint32_t tmpccmr2 = 0;
  /* Check the parameters */

  tmpccmr2 = TIMx->CCMR2;
  /* Reset the OC2M Bits */
  tmpccmr2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Output_OC4M_Msk);
  /* Configure The Forced output Mode */
  tmpccmr2 |= (uint32_t)(TIM_ForcedAction << 8);
  /* Write to TIMx CCMR2 register */
  TIMx->CCMR2 = tmpccmr2;
}

/**
  * @name   TIM_ForcedOC5Config
  * @brief  EN: Forces the TIMx output 5 waveform to active or inactive level.
  *         CN: ǿTIMx5ΪǼƽ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  TIM_ForcedAction: 
  *         EN: specifies the forced Action to be set to the output waveform.
  *         CN: ָҪΪεǿƶ
  *             @arg TIM_ForcedAction_Active
  *             @arg TIM_ForcedAction_InActive
  * @retval None
  */
void TIM_ForcedOC5Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction)
{
  uint32_t tmpccmr3 = 0;
  /* Check the parameters */

  tmpccmr3 = TIMx->CCMR3;
  /* Reset the OC3M Bits */
  tmpccmr3 &= (uint32_t) ~((uint32_t)TIM_CCMR3_Output_OC5M_Msk);
  /* Configure The Forced output Mode */
  tmpccmr3 |= (uint32_t)TIM_ForcedAction;
  /* Write to TIMx CCMR3 register */
  TIMx->CCMR3 = tmpccmr3;
}

/**
  * @name   TIM_CCPreloadControl
  * @brief  EN: Sets or Resets the TIM peripheral Capture Compare Preload Control bit.
  *         CN: ûTIMΧ豸ȽԤؿλ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: new state of the Capture Compare Preload Control bit.
  *         CN: ȽԤؿλ״̬
  * @retval None
  */
void TIM_CCPreloadControl(TIM_TypeDef *TIMx, FunctionalState NewState)
{
  /* Check the parameters */
  if (NewState != DISABLE)
  {
    /* Set the CCPC Bit */
    TIMx->CR2 |= TIM_CR2_CCPC_Msk;
  }
  else
  {
    /* Reset the CCPC Bit */
    TIMx->CR2 &= (uint32_t) ~((uint32_t)TIM_CR2_CCPC_Msk);
  }
}

/**
  * @name   TIM_OC1PreloadConfig
  * @brief  EN: Enables or disables the TIMx peripheral Preload register on CCR1.
  *         CN: ûCCR1ϵTIMxԤؼĴ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_OCPreload: 
  *         EN: new state of the TIMx peripheral Preload register.
  *         CN: TIMxԤؼĴ״̬
  *             @arg TIM_OCPreload_Enable
  *             @arg TIM_OCPreload_Disable
  * @retval None
  */
void TIM_OC1PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload)
{
  uint32_t tmpccmr1 = 0;
  /* Check the parameters */

  tmpccmr1 = TIMx->CCMR1;
  /* Reset the OC1PE Bit */
  tmpccmr1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC1PE_Msk);
  /* Enable or Disable the Output Compare Preload feature */
  tmpccmr1 |= TIM_OCPreload;
  /* Write to TIMx CCMR1 register */
  TIMx->CCMR1 = tmpccmr1;
}

/**
  * @name   TIM_OC2PreloadConfig
  * @brief  EN: Enables or disables the TIMx peripheral Preload register on CCR2.
  *         CN: ûCCR2ϵTIMxԤؼĴ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @param  TIM_OCPreload: 
  *         EN: new state of the TIMx peripheral Preload register.
  *         CN: TIMxԤؼĴ״̬
  *             @arg TIM_OCPreload_Enable
  *             @arg TIM_OCPreload_Disable
  * @retval None
  */
void TIM_OC2PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload)
{
  uint32_t tmpccmr1 = 0;
  /* Check the parameters */

  tmpccmr1 = TIMx->CCMR1;
  /* Reset the OC2PE Bit */
  tmpccmr1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC2PE_Msk);
  /* Enable or Disable the Output Compare Preload feature */
  tmpccmr1 |= (uint32_t)(TIM_OCPreload << 8);
  /* Write to TIMx CCMR1 register */
  TIMx->CCMR1 = tmpccmr1;
}

/**
  * @name   TIM_OC3PreloadConfig
  * @brief  EN: Enables or disables the TIMx peripheral Preload register on CCR3.
  *         CN: ûCCR3ϵTIMxԤؼĴ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_OCPreload: 
  *         EN: new state of the TIMx peripheral Preload register.
  *         CN: TIMxԤؼĴ״̬
  *             @arg TIM_OCPreload_Enable
  *             @arg TIM_OCPreload_Disable
  * @retval None
  */
void TIM_OC3PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload)
{
  uint32_t tmpccmr2 = 0;

  /* Check the parameters */

  tmpccmr2 = TIMx->CCMR2;
  /* Reset the OC3PE Bit */
  tmpccmr2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Output_OC3PE_Msk);
  /* Enable or Disable the Output Compare Preload feature */
  tmpccmr2 |= TIM_OCPreload;
  /* Write to TIMx CCMR2 register */
  TIMx->CCMR2 = tmpccmr2;
}

/**
  * @name   TIM_OC4PreloadConfig
  * @brief  EN: Enables or disables the TIMx peripheral Preload register on CCR4.
  *         CN: ûCCR4ϵTIMxԤؼĴ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_OCPreload: 
  *         EN: new state of the TIMx peripheral Preload register.
  *         CN: TIMxԤؼĴ״̬
  *             @arg TIM_OCPreload_Enable
  *             @arg TIM_OCPreload_Disable
  * @retval None
  */
void TIM_OC4PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload)
{
  uint32_t tmpccmr2 = 0;

  /* Check the parameters */

  tmpccmr2 = TIMx->CCMR2;
  /* Reset the OC4PE Bit */
  tmpccmr2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Output_OC4PE_Msk);
  /* Enable or Disable the Output Compare Preload feature */
  tmpccmr2 |= (uint32_t)(TIM_OCPreload << 8);
  /* Write to TIMx CCMR2 register */
  TIMx->CCMR2 = tmpccmr2;
}

/**
  * @name   TIM_OC5PreloadConfig
  * @brief  EN: Enables or disables the TIMx peripheral Preload register on CCR5.
  *         CN: ûCCR5ϵTIMxԤؼĴ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  TIM_OCPreload: 
  *         EN: new state of the TIMx peripheral Preload register.
  *         CN: TIMxԤؼĴ״̬
  *             @arg TIM_OCPreload_Enable
  *             @arg TIM_OCPreload_Disable
  * @retval None
  */
void TIM_OC5PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload)
{
  uint32_t tmpccmr3 = 0;

  /* Check the parameters */

  tmpccmr3 = TIMx->CCMR3;
  /* Reset the OC5PE Bit */
  tmpccmr3 &= (uint32_t) ~((uint32_t)TIM_CCMR3_Output_OC5PE_Msk);
  /* Enable or Disable the Output Compare Preload feature */
  tmpccmr3 |= (uint32_t)TIM_OCPreload;
  /* Write to TIMx CCMR3 register */
  TIMx->CCMR3 = tmpccmr3;
}

/**
  * @name   TIM_OC1FastConfig
  * @brief  EN: Configures the TIMx Output Compare 1 Fast feature.
  *         CN: TIMxȽ1ٹܡ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_OCFast: 
  *         EN: new state of the Output Compare Fast Enable Bit.
  *         CN: ȽϿʹλ״̬
  *             @arg TIM_OCFast_Enable
  *             @arg TIM_OCFast_Disable
  * @retval None
  */
void TIM_OC1FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast)
{
  uint32_t tmpccmr1 = 0;

  /* Check the parameters */

  /* Get the TIMx CCMR1 register value */
  tmpccmr1 = TIMx->CCMR1;
  /* Reset the OC1FE Bit */
  tmpccmr1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC1FE_Msk);
  /* Enable or Disable the Output Compare Fast Bit */
  tmpccmr1 |= TIM_OCFast;
  /* Write to TIMx CCMR1 */
  TIMx->CCMR1 = tmpccmr1;
}

/**
  * @name   TIM_OC2FastConfig
  * @brief  EN: Configures the TIMx Output Compare 2 Fast feature.
  *         CN: TIMxȽ2ٹܡ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @param  TIM_OCFast: 
  *         EN: new state of the Output Compare Fast Enable Bit.
  *         CN: ȽϿʹλ״̬
  *             @arg TIM_OCFast_Enable
  *             @arg TIM_OCFast_Disable
  * @retval None
  */
void TIM_OC2FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast)
{
  uint32_t tmpccmr1 = 0;

  /* Check the parameters */

  /* Get the TIMx CCMR1 register value */
  tmpccmr1 = TIMx->CCMR1;
  /* Reset the OC2FE Bit */
  tmpccmr1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC2FE_Msk);
  /* Enable or Disable the Output Compare Fast Bit */
  tmpccmr1 |= (uint32_t)(TIM_OCFast << 8);
  /* Write to TIMx CCMR1 */
  TIMx->CCMR1 = tmpccmr1;
}

/**
  * @name   TIM_OC3FastConfig
  * @brief  EN: Configures the TIMx Output Compare 3 Fast feature.
  *         CN: TIMxȽ3ٹܡ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_OCFast: 
  *         EN: new state of the Output Compare Fast Enable Bit.
  *         CN: ȽϿʹλ״̬
  *             @arg TIM_OCFast_Enable
  *             @arg TIM_OCFast_Disable
  * @retval None
  */
void TIM_OC3FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast)
{
  uint32_t tmpccmr2 = 0;

  /* Check the parameters */

  /* Get the TIMx CCMR2 register value */
  tmpccmr2 = TIMx->CCMR2;
  /* Reset the OC3FE Bit */
  tmpccmr2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Output_OC3FE_Msk);
  /* Enable or Disable the Output Compare Fast Bit */
  tmpccmr2 |= TIM_OCFast;
  /* Write to TIMx CCMR2 */
  TIMx->CCMR2 = tmpccmr2;
}

/**
  * @name   TIM_OC4FastConfig
  * @brief  EN: Configures the TIMx Output Compare 4 Fast feature.
  *         CN: TIMxȽ4ٹܡ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_OCFast: 
  *         EN: new state of the Output Compare Fast Enable Bit.
  *         CN: ȽϿʹλ״̬
  *             @arg TIM_OCFast_Enable
  *             @arg TIM_OCFast_Disable
  * @retval None
  */
void TIM_OC4FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast)
{
  uint32_t tmpccmr2 = 0;

  /* Check the parameters */

  /* Get the TIMx CCMR2 register value */
  tmpccmr2 = TIMx->CCMR2;
  /* Reset the OC4FE Bit */
  tmpccmr2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Output_OC4FE_Msk);
  /* Enable or Disable the Output Compare Fast Bit */
  tmpccmr2 |= (uint32_t)(TIM_OCFast << 8);
  /* Write to TIMx CCMR2 */
  TIMx->CCMR2 = tmpccmr2;
}

/**
  * @name   TIM_OC5FastConfig
  * @brief  EN: Configures the TIMx Output Compare 5 Fast feature.
  *         CN: TIMxȽ5ٹܡ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_OCFast: 
  *         EN: new state of the Output Compare Fast Enable Bit.
  *         CN: ȽϿʹλ״̬
  *             @arg TIM_OCFast_Enable
  *             @arg TIM_OCFast_Disable
  * @retval None
  */
void TIM_OC5FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast)
{
  uint32_t tmpccmr3 = 0;

  /* Check the parameters */

  /* Get the TIMx CCMR2 register value */
  tmpccmr3 = TIMx->CCMR3;
  /* Reset the OC3FE Bit */
  tmpccmr3 &= (uint32_t) ~((uint32_t)TIM_CCMR3_Output_OC5FE_Msk);
  /* Enable or Disable the Output Compare Fast Bit */
  tmpccmr3 |= (uint32_t)TIM_OCFast;
  /* Write to TIMx CCMR3 */
  TIMx->CCMR3 = tmpccmr3;
}

/**
  * @name   TIM_ClearOC1Ref
  * @brief  EN: Clears or safeguards the OCREF1 signal on an external event.
  *         CN: 򱣻ⲿ¼OCREF1źš
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_OCClear: 
  *         EN: new state of the Output Compare Clear Enable Bit.
  *         CN: Ƚλ״̬
  *             @arg TIM_OCClear_Enable
  *             @arg TIM_OCClear_Disable
  * @retval None
  */
void TIM_ClearOC1Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear)
{
  uint32_t tmpccmr1 = 0;

  /* Check the parameters */

  tmpccmr1 = TIMx->CCMR1;
  /* Reset the OC1CE Bit */
  tmpccmr1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC1CE_Msk);
  /* Enable or Disable the Output Compare Clear Bit */
  tmpccmr1 |= TIM_OCClear;
  /* Write to TIMx CCMR1 register */
  TIMx->CCMR1 = tmpccmr1;
}

/**
  * @name   TIM_ClearOC2Ref
  * @brief  EN: Clears or safeguards the OCREF2 signal on an external event.
  *         CN: 򱣻ⲿ¼OCREF2źš
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @param  TIM_OCClear: 
  *         EN: new state of the Output Compare Clear Enable Bit.
  *         CN: Ƚλ״̬
  *             @arg TIM_OCClear_Enable
  *             @arg TIM_OCClear_Disable
  * @retval None
  */
void TIM_ClearOC2Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear)
{
  uint32_t tmpccmr1 = 0;

  /* Check the parameters */

  tmpccmr1 = TIMx->CCMR1;
  /* Reset the OC2CE Bit */
  tmpccmr1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Output_OC2CE_Msk);
  /* Enable or Disable the Output Compare Clear Bit */
  tmpccmr1 |= (uint32_t)(TIM_OCClear << 8);
  /* Write to TIMx CCMR1 register */
  TIMx->CCMR1 = tmpccmr1;
}

/**
  * @name   TIM_ClearOC3Ref
  * @brief  EN: Clears or safeguards the OCREF3 signal on an external event.
  *         CN: 򱣻ⲿ¼OCREF3źš
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_OCClear: 
  *         EN: new state of the Output Compare Clear Enable Bit.
  *         CN: Ƚλ״̬
  *             @arg TIM_OCClear_Enable
  *             @arg TIM_OCClear_Disable
  * @retval None
  */
void TIM_ClearOC3Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear)
{
  uint32_t tmpccmr2 = 0;

  /* Check the parameters */

  tmpccmr2 = TIMx->CCMR2;
  /* Reset the OC3CE Bit */
  tmpccmr2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Output_OC3CE_Msk);
  /* Enable or Disable the Output Compare Clear Bit */
  tmpccmr2 |= TIM_OCClear;
  /* Write to TIMx CCMR2 register */
  TIMx->CCMR2 = tmpccmr2;
}

/**
  * @name   TIM_ClearOC4Ref
  * @brief  EN: Clears or safeguards the OCREF4 signal on an external event.
  *         CN: 򱣻ⲿ¼OCREF4źš
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_OCClear: 
  *         EN: new state of the Output Compare Clear Enable Bit.
  *         CN: Ƚλ״̬
  *             @arg TIM_OCClear_Enable
  *             @arg TIM_OCClear_Disable
  * @retval None
  */
void TIM_ClearOC4Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear)
{
  uint32_t tmpccmr2 = 0;

  /* Check the parameters */

  tmpccmr2 = TIMx->CCMR2;
  /* Reset the OC4CE Bit */
  tmpccmr2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Output_OC4CE_Msk);
  /* Enable or Disable the Output Compare Clear Bit */
  tmpccmr2 |= (uint32_t)(TIM_OCClear << 8);
  /* Write to TIMx CCMR2 register */
  TIMx->CCMR2 = tmpccmr2;
}

/**
  * @name   TIM_ClearOC5Ref
  * @brief  EN: Clears or safeguards the OCREF5 signal on an external event.
  *         CN: 򱣻ⲿ¼OCREF5źš
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  TIM_OCClear: 
  *         EN: new state of the Output Compare Clear Enable Bit.
  *         CN: Ƚλ״̬
  *             @arg TIM_OCClear_Enable
  *             @arg TIM_OCClear_Disable
  * @retval None
  */
void TIM_ClearOC5Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear)
{
  uint32_t tmpccmr3 = 0;

  /* Check the parameters */

  tmpccmr3 = TIMx->CCMR3;
  /* Reset the OC5CE Bit */
  tmpccmr3 &= (uint32_t) ~((uint32_t)TIM_CCMR3_Output_OC5CE_Msk);
  /* Enable or Disable the Output Compare Clear Bit */
  tmpccmr3 |= (uint32_t)TIM_OCClear;
  /* Write to TIMx CCMR3 register */
  TIMx->CCMR3 = tmpccmr3;
}

/**
  * @name   TIM_OC1PolarityConfig
  * @brief  EN: Configures the TIMx Channel 1 polarity.
  *         CN: TIMxͨ1ԡ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_OCPolarity: 
  *         EN: specifies the OC1 Polarity.
  *         CN: ָOC1ԡ
  *             @arg TIM_OCPolarity_High
  *             @arg TIM_OCPolarity_Low
  * @retval None
  */
void TIM_OC1PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity)
{
  uint32_t tmpccer = 0;

  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC1P Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC1P_Msk);
  tmpccer |= TIM_OCPolarity;
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC1NPolarityConfig
  * @brief  EN: Configures the TIMx Channel 1N polarity.
  *         CN: TIMxͨ1Nԡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_CCxN: 
  *         EN: specifies the OC1N Polarity.
  *         CN: ָOC1Nԡ
  *             @arg TIM_OCNPolarity_High
  *             @arg TIM_OCNPolarity_Low
  * @retval None
  */
void TIM_OC1NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity)
{
  uint32_t tmpccer = 0;
  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC1NP Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC1NP_Msk);
  tmpccer |= TIM_OCNPolarity;
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC2PolarityConfig
  * @brief  EN: Configures the TIMx Channel 2 polarity.
  *         CN: TIMxͨ3ԡ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @param  TIM_CCxN: 
  *         EN: specifies the OC2 Polarity.
  *         CN: ָOC2ԡ
  *             @arg TIM_OCPolarity_High
  *             @arg TIM_OCPolarity_Low
  * @retval None
  */
void TIM_OC2PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity)
{
  uint32_t tmpccer = 0;

  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC2P Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC2P_Msk);
  tmpccer |= (uint32_t)(TIM_OCPolarity << TIM_CCER_CC2E_Pos);
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC2NPolarityConfig
  * @brief  EN: Configures the TIMx Channel 2N polarity.
  *         CN: TIMxͨ2Nԡ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @param  TIM_CCxN:
  *         EN: specifies the OC2N Polarity.
  *         CN: ָOC2Nԡ
  *             @arg TIM_OCNPolarity_High
  *             @arg TIM_OCNPolarity_Low
  * @retval None
  */
void TIM_OC2NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity)
{
  uint32_t tmpccer = 0;
  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC2NP Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC2NP_Msk);
  tmpccer |= (uint32_t)(TIM_OCNPolarity << TIM_CCER_CC2E_Pos);
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC3PolarityConfig
  * @brief  EN: Configures the TIMx Channel 3 polarity.
  *         CN: TIMxͨ3ԡ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_CCxN:
  *         EN: specifies the OC3 Polarity.
  *         CN: ָOC3ԡ
  *             @arg TIM_OCPolarity_High
  *             @arg TIM_OCPolarity_Low
  * @retval None
  */
void TIM_OC3PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity)
{
  uint32_t tmpccer = 0;

  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC3P Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC3P_Msk);
  tmpccer |= (uint32_t)(TIM_OCPolarity << TIM_CCER_CC3E_Pos);
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC3NPolarityConfig
  * @brief  EN: Configures the TIMx Channel 3N polarity.
  *         CN: TIMxͨ3Nԡ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_CCxN:
  *         EN: specifies the OC3N Polarity.
  *         CN: ָOC3Nԡ
  *             @arg TIM_OCNPolarity_High
  *             @arg TIM_OCNPolarity_Low
  * @retval None
  */
void TIM_OC3NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity)
{
  uint32_t tmpccer = 0;

  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC3NP Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC3NP_Msk);
  tmpccer |= (uint32_t)(TIM_OCNPolarity << TIM_CCER_CC3E_Pos);
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC4PolarityConfig
  * @brief  EN: Configures the TIMx Channel 4 polarity.
  *         CN: TIMxͨ4ԡ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_CCxN:
  *         EN: specifies the OC4 Polarity.
  *         CN: ָOC4ԡ
  *             @arg TIM_OCPolarity_High
  *             @arg TIM_OCPolarity_Low
  * @retval None
  */
void TIM_OC4PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity)
{
  uint32_t tmpccer = 0;

  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC4P Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC4P_Msk);
  tmpccer |= (uint32_t)(TIM_OCPolarity << TIM_CCER_CC4E_Pos);
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC4NPolarityConfig
  * @brief  EN: Configures the TIMx Channel 4N polarity.
  *         CN: TIMxͨ4Nԡ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_CCxN:
  *         EN: specifies the OC4N Polarity.
  *         CN: ָOC4Nԡ
  *             @arg TIM_OCNPolarity_High
  *             @arg TIM_OCNPolarity_Low
  * @retval None
  */
void TIM_OC4NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity)
{
  uint32_t tmpccer = 0;

  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC3NP Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC4NP_Msk);
  tmpccer |= (uint32_t)(TIM_OCNPolarity << TIM_CCER_CC4E_Pos);
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC5PolarityConfig
  * @brief  EN: Configures the TIMx Channel 5 polarity.
  *         CN: TIMxͨ5ԡ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  TIM_CCxN:
  *         EN: specifies the OC5 Polarity.
  *         CN: ָOC5ԡ
  *             @arg TIM_OCPolarity_High
  *             @arg TIM_OCPolarity_Low
  * @retval None
  */
void TIM_OC5PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity)
{
  uint32_t tmpccer = 0;

  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC4P Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC5P_Msk);
  tmpccer |= (uint32_t)(TIM_OCPolarity << TIM_CCER_CC5E_Pos);
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_OC5NPolarityConfig
  * @brief  EN: Configures the TIMx Channel 5N polarity.
  *         CN: TIMxͨ5Nԡ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  TIM_CCxN:
  *         EN: specifies the OC5N Polarity.
  *         CN: ָOC5Nԡ
  *             @arg TIM_OCNPolarity_High
  *             @arg TIM_OCNPolarity_Low
  * @retval None
  */
void TIM_OC5NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity)
{
  uint32_t tmpccer = 0;

  /* Check the parameters */

  tmpccer = TIMx->CCER;
  /* Set or Reset the CC3NP Bit */
  tmpccer &= (uint32_t) ~((uint32_t)TIM_CCER_CC5NP_Msk);
  tmpccer |= (uint32_t)(TIM_OCNPolarity << TIM_CCER_CC5E_Pos);
  /* Write to TIMx CCER register */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_SelectOCREFClear
  * @brief  EN: Selects the OCReference Clear source.
  *         CN: ѡOCReferenceԴ
  * @param  TIMx:
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_CCxN:
  *         EN: specifies the OCReference Clear source.
  *         CN: ָOCReferenceԴ
  *             @arg TIM_OCReferenceClear_ETRF
  *             @arg TIM_OCReferenceClear_OCREFCLR
  * @retval None
  */
void TIM_SelectOCREFClear(TIM_TypeDef *TIMx, uint32_t TIM_OCReferenceClear)
{
  /* Check the parameters */

  /* Set the TIM_OCReferenceClear source */
  TIMx->SMCR &= (uint32_t) ~((uint32_t)TIM_SMCR_OCCS_Msk);
  TIMx->SMCR |= TIM_OCReferenceClear;
}

/**
  * @name   TIM_CCxNCmd
  * @brief  EN: Enables or disables the TIM Capture Compare Channel x.
  *         CN: ûTIMȽͨx
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_Channel: 
  *         EN: specifies the TIM Channel
  *         CN: ָTIMͨ
  *             @arg TIM_Channel_1
  *             @arg TIM_Channel_2
  *             @arg TIM_Channel_3 
  *             @arg TIM_Channel_4
  *             @arg TIM_Channel_5
  * @param  TIM_CCx: 
  *         EN: specifies the TIM Channel CCxE bit new state.
  *         CN: ָTIMŵCCxEλ״̬
  *             @arg TIM_CCx_Disable
  *             @arg TIM_CCx_Enable
  * @retval None
  */
void TIM_CCxCmd(TIM_TypeDef *TIMx, uint32_t TIM_Channel, uint32_t TIM_CCx)
{
  uint32_t tmp = 0;

  /* Check the parameters */

  tmp = TIM_CCER_CC1E_Msk << TIM_Channel;

  /* Reset the CCxE Bit */
  TIMx->CCER &= (uint32_t)~tmp;

  /* Set or reset the CCxE Bit */
  TIMx->CCER |= (uint32_t)(TIM_CCx << TIM_Channel);
}

/**
  * @name   TIM_CCxNCmd
  * @brief  EN: Enables or disables the TIM Capture Compare Channel xN.
  *         CN: ûTIMȽͨxN
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_Channel: 
  *         EN: specifies the TIM Channel
  *         CN: ָTIMͨ
  *             @arg TIM_Channel_1
  *             @arg TIM_Channel_2
  *             @arg TIM_Channel_3
  *             @arg TIM_Channel_4
  *             @arg TIM_Channel_5
  * @param  TIM_CCxN: 
  *         EN: specifies the TIM Channel CCxNE bit new state.
  *         CN: ָTIMŵCCxNEص״̬
  *             @arg TIM_CCxN_Disable
  *             @arg TIM_CCxN_Enable
  * @retval None
  */
void TIM_CCxNCmd(TIM_TypeDef *TIMx, uint32_t TIM_Channel, uint32_t TIM_CCxN)
{
  uint32_t tmp = 0;

  /* Check the parameters */
  tmp = TIM_CCER_CC1NE_Msk << TIM_Channel;

  /* Reset the CCxNE Bit */
  TIMx->CCER &= (uint32_t)~tmp;

  /* Set or reset the CCxNE Bit */
  TIMx->CCER |= (uint32_t)(TIM_CCxN << TIM_Channel);
}

/**
  * @name   TIM_SelectCOM
  * @brief  EN: Selects the TIM peripheral Commutation event.
  *         CN: ѡTIMCOM¼
  * @param  TIMx: EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *               CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: new state of the Commutation event.
  *         CN: COM¼״̬
  * @retval None
  */
void TIM_SelectCOM(TIM_TypeDef *TIMx, FunctionalState NewState)
{
  /* Check the parameters */

  if (NewState != DISABLE)
  {
    /* Set the COM Bit */
    TIMx->CR2 |= TIM_CR2_CCUS_Msk;
  }
  else
  {
    /* Reset the COM Bit */
    TIMx->CR2 &= (uint32_t) ~((uint32_t)TIM_CR2_CCUS_Msk);
  }
}

/**
  * @name   TIM_ICInit
  * @brief  EN: Initializes the TIM peripheral according to the specified
  *             parameters in the TIM_ICInitStruct.
  *         CN: TIM_ICInitStructָĲʼTIMΧ豸
  * @param  TIMx: EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *               CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_ICInitStruct: 
  *         EN: pointer to a TIM_ICInitTypeDef structure
  *             that contains the configuration information for the specified TIM
  *         CN:ָTIM_ICInitTypeDefṹָ룬ýṹָTIMϢ
  * @retval None
  */
void TIM_ICInit(TIM_TypeDef *TIMx, TIM_ICInitTypeDef *TIM_ICInitStruct)
{
  /* Check the parameters */

  if (TIM_ICInitStruct->TIM_Channel == TIM_Channel_1)
  {

    /* TI1 Configuration */
    TI1_Config(TIMx, TIM_ICInitStruct->TIM_ICPolarity,
               TIM_ICInitStruct->TIM_ICSelection,
               TIM_ICInitStruct->TIM_ICFilter);
    /* Set the Input Capture Prescaler value */
    TIM_SetIC1Prescaler(TIMx, TIM_ICInitStruct->TIM_ICPrescaler);
  }
  else if (TIM_ICInitStruct->TIM_Channel == TIM_Channel_2)
  {

    /* TI2 Configuration */
    TI2_Config(TIMx, TIM_ICInitStruct->TIM_ICPolarity,
               TIM_ICInitStruct->TIM_ICSelection,
               TIM_ICInitStruct->TIM_ICFilter);
    /* Set the Input Capture Prescaler value */
    TIM_SetIC2Prescaler(TIMx, TIM_ICInitStruct->TIM_ICPrescaler);
  }
  else if (TIM_ICInitStruct->TIM_Channel == TIM_Channel_3)
  {

    /* TI3 Configuration */
    TI3_Config(TIMx, TIM_ICInitStruct->TIM_ICPolarity,
               TIM_ICInitStruct->TIM_ICSelection,
               TIM_ICInitStruct->TIM_ICFilter);
    /* Set the Input Capture Prescaler value */
    TIM_SetIC3Prescaler(TIMx, TIM_ICInitStruct->TIM_ICPrescaler);
  }
  else if (TIM_ICInitStruct->TIM_Channel == TIM_Channel_4)
  {

    /* TI4 Configuration */
    TI4_Config(TIMx, TIM_ICInitStruct->TIM_ICPolarity,
               TIM_ICInitStruct->TIM_ICSelection,
               TIM_ICInitStruct->TIM_ICFilter);
    /* Set the Input Capture Prescaler value */
    TIM_SetIC4Prescaler(TIMx, TIM_ICInitStruct->TIM_ICPrescaler);
  }
  else if (TIM_ICInitStruct->TIM_Channel == TIM_Channel_5)
  {

    /* TI5 Configuration */
    TI5_Config(TIMx, TIM_ICInitStruct->TIM_ICPolarity,
               TIM_ICInitStruct->TIM_ICSelection,
               TIM_ICInitStruct->TIM_ICFilter);
    /* Set the Input Capture Prescaler value */
    TIM_SetIC5Prescaler(TIMx, TIM_ICInitStruct->TIM_ICPrescaler);
  }
}

/**
  * @name   TIM_ICStructInit
  * @brief  EN: Fills each TIM_ICInitStruct member with its default value.
  *         CN: Ĭֵÿ TIM_ICInitStruct Ա
  * @param  TIM_ICInitStruct:
  *         EN: pointer to a TIM_ICInitTypeDef structure that contains the 
  *             configuration information for the specified TIM Base peripheral.
  *         CN: ָ TIM_ICInitTypeDef ṹָ룬ýṹָ TIM ICϢ
  * @retval None
  */
void TIM_ICStructInit(TIM_ICInitTypeDef *TIM_ICInitStruct)
{
  /* Set the default configuration */
  TIM_ICInitStruct->TIM_Channel = TIM_Channel_1;
  TIM_ICInitStruct->TIM_ICPolarity = TIM_ICPolarity_Rising;
  TIM_ICInitStruct->TIM_ICSelection = TIM_ICSelection_DirectTI;
  TIM_ICInitStruct->TIM_ICPrescaler = TIM_ICPSC_DIV1;
  TIM_ICInitStruct->TIM_ICFilter = 0x00;
}

/**
  * @name   TIM_PWMIConfig
  * @brief  EN: Configures the TIM peripheral according to the specified
  *             parameters in the TIM_ICInitStruct to measure an external PWM signal.
  *         CN: TIM_ICInitStructеָTIMΧ豸ԲⲿPWMźš
  * @param  TIMx: EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *               CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_ICInitStruct: 
  *         EN: pointer to a TIM_ICInitTypeDef structure
  *             that contains the configuration information for the specified TIM
  *         CN:ָTIM_ICInitTypeDefṹָ룬ýṹָTIMϢ
  * @retval None
  */
void TIM_PWMIConfig(TIM_TypeDef *TIMx, TIM_ICInitTypeDef *TIM_ICInitStruct)
{
  uint32_t icoppositepolarity = TIM_ICPolarity_Rising;
  uint32_t icoppositeselection = TIM_ICSelection_DirectTI;
  /* Check the parameters */

  /* Select the Opposite Input Polarity */
  if (TIM_ICInitStruct->TIM_ICPolarity == TIM_ICPolarity_Rising)
  {
    icoppositepolarity = TIM_ICPolarity_Falling;
  }
  else
  {
    icoppositepolarity = TIM_ICPolarity_Rising;
  }
  /* Select the Opposite Input */
  if (TIM_ICInitStruct->TIM_ICSelection == TIM_ICSelection_DirectTI)
  {
    icoppositeselection = TIM_ICSelection_IndirectTI;
  }
  else
  {
    icoppositeselection = TIM_ICSelection_DirectTI;
  }
  if (TIM_ICInitStruct->TIM_Channel == TIM_Channel_1)
  {
    /* TI1 Configuration */
    TI1_Config(TIMx, TIM_ICInitStruct->TIM_ICPolarity, TIM_ICInitStruct->TIM_ICSelection,
               TIM_ICInitStruct->TIM_ICFilter);
    /* Set the Input Capture Prescaler value */
    TIM_SetIC1Prescaler(TIMx, TIM_ICInitStruct->TIM_ICPrescaler);
    /* TI2 Configuration */
    TI2_Config(TIMx, icoppositepolarity, icoppositeselection, TIM_ICInitStruct->TIM_ICFilter);
    /* Set the Input Capture Prescaler value */
    TIM_SetIC2Prescaler(TIMx, TIM_ICInitStruct->TIM_ICPrescaler);
  }
  else
  {
    /* TI2 Configuration */
    TI2_Config(TIMx, TIM_ICInitStruct->TIM_ICPolarity, TIM_ICInitStruct->TIM_ICSelection,
               TIM_ICInitStruct->TIM_ICFilter);
    /* Set the Input Capture Prescaler value */
    TIM_SetIC2Prescaler(TIMx, TIM_ICInitStruct->TIM_ICPrescaler);
    /* TI1 Configuration */
    TI1_Config(TIMx, icoppositepolarity, icoppositeselection, TIM_ICInitStruct->TIM_ICFilter);
    /* Set the Input Capture Prescaler value */
    TIM_SetIC1Prescaler(TIMx, TIM_ICInitStruct->TIM_ICPrescaler);
  }
}

/**
  * @name   TIM_GetCapture1
  * @brief  EN: Gets the TIMx Input Capture 1 value.
  *         CN: ȡTIMx Input Capture 1ֵ
  * @param  TIMx:
  *         EN: where x can be 1,2,15,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval EN: Capture Compare 1 Register value.
  *         CN: Ƚ1Ĵֵ
  */
uint32_t TIM_GetCapture1(TIM_TypeDef *TIMx)
{
  /* Check the parameters */

  /* Get the Capture 1 Register value */
  return TIMx->CCR1;
}

/**
  * @name   TIM_GetCapture2
  * @brief  EN: Gets the TIMx Input Capture 2 value.
  *         CN: ȡTIMx Input Capture 2ֵ
  * @param  TIMx:
  *         EN: where x can be 1,2,15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @retval EN: Capture Compare 2 Register value.
  *         CN: Ƚ2Ĵֵ
  */
uint32_t TIM_GetCapture2(TIM_TypeDef *TIMx)
{
  /* Check the parameters */

  /* Get the Capture 2 Register value */
  return TIMx->CCR2;
}

/**
  * @name   TIM_GetCapture3
  * @brief  EN: Gets the TIMx Input Capture 3 value.
  *         CN: ȡTIMx Input Capture 3ֵ
  * @param  TIMx:
  *         EN: where x can be 1,2,15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @retval EN: Capture Compare 3 Register value.
  *         CN: Ƚ3Ĵֵ
  */
uint32_t TIM_GetCapture3(TIM_TypeDef *TIMx)
{
  /* Check the parameters */

  /* Get the Capture 3 Register value */
  return TIMx->CCR3;
}

/**
  * @name   TIM_GetCapture4
  * @brief  EN: Gets the TIMx Input Capture 4 value.
  *         CN: ȡTIMx Input Capture 4ֵ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @retval EN: Capture Compare 4 Register value.
  *         CN: Ƚ4Ĵֵ
  */
uint32_t TIM_GetCapture4(TIM_TypeDef *TIMx)
{
  /* Check the parameters */

  /* Get the Capture 4 Register value */
  return TIMx->CCR4;
}

/**
  * @name   TIM_GetCapture5
  * @brief  EN: Gets the TIMx Input Capture 5 value.
  *         CN: ȡTIMx Input Capture 5ֵ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @retval EN: Capture Compare 5 Register value.
  *         CN: Ƚ5Ĵֵ
  */
uint32_t TIM_GetCapture5(TIM_TypeDef *TIMx)
{
  /* Check the parameters */

  /* Get the Capture 4 Register value */
  return TIMx->CCR5;
}

/**
  * @name   TIM_SetIC1Prescaler
  * @brief  EN: Sets the TIMx Input Capture 1 prescaler.
  *         CN: TIMx Input Capture 1 ԤƵ
  * @param  TIMx:
  *         EN: where x can be 1,2,15,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_ICPSC: 
  *         EN: specifies the Input Capture1 prescaler new value.
  *         CN: ָInput Capture1ԤƵֵ
  *             @arg TIM_ICPSC_DIV1
  *             @arg TIM_ICPSC_DIV2
  *             @arg TIM_ICPSC_DIV4
  *             @arg TIM_ICPSC_DIV8
  * @retval None
  */
void TIM_SetIC1Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC)
{
  /* Check the parameters */

  /* Reset the IC1PSC Bits */
  TIMx->CCMR1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Input_IC1PSC_Msk);
  /* Set the IC1PSC value */
  TIMx->CCMR1 |= TIM_ICPSC;
}


/**
  * @name   TIM_SetIC2Prescaler
  * @brief  EN: Sets the TIMx Input Capture 2 prescaler.
  *         CN: TIMx Input Capture 2 ԤƵ
  * @param  TIMx:
  *         EN: where x can be 1,2,15 or 16 to select the TIM peripheral.
  *         CN: x1,2,1516ѡTIMΧ豸
  * @param  TIM_ICPSC: 
  *         EN: specifies the Input Capture2 prescaler new value.
  *         CN: ָInput Capture2ԤƵֵ
  *             @arg TIM_ICPSC_DIV1
  *             @arg TIM_ICPSC_DIV2
  *             @arg TIM_ICPSC_DIV4
  *             @arg TIM_ICPSC_DIV8
  * @retval None
  */
void TIM_SetIC2Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC)
{
  /* Check the parameters */

  /* Reset the IC2PSC Bits */
  TIMx->CCMR1 &= (uint32_t) ~((uint32_t)TIM_CCMR1_Input_IC2PSC_Msk);
  /* Set the IC2PSC value */
  TIMx->CCMR1 |= (uint32_t)(TIM_ICPSC << 8);
}


/**
  * @name   TIM_SetIC3Prescaler
  * @brief  EN: Sets the TIMx Input Capture 3 prescaler.
  *         CN: TIMx Input Capture 3 ԤƵ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_ICPSC: 
  *         EN: specifies the Input Capture3 prescaler new value.
  *         CN: ָInput Capture3ԤƵֵ
  *             @arg TIM_ICPSC_DIV1
  *             @arg TIM_ICPSC_DIV2
  *             @arg TIM_ICPSC_DIV4
  *             @arg TIM_ICPSC_DIV8
  * @retval None
  */
void TIM_SetIC3Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC)
{
  /* Check the parameters */

  /* Reset the IC3PSC Bits */
  TIMx->CCMR2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Input_IC3PSC_Msk);
  /* Set the IC3PSC value */
  TIMx->CCMR2 |= TIM_ICPSC;
}

/**
  * @name   TIM_SetIC4Prescaler
  * @brief  EN: Sets the TIMx Input Capture 5 prescaler.
  *         CN: TIMx Input Capture 5 ԤƵ
  * @param  TIMx:
  *         EN: where x can be 1 or 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_ICPSC: 
  *         EN: specifies the Input Capture5 prescaler new value.
  *         CN: ָInput Capture5ԤƵֵ
  *             @arg TIM_ICPSC_DIV1
  *             @arg TIM_ICPSC_DIV2
  *             @arg TIM_ICPSC_DIV4
  *             @arg TIM_ICPSC_DIV8
  * @retval None
  */
void TIM_SetIC4Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC)
{
  /* Check the parameters */

  /* Reset the IC4PSC Bits */
  TIMx->CCMR2 &= (uint32_t) ~((uint32_t)TIM_CCMR2_Input_IC4PSC_Msk);
  /* Set the IC4PSC value */
  TIMx->CCMR2 |= (uint32_t)(TIM_ICPSC << 8);
}

/**
  * @name   TIM_SetIC5Prescaler
  * @brief  EN: Sets the TIMx Input Capture 5 prescaler.
  *         CN: TIMx Input Capture 5ԤƵ
  * @param  TIMx:
  *         EN: where x can be 1, to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  TIM_ICPSC: 
  *         EN: specifies the Input Capture5 prescaler new value.
  *         CN: ָInput Capture5ԤƵֵ
  *             @arg TIM_ICPSC_DIV1
  *             @arg TIM_ICPSC_DIV2
  *             @arg TIM_ICPSC_DIV4
  *             @arg TIM_ICPSC_DIV8
  * @retval None
  */
void TIM_SetIC5Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC)
{
  /* Check the parameters */

  /* Reset the IC3PSC Bits */
  TIMx->CCMR3 &= (uint32_t) ~((uint32_t)TIM_CCMR3_Input_IC5PSC_Msk);
  /* Set the IC3PSC value */
  TIMx->CCMR3 |= TIM_ICPSC;
}

/**
  * @name   TIM_ITConfig
  * @brief  EN: Enables or disables the specified TIM interrupts.
  *         CN: ûָTIMжϡ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_IT: 
  *         EN: specifies the TIM interrupts sources to be enabled or disabled.
  *         CN: ָҪûõTIMжԴ
  *             @arg TIM_IT_Update
  *             @arg TIM_IT_CC1
  *             @arg TIM_IT_CC2
  *             @arg TIM_IT_CC3
  *             @arg TIM_IT_CC4
  *             @arg TIM_IT_COM
  *             @arg TIM_IT_Trigger
  *             @arg TIM_IT_Break
  *             @arg TIM_IT_DMA
  *             @arg TIM_IT_CC5
  * @param  NewState:
  *         EN: new state of the TIM interrupts.
  *         CN: TIMжϵ״̬
  * @retval None
  */
void TIM_ITConfig(TIM_TypeDef *TIMx, uint32_t TIM_IT, FunctionalState NewState)
{
  /* Check the parameters */

  if (NewState != DISABLE)
  {
    /* Enable the Interrupt sources */
    TIMx->DIER |= TIM_IT;
  }
  else
  {
    /* Disable the Interrupt sources */
    TIMx->DIER &= (uint32_t)~TIM_IT;
  }
}

/**
  * @name   TIM_GenerateEvent
  * @brief  EN: Configures the TIMx event to be generate by software.
  *         CN: TIMx¼Ϊɡ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_EventSource: 
  *         EN: specifies the event source.
  *         CN: ָ¼Դ
  *             @arg TIM_EventSource_Update
  *             @arg TIM_EventSource_CC1
  *             @arg TIM_EventSource_CC2
  *             @arg TIM_EventSource_CC3
  *             @arg TIM_EventSource_CC4
  *             @arg TIM_EventSource_COM
  *             @arg TIM_EventSource_Trigger
  *             @arg TIM_EventSource_Break
  *             @arg TIM_EventSource_Break2
  * @retval None
  */
void TIM_GenerateEvent(TIM_TypeDef *TIMx, uint32_t TIM_EventSource)
{
  /* Check the parameters */

  /* Set the event sources */
  TIMx->EGR = TIM_EventSource;
}

/**
  * @name   TIM_GetFlagStatus
  * @brief  EN: Checks whether the specified TIM flag is set or not.
  *         CN: ǷָTIM־
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_FLAG: 
  *         EN: specifies the flag to check.
  *         CN: ָҪı־
  *             @arg TIM_FLAG_Update
  *             @arg TIM_FLAG_CC1
  *             @arg TIM_FLAG_CC2
  *             @arg TIM_FLAG_CC3
  *             @arg TIM_FLAG_CC4
  *             @arg TIM_FLAG_COM
  *             @arg TIM_FLAG_Trigger
  *             @arg TIM_FLAG_Break
  *             @arg TIM_FLAG_Break2
  *             @arg TIM_FLAG_CC1OF
  *             @arg TIM_FLAG_CC2OF
  *             @arg TIM_FLAG_CC3OF
  *             @arg TIM_FLAG_CC4OF
  * @retval FlagStatus
  */
FlagStatus TIM_GetFlagStatus(TIM_TypeDef *TIMx, uint32_t TIM_FLAG)
{
  ITStatus bitstatus = RESET;

  /* Check the parameters */

  if ((TIMx->SR & TIM_FLAG) != (uint32_t)RESET)
  {
    bitstatus = SET;
  }
  else
  {
    bitstatus = RESET;
  }
  return bitstatus;
}

/**
  * @name   TIM_ClearFlag
  * @brief  EN: Clears the TIMx's pending flags.
  *         CN: TIMxĹ־
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_FLAG: 
  *         EN: specifies the flag bit to clear.
  *         CN: ָҪı־λ
  *             @arg TIM_FLAG_Update
  *             @arg TIM_FLAG_CC1
  *             @arg TIM_FLAG_CC2
  *             @arg TIM_FLAG_CC3
  *             @arg TIM_FLAG_CC4
  *             @arg TIM_FLAG_COM
  *             @arg TIM_FLAG_Trigger
  *             @arg TIM_FLAG_Break
  *             @arg TIM_FLAG_Break2
  *             @arg TIM_FLAG_CC1OF
  *             @arg TIM_FLAG_CC2OF
  *             @arg TIM_FLAG_CC3OF
  *             @arg TIM_FLAG_CC4OF
  * @retval None
  */
void TIM_ClearFlag(TIM_TypeDef *TIMx, uint32_t TIM_FLAG)
{
  /* Check the parameters */

  /* Clear the flags */
  TIMx->SR = (uint32_t)~TIM_FLAG;
}

/**
  * @name   TIM_GetITStatus
  * @brief  EN: Checks whether the TIM interrupt has occurred or not.
  *         CN: TIMжǷ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_IT: 
  *         EN: specifies the pending bit to clear.
  *         CN: ָҪĹλ
  *             @arg TIM_IT_Update
  *             @arg TIM_IT_CC1
  *             @arg TIM_IT_CC2
  *             @arg TIM_IT_CC3
  *             @arg TIM_IT_CC4
  *             @arg TIM_IT_COM
  *             @arg TIM_IT_Trigger
  *             @arg TIM_IT_Break
  *             @arg TIM_IT_DMA
  *             @arg TIM_IT_CC5
  * @retval ITStatus
  */
ITStatus TIM_GetITStatus(TIM_TypeDef *TIMx, uint32_t TIM_IT)
{
  ITStatus bitstatus = RESET;
  uint32_t itstatus = 0x0, itenable = 0x0;

  /* Check the parameters */

  itstatus = TIMx->SR & TIM_IT;

  itenable = TIMx->DIER & TIM_IT;
  if ((itstatus != (uint32_t)RESET) && (itenable != (uint32_t)RESET))
  {
    bitstatus = SET;
  }
  else
  {
    bitstatus = RESET;
  }
  return bitstatus;
}

/**
  * @name   TIM_ClearITPendingBit
  * @brief  EN: Clears the TIMx's interrupt pending bits.
  *         CN: TIMxжϹλ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_IT: 
  *         EN: specifies the pending bit to clear.
  *         CN: ָҪĹλ
  *             @arg TIM_IT_Update
  *             @arg TIM_IT_CC1
  *             @arg TIM_IT_CC2
  *             @arg TIM_IT_CC3
  *             @arg TIM_IT_CC4
  *             @arg TIM_IT_COM
  *             @arg TIM_IT_Trigger
  *             @arg TIM_IT_Break
  *             @arg TIM_IT_DMA
  *             @arg TIM_IT_CC5
  * @retval None
  */
void TIM_ClearITPendingBit(TIM_TypeDef *TIMx, uint32_t TIM_IT)
{
  /* Check the parameters */

  /* Clear the IT pending Bit */
  TIMx->SR &= (uint32_t)~TIM_IT;
}

/**
  * @name   TIM_DMAConfig
  * @brief  EN: Enables or disables the TIMx's DMA Requests.
  *         CN: ûTIMxDMA
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_DMABase: 
  *         EN: DMA Base address.
  *         CN: DMAַ
  *             @arg TIM_DMABase_CR1
  *             @arg TIM_DMABase_CR2
  *             @arg TIM_DMABase_SMCR
  *             @arg TIM_DMABase_DIER
  *             @arg TIM_DMABase_SR
  *             @arg TIM_DMABase_EGR
  *             @arg TIM_DMABase_CCMR1
  *             @arg TIM_DMABase_CCMR2
  *             @arg TIM_DMABase_CCER
  *             @arg TIM_DMABase_CNT
  *             @arg TIM_DMABase_PSC
  *             @arg TIM_DMABase_ARR
  *             @arg TIM_DMABase_CCR1
  *             @arg TIM_DMABase_CCR2
  *             @arg TIM_DMABase_CCR3
  *             @arg TIM_DMABase_CCR4
  *             @arg TIM_DMABase_DCR
  *             @arg TIM_DMABase_OR
  * @param  TIM_DMABurstLength: 
  *         EN: DMA Burst length.
  *         CN: DMAͻȡ
  *             @arg TIM_DMABurstLength_1Transfer
  *             @arg ...
  *             @arg TIM_DMABurstLength_18Transfers
  * @retval None
  */
void TIM_DMAConfig(TIM_TypeDef *TIMx, uint32_t TIM_DMABase, uint32_t TIM_DMABurstLength)
{
  /* Check the parameters */

  /* Set the DMA Base and the DMA Burst Length */
  TIMx->DCR = TIM_DMABase | TIM_DMABurstLength;
}

/**
  * @name   TIM_WriteDMAR
  * @brief  EN: Write DMA transfer register value.
  *         CN: дDMAĴֵ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_DMAB: 
  *         EN: DMA transfer register value.
  *         CN: DMAĴֵ
  * @retval None
  */
void TIM_WriteDMAR(TIM_TypeDef *TIMx, uint32_t TIM_DMAB)
{
  TIMx->DMAR = TIM_DMAB;
}

/**
  * @name   TIM_ReadDMAR
  * @brief  EN: Read DMA transfer register value.
  *         CN: DMAĴֵ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval uint32_t
  */
uint32_t TIM_ReadDMAR(TIM_TypeDef *TIMx)
{
  return TIMx->DMAR;
}

 /**
  * @name   TIM_DMACmd
  * @brief  EN: Enables or disables the TIMx's DMA Requests.
  *         CN: ûTIMxDMA
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_DMASource: 
  *         EN: specifies the DMA Request sources.
  *         CN: ָDMAԴ
  *             @arg TIM_DMA_Update
  *             @arg TIM_DMA_CC1
  *             @arg TIM_DMA_CC2
  *             @arg TIM_DMA_CC3
  *             @arg TIM_DMA_CC4
  *             @arg TIM_DMA_COM
  *             @arg TIM_DMA_Trigger
  * @param  NewState: 
  *         EN: new state of the DMA Request sources.
  *         CN: DMAԴ״̬
  * @retval None
  */
void TIM_DMACmd(TIM_TypeDef *TIMx, uint32_t TIM_DMASource, FunctionalState NewState)
{
  /* Check the parameters */

  if (NewState != DISABLE)
  {
    /* Enable the DMA sources */
    TIMx->DIER |= TIM_DMASource;
  }
  else
  {
    /* Disable the DMA sources */
    TIMx->DIER &= (uint32_t)~TIM_DMASource;
  }
}

 /**
  * @name   TIM_SelectCCDMA
  * @brief  EN: Selects the TIMx peripheral Capture Compare DMA source.
  *         CN: ѡTIMxΧ豸ȽDMAԴ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: new state of the Capture Compare DMA source.
  *         CN: ȽDMAԴ״̬
  * @retval None
  */
void TIM_SelectCCDMA(TIM_TypeDef *TIMx, FunctionalState NewState)
{
  /* Check the parameters */

  if (NewState != DISABLE)
  {
    /* Set the CCDS Bit */
    TIMx->CR2 |= TIM_CR2_CCDS_Msk;
  }
  else
  {
    /* Reset the CCDS Bit */
    TIMx->CR2 &= (uint32_t) ~((uint32_t)TIM_CR2_CCDS_Msk);
  }
}

 /**
  * @name   TIM_InternalClockConfig
  * @brief  EN: Configures the TIMx internal Clock.
  *         CN: TIMxڲʱӡ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15 ,16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @retval None
  */
void TIM_InternalClockConfig(TIM_TypeDef *TIMx)
{
  /* Check the parameters */

  /* Disable slave mode to clock the prescaler directly with the internal clock */
  TIMx->SMCR &= (uint32_t)(~((uint32_t)TIM_SMCR_SMS_Msk));
}

/**
  * @name   TIM_ITRxExternalClockConfig
  * @brief  EN: Configures the TIMx Internal Trigger as External Clock.
  *         CN: TIMxڲΪⲿʱӡ
  * @param  TIMx:
  *         EN: where x can be 1, 2 or 15 to select the TIM peripheral.
  *         CN: x1,215ѡTIMΧ豸
  * @param  TIM_InputTriggerSource: 
  *         EN: Trigger source.
  *         CN: Դ
  *             @arg TIM_TS_ITR0   
  *             @arg TIM_TS_ITR1   
  *             @arg TIM_TS_ITR2   
  *             @arg TIM_TS_ITR3   
  *             @arg TIM_TS_TI1F_ED
  *             @arg TIM_TS_TI1FP1 
  *             @arg TIM_TS_TI2FP2 
  *             @arg TIM_TS_ETRF   
  * @retval None
  */
void TIM_ITRxExternalClockConfig(TIM_TypeDef *TIMx, uint32_t TIM_InputTriggerSource)
{
  /* Check the parameters */

  /* Select the Internal Trigger */
  TIM_SelectInputTrigger(TIMx, TIM_InputTriggerSource);
  /* Select the External clock mode1 */
  TIMx->SMCR |= TIM_SlaveMode_External1;
}


/**
  * @name   TIM_TIxExternalClockConfig
  * @brief  EN: Configures the TIMx Trigger as External Clock.
  *         CN: TIMxΪⲿʱӡ
  * @param  TIMx:
  *         EN: where x can be 1, 2 or 15 to select the TIM peripheral.
  *         CN: x1,215ѡTIMΧ豸
  * @param  TIM_TIxExternalCLKSource: 
  *         EN: Trigger source.
  *         CN: Դ
  *             @arg TIM_TIxExternalCLK1Source_TI1
  *             @arg TIM_TIxExternalCLK1Source_TI2
  *             @arg TIM_TIxExternalCLK1Source_TI1ED
  * @param  TIM_ICPolarity: 
  *         EN: specifies the TIx Polarity.
  *         CN: ָTIxԡ
  *             @arg TIM_ICPolarity_Rising
  *             @arg TIM_ICPolarity_Falling
  * @param  ICFilter: 
  *         EN: specifies the filter value.
  *         CN: ָ˲ֵ
  *         @range 0x0 - 0xF.
  * @retval None
  */
void TIM_TIxExternalClockConfig(TIM_TypeDef *TIMx, uint32_t TIM_TIxExternalCLKSource,
                                uint32_t TIM_ICPolarity, uint32_t ICFilter)
{
  /* Check the parameters */

  /* Configure the Timer Input Clock Source */
  if (TIM_TIxExternalCLKSource == TIM_TIxExternalCLK1Source_TI2)
  {
    TI2_Config(TIMx, TIM_ICPolarity, TIM_ICSelection_DirectTI, ICFilter);
  }
  else
  {
    TI1_Config(TIMx, TIM_ICPolarity, TIM_ICSelection_DirectTI, ICFilter);
  }
  /* Select the Trigger source */
  TIM_SelectInputTrigger(TIMx, TIM_TIxExternalCLKSource);
  /* Select the External clock mode1 */
  TIMx->SMCR |= TIM_SlaveMode_External1;
}

/**
  * @name   TIM_ETRClockMode1Config
  * @brief  EN: Configures the External clock Mode1.
  *         CN: ⲿʱģʽ1
  * @param  TIMx:
  *         EN: where x can be 1,2 or to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_ExtTRGPrescaler: 
  *         EN: The external Trigger Prescaler.
  *         CN: ⲿԤƵ
  *             @arg TIM_ExtTRGPSC_OFF
  *             @arg TIM_ExtTRGPSC_DIV2
  *             @arg TIM_ExtTRGPSC_DIV4
  *             @arg TIM_ExtTRGPSC_DIV8
  * @param  TIM_ExtTRGPolarity: 
  *         EN: The external Trigger Polarity.
  *         CN: ⲿԡ
  *             @arg TIM_ExtTRGPolarity_Inverted
  *             @arg TIM_ExtTRGPolarity_NonInverted
  * @param  ExtTRGFilter: 
  *         EN: External Trigger Filter.
  *         CN: ⲿ
  *         @range 0x00 - 0x0F.
  * @retval None
  */
void TIM_ETRClockMode1Config(TIM_TypeDef *TIMx, uint32_t TIM_ExtTRGPrescaler, uint32_t TIM_ExtTRGPolarity,
                             uint32_t ExtTRGFilter)
{
  uint32_t tmpsmcr = 0;

  /* Check the parameters */

  /* Configure the ETR Clock source */
  TIM_ETRConfig(TIMx, TIM_ExtTRGPrescaler, TIM_ExtTRGPolarity, ExtTRGFilter);

  /* Get the TIMx SMCR register value */
  tmpsmcr = TIMx->SMCR;
  /* Reset the SMS Bits */
  tmpsmcr &= (uint32_t)(~((uint32_t)TIM_SMCR_SMS_Msk));
  /* Select the External clock mode1 */
  tmpsmcr |= TIM_SlaveMode_External1;
  /* Select the Trigger selection : ETRF */
  tmpsmcr &= (uint32_t)(~((uint32_t)TIM_SMCR_TS_Msk));
  tmpsmcr |= TIM_TS_ETRF;
  /* Write to TIMx SMCR */
  TIMx->SMCR = tmpsmcr;
}

/**
  * @name   TIM_ETRClockMode2Config
  * @brief  EN: Configures the External clock Mode2.
  *         CN: ⲿʱģʽ2
  * @param  TIMx:
  *         EN: where x can be 1,2 or to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_ExtTRGPrescaler: 
  *         EN: The external Trigger Prescaler.
  *         CN: ⲿԤƵ
  *             @arg TIM_ExtTRGPSC_OFF
  *             @arg TIM_ExtTRGPSC_DIV2
  *             @arg TIM_ExtTRGPSC_DIV4
  *             @arg TIM_ExtTRGPSC_DIV8
  * @param  TIM_ExtTRGPolarity: 
  *         EN: The external Trigger Polarity.
  *         CN: ⲿԡ
  *             @arg TIM_ExtTRGPolarity_Inverted
  *             @arg TIM_ExtTRGPolarity_NonInverted
  * @param  ExtTRGFilter: 
  *         EN: External Trigger Filter.
  *         CN: ⲿ
  *         @range 0x00 - 0x0F.
  * @retval None
  */
void TIM_ETRClockMode2Config(TIM_TypeDef *TIMx, uint32_t TIM_ExtTRGPrescaler,
                             uint32_t TIM_ExtTRGPolarity, uint32_t ExtTRGFilter)
{
  /* Check the parameters */

  /* Configure the ETR Clock source */
  TIM_ETRConfig(TIMx, TIM_ExtTRGPrescaler, TIM_ExtTRGPolarity, ExtTRGFilter);
  /* Enable the External clock mode2 */
  TIMx->SMCR |= TIM_SMCR_ECE_Msk;
}


/**
  * @name   TIM_SelectInputTrigger
  * @brief  EN: Selects the Input Trigger source.
  *         CN: ѡ봥Դ
  * @param  TIMx:
  *         EN: where x can be 1, 2 or 15 to select the TIM peripheral.
  *         CN: x1,215ѡTIMΧ豸
  * @param  TIM_InputTriggerSource: 
  *         EN: The Input Trigger source.
  *         CN: 봥Դ
  *             @arg TIM_TS_ITR0
  *             @arg TIM_TS_ITR1
  *             @arg TIM_TS_ITR2
  *             @arg TIM_TS_ITR3
  *             @arg TIM_TS_TI1F_ED
  *             @arg TIM_TS_TI1FP1
  *             @arg TIM_TS_TI2FP2
  *             @arg TIM_TS_ETRF
  * @retval None
  */
void TIM_SelectInputTrigger(TIM_TypeDef *TIMx, uint32_t TIM_InputTriggerSource)
{
  uint32_t tmpsmcr = 0;

  /* Check the parameters */

  /* Get the TIMx SMCR register value */
  tmpsmcr = TIMx->SMCR;
  /* Reset the TS Bits */
  tmpsmcr &= (uint32_t)(~((uint32_t)TIM_SMCR_TS_Msk));
  /* Set the Input Trigger source */
  tmpsmcr |= TIM_InputTriggerSource; // 0060
  /* Write to TIMx SMCR */
  TIMx->SMCR = tmpsmcr;
}

/**
  * @name   TIM_SelectOutputTrigger
  * @brief  EN: Selects the TIMx Trigger Output Mode.
  *         CN: ѡTIMxģʽ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15,16or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_TRGOSource: 
  *         EN: specifies the Trigger Output source.
  *         CN: ָԴ
  *             @arg TIM_TRGOSource_Reset
  *             @arg TIM_TRGOSource_Enable
  *             @arg TIM_TRGOSource_Update
  *             @arg TIM_TRGOSource_OC1
  *             @arg TIM_TRGOSource_OC1Ref
  *             @arg TIM_TRGOSource_OC2Ref
  *             @arg TIM_TRGOSource_OC3Ref
  *             @arg TIM_TRGOSource_OC4Ref
  * @retval None
  */
void TIM_SelectOutputTrigger(TIM_TypeDef *TIMx, uint32_t TIM_TRGOSource)
{
  /* Check the parameters */

  /* Reset the MMS Bits */
  TIMx->CR2 &= (uint32_t) ~((uint32_t)TIM_CR2_MMS_Msk);
  /* Select the TRGO source */
  TIMx->CR2 |= TIM_TRGOSource;
}

/**
  * @name   TIM_SelectSlaveMode
  * @brief  EN: Selects the TIMx Slave Mode.
  *         CN: ѡTIMxģʽ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15,16or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_SlaveMode: 
  *         EN: specifies the Timer Slave Mode.
  *         CN: ָʱģʽ
  *             @arg TIM_SlaveMode_Reset
  *             @arg TIM_SlaveMode_Gated
  *             @arg TIM_SlaveMode_Trigger
  *             @arg TIM_SlaveMode_External1
  * @retval None
  */
void TIM_SelectSlaveMode(TIM_TypeDef *TIMx, uint32_t TIM_SlaveMode)
{
  /* Check the parameters */

  /* Reset the SMS Bits */
  TIMx->SMCR &= (uint32_t) ~((uint32_t)TIM_SMCR_SMS_Msk);
  /* Select the Slave Mode */
  TIMx->SMCR |= TIM_SlaveMode;
}

/**
  * @name   TIM_SelectMasterSlaveMode
  * @brief  EN: Sets or Resets the TIMx Master/Slave Mode.
  *         CN: ûTIMx/ģʽ
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15,16or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_MasterSlaveMode: 
  *         EN: specifies the Timer Master Slave Mode.
  *         CN: ָʱģʽ
  *             @arg TIM_MasterSlaveMode_Enable
  *             @arg TIM_MasterSlaveMode_Disable
  * @retval None
  */
void TIM_SelectMasterSlaveMode(TIM_TypeDef *TIMx, uint32_t TIM_MasterSlaveMode)
{
  /* Check the parameters */

  /* Reset the MSM Bit */
  TIMx->SMCR &= (uint32_t) ~((uint32_t)TIM_SMCR_MSM_Msk);

  /* Set or Reset the MSM Bit */
  TIMx->SMCR |= TIM_MasterSlaveMode;
}

/**
  * @name   TIM_ETRConfig
  * @brief  EN: Configures the TIMx External Trigger (ETR).
  *         CN: TIMxⲿETR
  * @param  TIMx:
  *         EN: where x can be 1, 2 ,15,16or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  TIM_ExtTRGPrescaler: 
  *         EN: The external Trigger Prescaler.
  *         CN: ⲿԤƵ
  *             @arg TIM_ExtTRGPSC_OFF
  *             @arg TIM_ExtTRGPSC_DIV2
  *             @arg TIM_ExtTRGPSC_DIV4
  *             @arg TIM_ExtTRGPSC_DIV8
  * @param  TIM_ExtTRGPolarity: 
  *         EN: The external Trigger Polarity.
  *         CN: ⲿԡ
  *             @arg TIM_ExtTRGPolarity_Inverted
  *             @arg TIM_ExtTRGPolarity_NonInverted
  * @param  ExtTRGFilter: 
  *         EN: External Trigger Filter.
  *         CN: ⲿ
  *         @range 0x00 - 0x0F.
  * @retval None
  */
void TIM_ETRConfig(TIM_TypeDef *TIMx, uint32_t TIM_ExtTRGPrescaler, uint32_t TIM_ExtTRGPolarity,
                   uint32_t ExtTRGFilter)
{
  uint32_t tmpsmcr = 0;

  /* Check the parameters */

  tmpsmcr = TIMx->SMCR;
  /* Reset the ETR Bits */
  tmpsmcr &= 0xFFFF00FF;
  /* Set the Prescaler, the Filter value and the Polarity */
  tmpsmcr |= (uint32_t)(TIM_ExtTRGPrescaler | (uint32_t)(TIM_ExtTRGPolarity | (uint32_t)(ExtTRGFilter << (uint32_t)8)));
  /* Write to TIMx SMCR */
  TIMx->SMCR = tmpsmcr;
}

/**
  * @name   TIM_EncoderInterfaceConfig
  * @brief  EN: Enables or disables the TIMx's Hall sensor interface.
  *         CN: ûTIMxĻӿڡ
  * @param  TIMx:
  *         EN: where x can be 2 to select the TIM peripheral.
  *         CN: x2ѡTIMΧ豸
  * @param  TIM_EncoderMode: 
  *         EN: specifies the TIMx Encoder Mode.
  *         CN: ָTIMxģʽ
  *             @arg TIM_EncoderMode_TI1
  *             @arg TIM_EncoderMode_TI2
  *             @arg TIM_EncoderMode_TI12
  * @param  TIM_IC1Polarity: 
  *         EN: specifies the IC1 Polarity.
  *         CN: ָIC1ԡ
  *             @arg TIM_ICPolarity_Falling
  *             @arg TIM_ICPolarity_Rising
  * @param  TIM_IC2Polarity: 
  *         EN: specifies the IC2 Polarity.
  *         CN: ָIC2ԡ
  *             @arg TIM_ICPolarity_Falling
  *             @arg TIM_ICPolarity_Rising
  * @retval None
  */
void TIM_EncoderInterfaceConfig(TIM_TypeDef *TIMx, uint32_t TIM_EncoderMode,
                                uint32_t TIM_IC1Polarity, uint32_t TIM_IC2Polarity)
{
  uint32_t tmpsmcr = 0;
  uint32_t tmpccmr1 = 0;
  uint32_t tmpccer = 0;

  /* Check the parameters */

  /* Get the TIMx SMCR register value */
  tmpsmcr = TIMx->SMCR;
  /* Get the TIMx CCMR1 register value */
  tmpccmr1 = TIMx->CCMR1;
  /* Get the TIMx CCER register value */
  tmpccer = TIMx->CCER;
  /* Set the encoder Mode */
  tmpsmcr &= (uint32_t)(~((uint32_t)TIM_SMCR_SMS_Msk));
  tmpsmcr |= TIM_EncoderMode;
  /* Select the Capture Compare 1 and the Capture Compare 2 as input */
  tmpccmr1 &= (uint32_t)(((uint32_t) ~((uint32_t)TIM_CCMR1_Input_CC1S_Msk)) & (uint32_t)(~((uint32_t)TIM_CCMR1_Input_CC2S_Msk)));
  tmpccmr1 |= TIM_CCMR1_CC1S_0 | TIM_CCMR1_CC2S_0;
  /* Set the TI1 and the TI2 Polarities */
  tmpccer &= (uint32_t) ~((uint32_t)(TIM_CCER_CC1P_Msk | TIM_CCER_CC1NP_Msk)) & (uint32_t) ~((uint32_t)(TIM_CCER_CC2P_Msk | TIM_CCER_CC2NP_Msk));
  tmpccer |= (uint32_t)(TIM_IC1Polarity | (uint32_t)(TIM_IC2Polarity << (uint32_t)4));
  /* Write to TIMx SMCR */
  TIMx->SMCR = tmpsmcr;
  /* Write to TIMx CCMR1 */
  TIMx->CCMR1 = tmpccmr1;
  /* Write to TIMx CCER */
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM_SelectHallSensor
  * @brief  EN: Enables or disables the TIMx's Hall sensor interface.
  *         CN: ûTIMxĻӿڡ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x2ѡTIMΧ豸
  * @param  TIM_ICPolarity: 
  *         EN: new state of the TIMx Hall sensor interface.
  *         CN: TIMxӿڵ״̬
  * @retval None
  */
void TIM_SelectHallSensor(TIM_TypeDef *TIMx, FunctionalState NewState)
{
  /* Check the parameters */

  if (NewState != DISABLE)
  {
    /* Set the TI1S Bit */
    TIMx->CR2 |= TIM_CR2_TI1S_Msk;
  }
  else
  {
    /* Reset the TI1S Bit */
    TIMx->CR2 &= (uint32_t) ~((uint32_t)TIM_CR2_TI1S_Msk);
  }
}

/**
  * @name   TI1_Config
  * @brief  EN: Configure the TI1 as Input.
  *         CN: TI1Ϊ롣
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16or 17 to select the TIM peripheral.
  *         CN: x12151617ѡTIMΧ豸
  * @param  TIM_ICPolarity: 
  *         EN: The Input Polarity.
  *         CN: 뼫ԡ
  *             @arg TIM_ICPolarity_Rising
  *             @arg TIM_ICPolarity_Falling
  * @param  TIM_ICSelection: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *             @arg TIM_ICSelection_DirectTI:
  *             @arg TIM_ICSelection_TRC:
  * @param  TIM_ICFilter: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *         @range 0x00 - 0x0F.
  * @retval None
  */
static void TI1_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                       uint32_t TIM_ICFilter)
{
  uint32_t tmpccmr1 = 0, tmpccer = 0;
  /* Disable the Channel 1: Reset the CC1E Bit */
  TIMx->CCER &= (uint32_t) ~((uint32_t)TIM_CCER_CC1E_Msk);
  tmpccmr1 = TIMx->CCMR1;
  tmpccer = TIMx->CCER;
  /* Select the Input and set the filter */
  tmpccmr1 &= (uint32_t)(((uint32_t) ~((uint32_t)TIM_CCMR1_Input_CC1S_Msk)) & ((uint32_t) ~((uint32_t)TIM_CCMR1_Input_IC1F_Msk)));
  tmpccmr1 |= (uint32_t)(TIM_ICSelection | (uint32_t)(TIM_ICFilter << (uint32_t)4));

  /* Select the Polarity and set the CC1E Bit */
  tmpccer &= (uint32_t) ~((uint32_t)(TIM_CCER_CC1P_Msk | TIM_CCER_CC1NP_Msk));
  tmpccer |= (uint32_t)(TIM_ICPolarity | (uint32_t)TIM_CCER_CC1E_Msk);
  /* Write to TIMx CCMR1 and CCER registers */
  TIMx->CCMR1 = tmpccmr1;
  TIMx->CCER = tmpccer;
}

/**
  * @name   TI2_Config
  * @brief  EN: Configure the TI2 as Input.
  *         CN: TI2Ϊ롣
  * @param  TIMx: 
  *         EN: where x can be 1, 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_ICPolarity: 
  *         EN: The Input Polarity.
  *         CN: 뼫ԡ
  *             @arg TIM_ICPolarity_Rising
  *             @arg TIM_ICPolarity_Falling
  * @param  TIM_ICSelection: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *             @arg TIM_ICSelection_DirectTI:
  *             @arg TIM_ICSelection_TRC:
  * @param  TIM_ICFilter: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *         @range 0x00 - 0x0F.
  * @retval None
  */
static void TI2_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                uint32_t TIM_ICFilter)
{
  uint32_t tmpccmr1 = 0, tmpccer = 0, tmp = 0;
  /* Disable the Channel 2: Reset the CC2E Bit */
  TIMx->CCER &= (uint32_t) ~((uint32_t)TIM_CCER_CC2E_Msk);
  tmpccmr1 = TIMx->CCMR1;
  tmpccer = TIMx->CCER;
  tmp = (uint32_t)(TIM_ICPolarity << 4);
  /* Select the Input and set the filter */
  tmpccmr1 &= (uint32_t)(((uint32_t) ~((uint32_t)TIM_CCMR1_Input_CC2S_Msk)) & ((uint32_t) ~((uint32_t)TIM_CCMR1_Input_IC2F_Msk)));
  tmpccmr1 |= (uint32_t)(TIM_ICFilter << 12);
  tmpccmr1 |= (uint32_t)(TIM_ICSelection << 8);
  /* Select the Polarity and set the CC2E Bit */
  tmpccer &= (uint32_t) ~((uint32_t)(TIM_CCER_CC2P_Msk | TIM_CCER_CC2NP_Msk));
  tmpccer |= (uint32_t)(tmp | (uint32_t)TIM_CCER_CC2E_Msk);
  /* Write to TIMx CCMR1 and CCER registers */
  TIMx->CCMR1 = tmpccmr1;
  TIMx->CCER = tmpccer;
}

/**
  * @name   TI3_Config
  * @brief  EN: Configure the TI3 as Input.
  *         CN: TI3Ϊ롣
  * @param  TIMx: 
  *         EN: where x can be 1, 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_ICPolarity: 
  *         EN: The Input Polarity.
  *         CN: 뼫ԡ
  *             @arg TIM_ICPolarity_Rising
  *             @arg TIM_ICPolarity_Falling
  * @param  TIM_ICSelection: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *             @arg TIM_ICSelection_DirectTI:
  *             @arg TIM_ICSelection_TRC:
  * @param  TIM_ICFilter: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *         @range 0x00 - 0x0F.
  * @retval None
  */
static void TI3_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                       uint32_t TIM_ICFilter)
{
  uint32_t tmpccmr2 = 0, tmpccer = 0, tmp = 0;
  /* Disable the Channel 3: Reset the CC3E Bit */
  TIMx->CCER &= (uint32_t) ~((uint32_t)TIM_CCER_CC3E_Msk);
  tmpccmr2 = TIMx->CCMR2;
  tmpccer = TIMx->CCER;
  tmp = (uint32_t)(TIM_ICPolarity << 8);
  /* Select the Input and set the filter */
  tmpccmr2 &= (uint32_t)(((uint32_t) ~((uint32_t)TIM_CCMR2_Input_CC3S_Msk)) & ((uint32_t) ~((uint32_t)TIM_CCMR2_Input_IC3F_Msk)));
  tmpccmr2 |= (uint32_t)(TIM_ICSelection | (uint32_t)(TIM_ICFilter << (uint32_t)4));
  /* Select the Polarity and set the CC3E Bit */
  tmpccer &= (uint32_t) ~((uint32_t)(TIM_CCER_CC3P_Msk | TIM_CCER_CC3NP_Msk));
  tmpccer |= (uint32_t)(tmp | (uint32_t)TIM_CCER_CC3E_Msk);
  /* Write to TIMx CCMR2 and CCER registers */
  TIMx->CCMR2 = tmpccmr2;
  TIMx->CCER = tmpccer;
}

/**
  * @name   TI4_Config
  * @brief  EN: Configure the TI4 as Input.
  *         CN: TI4Ϊ롣
  * @param  TIMx: 
  *         EN: where x can be 1, 2 to select the TIM peripheral.
  *         CN: x12ѡTIMΧ豸
  * @param  TIM_ICPolarity: 
  *         EN: The Input Polarity.
  *         CN: 뼫ԡ
  *             @arg TIM_ICPolarity_Rising
  *             @arg TIM_ICPolarity_Falling
  * @param  TIM_ICSelection: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *             @arg TIM_ICSelection_DirectTI:
  *             @arg TIM_ICSelection_TRC:
  * @param  TIM_ICFilter: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *         @range 0x00 - 0x0F.
  * @retval None
  */
static void TI4_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                       uint32_t TIM_ICFilter)
{
  uint32_t tmpccmr2 = 0, tmpccer = 0, tmp = 0;

  /* Disable the Channel 4: Reset the CC4E Bit */
  TIMx->CCER &= (uint32_t) ~((uint32_t)TIM_CCER_CC4E_Msk);
  tmpccmr2 = TIMx->CCMR2;
  tmpccer = TIMx->CCER;
  tmp = (uint32_t)(TIM_ICPolarity << 12);
  /* Select the Input and set the filter */
  tmpccmr2 &= (uint32_t)((uint32_t)(~(uint32_t)TIM_CCMR2_Input_CC4S_Msk) & ((uint32_t) ~((uint32_t)TIM_CCMR2_Input_IC4F_Msk)));
  tmpccmr2 |= (uint32_t)(TIM_ICSelection << 8);
  tmpccmr2 |= (uint32_t)(TIM_ICFilter << 12);
  /* Select the Polarity and set the CC4E Bit */
  tmpccer &= (uint32_t) ~((uint32_t)(TIM_CCER_CC4P_Msk | TIM_CCER_CC4NP_Msk));
  tmpccer |= (uint32_t)(tmp | (uint32_t)TIM_CCER_CC4E_Msk);
  /* Write to TIMx CCMR2 and CCER registers */
  TIMx->CCMR2 = tmpccmr2;
  TIMx->CCER = tmpccer;
}

/**
  * @name   TI5_Config
  * @brief  EN: Configure the TI5 as Input.
  *         CN: TI5Ϊ롣
  * @param  TIMx: 
  *         EN: where x can be 1to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  TIM_ICPolarity: 
  *         EN: The Input Polarity.
  *         CN: 뼫ԡ
  *             @arg TIM_ICPolarity_Rising
  *             @arg TIM_ICPolarity_Falling
  * @param  TIM_ICSelection: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *             @arg TIM_ICSelection_DirectTI:
  *             @arg TIM_ICSelection_TRC:
  * @param  TIM_ICFilter: 
  *         EN: specifies the input to be used.
  *         CN: ָҪʹõ롣
  *         @range 0x00 - 0x0F.
  * @retval None
  */
static void TI5_Config(TIM_TypeDef *TIMx, uint32_t TIM_ICPolarity, uint32_t TIM_ICSelection,
                       uint32_t TIM_ICFilter)
{
  uint32_t tmpccmr2 = 0, tmpccer = 0, tmp = 0;

  /* Disable the Channel 4: Reset the CC5E Bit */
  TIMx->CCER &= (uint32_t) ~((uint32_t)TIM_CCER_CC5E_Msk);
  tmpccmr2 = TIMx->CCMR3;
  tmpccer = TIMx->CCER;
  tmp = (uint32_t)(TIM_ICPolarity << 16);
  /* Select the Input and set the filter */
  tmpccmr2 &= (uint32_t)((uint32_t)(~(uint32_t)TIM_CCMR3_Input_CC5S_Msk) & ((uint32_t) ~((uint32_t)TIM_CCMR3_Input_IC5F_Msk)));
  tmpccmr2 |= (uint32_t)(TIM_ICSelection) | (uint32_t)(TIM_ICFilter);
  /* Select the Polarity and set the CC5E Bit */
  tmpccer &= (uint32_t) ~((uint32_t)(TIM_CCER_CC5P_Msk | TIM_CCER_CC5NP_Msk));
  tmpccer |= (uint32_t)(tmp | (uint32_t)TIM_CCER_CC5E_Msk);
  /* Write to TIMx CCMR3 and CCER registers */
  TIMx->CCMR3 = tmpccmr2;
  TIMx->CCER = tmpccer;
}

/**
  * @name   TIM1_IntputSourceCfg
  * @brief  EN: TIM1 external trigger input source configuration TIx ETRx.
  *         CN: TIM1ⲿԴá
  * @param  TIx: 
  *         EN: External trigger input.
  *         CN: ⲿTIx ETRx
  * @param  TIM_IntputSource: 
  *         EN: TIx ETRx Input trigger source selection.
  *         CN: TIx ETRx 봥Դѡ
  * @retval None
  */
void TIM1_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource)
{
  assert_param(IS_TIM1_INPUT_TI(TIx));
  assert_param(IS_TIM1_IntputSource(TIM_IntputSource));
  sysctrl_access();
  SYSCTRL->EDU_CFG1 &= (uint32_t) ~(0xF<< (TIx * 4));
  SYSCTRL->EDU_CFG1 |= TIM_IntputSource << (TIx * 4); // TIM1 TIx ETR 봥Դѡ
  __dekey();
}

/**
  * @name   TIM2_IntputSourceCfg
  * @brief  EN: TIM2 external trigger input source configuration TIx ETRx.
  *         CN: TIM2ⲿԴá
  * @param  TIx: 
  *         EN: External trigger input.
  *         CN: ⲿTIx ETRx
  * @param  TIM_IntputSource: 
  *         EN: TIx ETRx Input trigger source selection.
  *         CN: TIx ETRx 봥Դѡ
  * @retval None
  */
void TIM2_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource)
{
  assert_param(IS_TIM2_INPUT_TI(TIx));
  assert_param(IS_TIM2_IntputSource(TIM_IntputSource));
  sysctrl_access();
  SYSCTRL->EDU_CFG8 &= (uint32_t) ~(0xF<< (TIx * 4));
  SYSCTRL->EDU_CFG8 |= TIM_IntputSource << (TIx * 4); // TIM2 TIx ETR 봥Դѡ
  __dekey();
}

/**
  * @name   TIM15_IntputSourceCfg
  * @brief  EN: TIM15 external trigger input source configuration TIx ETRx.
  *         CN: TIM15ⲿԴá
  * @param  TIx: 
  *         EN: External trigger input.
  *         CN: ⲿTIx ETRx
  * @param  TIM_IntputSource: 
  *         EN: TIx ETRx Input trigger source selection.
  *         CN: TIx ETRx 봥Դѡ
  * @retval None
  */
void TIM15_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource)
{
  assert_param(IS_TIM15_INPUT_TI(TIx));
  assert_param(IS_TIM15_IntputSource(TIM_IntputSource));
  sysctrl_access();
  SYSCTRL->EDU_CFG3 &= (uint32_t) ~(0xF<< (TIx * 4));
  SYSCTRL->EDU_CFG3 |= TIM_IntputSource << (TIx * 4); // TIM15 TIx ETR 봥Դѡ
  __dekey();
}

/**
  * @name   TIM16_IntputSourceCfg
  * @brief  EN: TIM16 external trigger input source configuration TIx ETRx.
  *         CN: TIM16ⲿԴá
  * @param  TIx: 
  *         EN: External trigger input.
  *         CN: ⲿTIx ETRx
  * @param  TIM_IntputSource: 
  *         EN: TIx ETRx Input trigger source selection.
  *         CN: TIx ETRx 봥Դѡ
  * @retval None
  */
void TIM16_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource)
{
  assert_param(IS_TIM16_INPUT_TI(TIx));
  assert_param(IS_TIM16_IntputSource(TIM_IntputSource));
  sysctrl_access();
  SYSCTRL->EDU_CFG3 &= (uint32_t) ~(0xF<< (TIx * 4));
  SYSCTRL->EDU_CFG3 |= TIM_IntputSource << ((TIx * 4) + 12); // TIM16 TIx ETR 봥Դѡ
  __dekey();
}

/**
  * @name   TIM17_IntputSourceCfg
  * @brief  EN: TIM17 external trigger input source configuration TIx ETRx.
  *         CN: TIM17ⲿԴá
  * @param  TIx: 
  *         EN: External trigger input.
  *         CN: ⲿTIx ETRx
  * @param  TIM_IntputSource: 
  *         EN: TIx ETRx Input trigger source selection.
  *         CN: TIx ETRx 봥Դѡ
  * @retval None
  */
void TIM17_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource)
{
  assert_param(IS_TIM17_INPUT_TI(TIx));
  assert_param(IS_TIM17_IntputSource(TIM_IntputSource));
  sysctrl_access();
  SYSCTRL->EDU_CFG3 &= (uint32_t) ~(0xF<< (TIx * 4));
  SYSCTRL->EDU_CFG3 |= TIM_IntputSource << ((TIx * 4) + 24); // TIM17 TIx ETR 봥Դѡ
  __dekey();
}

/**
  * @name   TIM1_BreakSourceCfg
  * @brief  EN: TIM1 brake trigger input source configuration.
  *         CN: TIM1 ɲԴá
  * @param  TIM_BreakSource: 
  *         EN: Brake input trigger source selection.
  *         CN: ɲ봥Դѡ
  * @param  TIM_Level: 
  *         EN: Effective level selection of TIM1 brake input trigger source.
  *         CN: TIM1 ɲ봥ԴЧƽѡ
  * @param  NewState: 
  *         EN: New status of brake input trigger source.
  *         CN: ɲ봥Դ״̬
  * @retval None
  */
void TIM1_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState)
{
  uint32_t temprg;
  assert_param(IS_TIM1_BreakSource(TIM_BreakSource));
  assert_param(IS_TIM_InputLevel(TIM_Level));

  if (NewState != DISABLE)
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG4 |= TIM_BreakSource; // TIM1 break 봥Դ
    __dekey();
  }
  else
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG4 &= (~TIM_BreakSource); // TIM1 break 봥Դ
    __dekey();
  }
   
  temprg = SYSCTRL->EDU_CFG4;
  if (TIM_Level == TIM_InputLevel_High)       // TIM1 break ԴЧƽΪ
  {
    if (temprg & 0x00007F00)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG4 &= (~(TIM_BreakSource >> 8));
      __dekey();
    }
    else if (temprg & 0x003F0000)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG4 &= (~(TIM_BreakSource >> 16));
      __dekey();
    }
  }
  else if (TIM_Level == TIM_InputLevel_Low)  // TIM1 break ԴЧƽΪ
  {
    if (temprg & 0x00007F00)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG4 |= (TIM_BreakSource >> 8);
      __dekey();
    }
    else if (temprg & 0x003F0000)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG4 |= (TIM_BreakSource >> 16);
      __dekey();
    }
  }
}

/**
  * @name   TIM1_OCREF_CLR_SourceCfg
  * @brief  EN: TIM1  OCREF_CLR trigger input source configuration.
  *         CN: TIM1 OCREF_CLRԴá
  * @param  TIM_OCREF_CLR_Source: 
  *         EN:  OCREF_CLR input trigger source selection.
  *         CN:  OCREF_CLR봥Դѡ
  * @param  TIM_Level: 
  *         EN: Effective level selection of TIM1  OCREF_CLR input trigger source.
  *         CN: TIM1  OCREF_CLR봥ԴЧƽѡ
  * @param  NewState: 
  *         EN: New status of  OCREF_CLR input trigger source.
  *         CN: OCREF_CLR봥Դ״̬
  * @retval None
  */
void TIM1_OCREF_CLR_SourceCfg(uint32_t TIM_OCREF_CLR_Source, uint32_t TIM_Level, FunctionalState NewState)
{
  assert_param(IS_TIM1_OCREF_CLR_Source(TIM_OCREF_CLR_Source));
  assert_param(IS_TIM_InputLevel(TIM_Level));
  
  if (NewState != DISABLE)
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG4 |= TIM_OCREF_CLR_Source; // TIM1 OCREF_CLR 봥Դ
    __dekey();
  }
  else
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG4 &= (~TIM_OCREF_CLR_Source); // TIM OCREF_CLR 봥Դ
    __dekey();
  }

  if (TIM_Level == TIM_InputLevel_High)       // TIM1 OCREF_CLR ԴЧƽΪ
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG4 &= (~(TIM_OCREF_CLR_Source >> 24));
    __dekey();
  }
  else if (TIM_Level == TIM_InputLevel_Low)  // TIM1 OCREF_CLR ԴЧƽΪ
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG4 |= (TIM_OCREF_CLR_Source >> 24);
    __dekey();
  }
}

/**
  * @name   TIM17_BreakSourceCfg
  * @brief  EN: TIM17 brake trigger input source configuration.
  *         CN: TIM17 ɲԴá
  * @param  TIM_BreakSource: 
  *         EN: Brake input trigger source selection.
  *         CN: ɲ봥Դѡ
  * @param  TIM_Level: 
  *         EN: Effective level selection of TIM17 brake input trigger source.
  *         CN: TIM17 ɲ봥ԴЧƽѡ
  * @param  NewState: 
  *         EN: New status of brake input trigger source.
  *         CN: ɲ봥Դ״̬
  * @retval None
  */
void TIM17_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState)
{
  assert_param(IS_TIM17_BreakSource(TIM_BreakSource));
  assert_param(IS_TIM_InputLevel(TIM_Level));

  if (NewState != DISABLE)
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG5 |= TIM_BreakSource; // TIM17 break 봥Դ
    __dekey();
  }
  else
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG5 &= (~TIM_BreakSource); // TIM17 break 봥Դ
    __dekey();
  }

  if (TIM_Level == TIM_InputLevel_High)       // TIM17 break ԴЧƽΪ
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG5 &= (~(TIM_BreakSource >> 8));
    __dekey();
  }
  else if (TIM_Level == TIM_InputLevel_Low)  // TIM17 break ԴЧƽΪ
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG5 |= (TIM_BreakSource >> 8);
    __dekey();
  }
}

/**
  * @name   TIM15_BreakSourceCfg
  * @brief  EN: TIM15 brake trigger input source configuration.
  *         CN: TIM15 ɲԴá
  * @param  TIM_BreakSource: 
  *         EN: Brake input trigger source selection.
  *         CN: ɲ봥Դѡ
  * @param  TIM_Level: 
  *         EN: Effective level selection of TIM15 brake input trigger source.
  *         CN: TIM15 ɲ봥ԴЧƽѡ
  * @param  NewState: 
  *         EN: New status of brake input trigger source.
  *         CN: ɲ봥Դ״̬
  * @retval None
  */
void TIM15_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState)
{
  uint32_t temprg;
  assert_param(IS_TIM15_BreakSource(TIM_BreakSource));
  assert_param(IS_TIM_InputLevel(TIM_Level));

  if (NewState != DISABLE)
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG6 |= TIM_BreakSource; // TIM15 break/break2 봥Դ
    __dekey();
  }
  else
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG6 &= (~TIM_BreakSource); // TIM15 break/break2 봥Դ
    __dekey();
  }
  
  temprg = SYSCTRL->EDU_CFG6;
  if (TIM_Level == TIM_InputLevel_High)       // TIM15 break ԴЧƽΪ
  {
    if (temprg & 0x00007F00)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG6 &= (~(TIM_BreakSource >> 8));
      __dekey();
    }
    else if (temprg & 0x003F0000)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG6 &= (~(TIM_BreakSource >> 16));
      __dekey();
    }
  }
  else if (TIM_Level == TIM_InputLevel_Low)  // TIM15 break ԴЧƽΪ
  {
    if (temprg & 0x00007F00)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG6 |= (TIM_BreakSource >> 8);
      __dekey();
    }
    else if (temprg & 0x003F0000)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG6 |= (TIM_BreakSource >> 16);
      __dekey();
    }
  }
}

/**
  * @name   TIM16_BreakSourceCfg
  * @brief  EN: TIM16 brake trigger input source configuration.
  *         CN: TIM16 ɲԴá
  * @param  TIM_BreakSource: 
  *         EN: Brake input trigger source selection.
  *         CN: ɲ봥Դѡ
  * @param  TIM_Level: 
  *         EN: Effective level selection of TIM15 brake input trigger source.
  *         CN: TIM16 ɲ봥ԴЧƽѡ
  * @param  NewState: 
  *         EN: New status of brake input trigger source.
  *         CN: ɲ봥Դ״̬
  * @retval None
  */
void TIM16_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState)
{
  uint32_t temprg;
  assert_param(IS_TIM16_BreakSource(TIM_BreakSource));
  assert_param(IS_TIM_InputLevel(TIM_Level));

  if (NewState != DISABLE)
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG7 |= TIM_BreakSource; // TIM16 break/break2 봥Դ
    __dekey();
  }
  else
  {
      sysctrl_access();
      SYSCTRL->EDU_CFG7 &= (~TIM_BreakSource); // TIM16 break/break2 봥Դ
      __dekey();
  }

  temprg = SYSCTRL->EDU_CFG7;
  if (TIM_Level == TIM_InputLevel_High)       // TIM16 break ԴЧƽΪ
  {
    if (temprg & 0x00007F00)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG7 &= (~(TIM_BreakSource >> 8));
      __dekey();
    }
    else if (temprg & 0x003F0000)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG7 &= (~(TIM_BreakSource >> 16));
      __dekey();
    }
  }
  else if (TIM_Level == TIM_InputLevel_Low)  // TIM16 break ԴЧƽΪ
  {
    if (temprg & 0x00007F00)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG7 |= (TIM_BreakSource >> 8);
      __dekey();
    }
    else if (temprg & 0x003F0000)
    {
      sysctrl_access();
      SYSCTRL->EDU_CFG7 |= (TIM_BreakSource >> 16);
      __dekey();
    }
  }
}


/**
  * @name   TIM2_BreakSourceCfg
  * @brief  EN: TIM2 brake trigger input source configuration.
  *         CN: TIM2 ɲԴá
  * @param  TIM_BreakSource: 
  *         EN: Brake input trigger source selection.
  *         CN: ɲ봥Դѡ
  * @param  TIM_Level: 
  *         EN: Effective level selection of TIM1 brake input trigger source.
  *         CN: TIM2 ɲ봥ԴЧƽѡ
  * @param  NewState: 
  *         EN: New status of brake input trigger source.
  *         CN: ɲ봥Դ״̬
  * @retval None
  */
void TIM2_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState)
{
  assert_param(IS_TIM2_BreakSource(TIM_BreakSource));
  assert_param(IS_TIM_InputLevel(TIM_Level));

  if (NewState != DISABLE)
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG9 |= TIM_BreakSource; // TIM2 break 봥Դ
    __dekey();
  }
  else
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG9 &= (~TIM_BreakSource); // TIM2 break 봥Դ
    __dekey();
  }

  if (TIM_Level == TIM_InputLevel_High)       // TIM2 break ԴЧƽΪ
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG9 &= (~(TIM_BreakSource >> 8));
    __dekey();
  }
  else if (TIM_Level == TIM_InputLevel_Low)  // TIM2 break ԴЧƽΪ
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG9 |= (TIM_BreakSource >> 8);
    __dekey();
  }
}

/**
  * @name   TIM2_OCREF_CLR_SourceCfg
  * @brief  EN: TIM2 OCREF_CLR trigger input source configuration.
  *         CN: TIM2 OCREF_CLRԴá
  * @param  TIM_BreakSource: 
  *         EN: OCREF_CLR input trigger source selection.
  *         CN: OCREF_CLR봥Դѡ
  * @param  TIM_Level: 
  *         EN: Effective level selection of TIM2 OCREF_CLR input trigger source.
  *         CN: TIM2 OCREF_CLR봥ԴЧƽѡ
  * @param  NewState: 
  *         EN: New status of  OCREF_CLR input trigger source.
  *         CN: ɲ봥Դ״̬
  * @retval None
  */
void TIM2_OCREF_CLR_SourceCfg(uint32_t TIM_OCREF_CLR_Source, uint32_t TIM_Level, FunctionalState NewState)
{
  assert_param(IS_TIM2_OCREF_CLR_Source(TIM_OCREF_CLR_Source));
  assert_param(IS_TIM_InputLevel(TIM_Level));
  
  if (NewState != DISABLE)
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG9 |= TIM_OCREF_CLR_Source; // TIM2 OCREF_CLR 봥Դ
    __dekey();
  }
  else
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG9 &= (~TIM_OCREF_CLR_Source); // TIM OCREF_CLR 봥Դ
    __dekey();
  }

  if (TIM_Level == TIM_InputLevel_High)       // TIM2 OCREF_CLR ԴЧƽΪ
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG9 &= (~(TIM_OCREF_CLR_Source >> 24));
    __dekey();
  }
  else if (TIM_Level == TIM_InputLevel_Low)  // TIM1 OCREF_CLR ԴЧƽΪ
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG9 |= (TIM_OCREF_CLR_Source >> 24);
    __dekey();
  }
}

/**
  * @name   TIM1_PWMxSel_Config
  * @brief  EN: TIM1 PWM x output mapping config.
  *         CN: TIM1 PWM x ӳá
  * @param  PWM_Num: 
  *         EN: Select PWM X.
  *         CN: ѡPWM X
  *             @arg TIM1_PWM1
  *             @arg TIM1_PWM2
  *             @arg TIM1_PWM3
  *             @arg TIM1_PWM4
  *             @arg TIM1_PWM5
  *             @arg TIM1_PWM6
  *             @arg TIM1_PWM7
  *             @arg TIM1_PWM8
  * @param  OCx_Sel: 
  *         EN: Select OCx signal output from PWM port.
  *         CN: ѡPWM˿OCxźš
  *             @arg TIM1_PWMx_OC1
  *             @arg TIM1_PWMx_OC2
  *             @arg TIM1_PWMx_OC3
  *             @arg TIM1_PWMx_OC4
  *             @arg TIM1_PWMx_OC1N
  *             @arg TIM1_PWMx_OC2N
  *             @arg TIM1_PWMx_OC3N
  *             @arg TIM1_PWMx_OC4N
  * @retval None
  */
void TIM1_PWMxSel_Config (uint32_t PWM_Num, uint32_t OCx_Sel)
{
  TIM1->OMR &= ~(0xF << (PWM_Num * 4));
  TIM1->OMR |= OCx_Sel << (PWM_Num * 4);
}

/**
  * @name   TIM1_CompareOffsetMode
  * @brief  EN: TIM1 Compare offset modes.
  *         CN: TIM1 Ƚƫģʽ
  * @param  TIM_Channel: 
  *         EN: specifies the TIM Channel
  *         CN: ָTIMͨ
  *             @arg TIM_Channel_1
  *             @arg TIM_Channel_2
  *             @arg TIM_Channel_3
  *             @arg TIM_Channel_4
  *             @arg TIM_Channel_5
  * @param  OCx_Mode: 
  *         EN: Select OCx signal output from PWM port.
  *         CN: ѡͨXƫģʽ
  *             @arg TIM1_OCx_Mode_UP_CCR1
  *             @arg TIM1_OCx_Mode_DOWN_CCR1
  *             @arg TIM1_OCx_Mode_UP_CCOR1
  *             @arg TIM1_OCx_Mode_DOWN_CCOR1
  * @retval None
  */
void TIM1_CompareOffsetMode (uint32_t TIM_Channel, uint32_t OCx_Mode)
{
  TIM1->OCR &= ~(0xF << TIM_Channel);
  TIM1->OCR |= OCx_Mode << TIM_Channel;
}

/**
  * @name   TIM1_SetCompareOffset1
  * @brief  EN: Sets the TIM1 Capture Compare Offset 1 Register value.
  *         CN: TIM1 Capture Compare Offset 1Ĵֵ
  * @param  CompareOffset: 
  *         EN: specifies the Capture Compare Offset 1 register new value.
  *         CN: ָCapture Compare Offset 1 Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM1_SetCompareOffset1(uint32_t CompareOffset)
{
  /* Set the Capture Compare Offset 1 Register value */
  TIM1->CCOR1 = CompareOffset;
}

/**
  * @name   TIM1_SetCompareOffset2
  * @brief  EN: Sets the TIM1 Capture Compare Offset 2 Register value.
  *         CN: TIM1 Capture Compare Offset 2 Ĵֵ
  * @param  CompareOffset: 
  *         EN: specifies the Capture Compare Offset 2 register new value.
  *         CN: ָCapture Compare Offset 2 Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM1_SetCompareOffset2(uint32_t CompareOffset)
{
  /* Set the Capture Compare Offset 2 Register value */
  TIM1->CCOR2 = CompareOffset;
}

/**
  * @name   TIM1_SetCompareOffset3
  * @brief  EN: Sets the TIM1 Capture Compare Offset 3 Register value.
  *         CN: TIM1 Capture Compare Offset 3 Ĵֵ
  * @param  CompareOffset: 
  *         EN: specifies the Capture Compare Offset 3 register new value.
  *         CN: ָCapture Compare Offset 3 Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM1_SetCompareOffset3(uint32_t CompareOffset)
{
  /* Set the Capture Compare Offset 3 Register value */
  TIM1->CCOR3 = CompareOffset;
}

/**
  * @name   TIM1_SetCompareOffset4
  * @brief  EN: Sets the TIM1 Capture Compare Offset 4 Register value.
  *         CN: TIM1 Capture Compare Offset 4 Ĵֵ
  * @param  CompareOffset: 
  *         EN: specifies the Capture Compare Offset 4 register new value.
  *         CN: ָCapture Compare Offset 4 Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM1_SetCompareOffset4(uint32_t CompareOffset)
{
  /* Set the Capture Compare Offset 4 Register value */
  TIM1->CCOR4 = CompareOffset;
}

/**
  * @name   TIM1_SetCompareOffset5
  * @brief  EN: Sets the TIM1 Capture Compare Offset 5 Register value.
  *         CN: TIM1 Capture Compare Offset 5 Ĵֵ
  * @param  Compare1: 
  *         EN: specifies the Capture Compare Offset 5 register new value.
  *         CN: ָCapture Compare Offset 5 Ĵֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM1_SetCompareOffset5(uint32_t CompareOffset)
{
  /* Set the Capture Compare Offset 5 Register value */
  TIM1->CCOR5 = CompareOffset;
}

/**
  * @name   TIM_GetCaptureOffset1
  * @brief  EN: Gets the TIMx Capture Offset Register 1 value.
  *         CN: ȡTIMx ȽƫƼĴ 1 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @retval EN: Capture Compare Offset 1 Register value.
  *         CN: Ƚƫ1Ĵֵ
  */
uint32_t TIM_GetCaptureOffset1(TIM_TypeDef *TIMx)
{
  return TIMx->CCOR1;
}

/**
  * @name   TIM_GetCaptureOffset2
  * @brief  EN: Gets the TIMx Capture Offset Register 2 value.
  *         CN: ȡTIMx ȽƫƼĴ 2 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @retval EN: Capture Compare Offset 2 Register value.
  *         CN: Ƚƫ2Ĵֵ
  */
uint32_t TIM_GetCaptureOffset2(TIM_TypeDef *TIMx)
{
  return TIMx->CCOR2;
}

/**
  * @name   TIM_GetCaptureOffset3
  * @brief  EN: Gets the TIMx Capture Offset Register 3 value.
  *         CN: ȡTIMx ȽƫƼĴ 3 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @retval EN: Capture Compare Offset 3 Register value.
  *         CN: Ƚƫ3Ĵֵ
  */
uint32_t TIM_GetCaptureOffset3(TIM_TypeDef *TIMx)
{
  return TIMx->CCOR3;
}

/**
  * @name   TIM_GetCaptureOffset4
  * @brief  EN: Gets the TIMx Capture Offset Register 4 value.
  *         CN: ȡTIMx ȽƫƼĴ 4 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @retval EN: Capture Compare Offset 4 Register value.
  *         CN: Ƚƫ4Ĵֵ
  */
uint32_t TIM_GetCaptureOffset4(TIM_TypeDef *TIMx)
{
  return TIMx->CCOR4;
}

/**
  * @name   TIM_GetCaptureOffset5
  * @brief  EN: Gets the TIMx Capture Offset Register 5 value.
  *         CN: ȡTIMx ȽƫƼĴ 5 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @retval EN: Capture Compare Offset 5 Register value.
  *         CN: Ƚƫ5Ĵֵ
  */
uint32_t TIM_GetCaptureOffset5(TIM_TypeDef *TIMx)
{
  return TIMx->CCOR5;
}

/**
  * @name   TIM_SetCompare1
  * @brief  EN: Sets the TIMx Capture tuning 1 Register value.
  *         CN: TIMx Ƚ޵Ĵ 1 ֵ
  * @param  TIMx:
  *         EN: where x can be 1, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,15,1617ѡTIMΧ豸
  * @param  Tuning: 
  *         EN: specifies the Capture tuning 1 register new value.
  *         CN: ָȽ޵Ĵ1ֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompareTuning1(TIM_TypeDef *TIMx, uint32_t Tuning)
{
  TIMx->CCTR1 = Tuning;
}

/**
  * @name   TIM_SetCompareTuning2
  * @brief  EN: Sets the TIMx Capture tuning 2 Register value.
  *         CN: TIMx Ƚ޵Ĵ 2 ֵ
  * @param  TIMx:
  *         EN: where x can be 1, 15, or 16 to select the TIM peripheral.
  *         CN: x1,1516ѡTIMΧ豸
  * @param  Tuning: 
  *         EN: specifies the Capture tuning 2 register new value.
  *         CN: ָȽ޵Ĵ2ֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompareTuning2(TIM_TypeDef *TIMx, uint32_t Tuning)
{
  TIMx->CCTR2 = Tuning;
}

/**
  * @name   TIM_SetCompareTuning3
  * @brief  EN: Sets the TIMx Capture tuning 3 Register value.
  *         CN: TIMx Ƚ޵Ĵ 3 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  Tuning: 
  *         EN: specifies the Capture tuning 3 register new value.
  *         CN: ָȽ޵Ĵ3ֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompareTuning3(TIM_TypeDef *TIMx, uint32_t Tuning)
{
  TIMx->CCTR3 = Tuning;
}

/**
  * @name   TIM_SetCompareTuning4
  * @brief  EN: Sets the TIMx Capture tuning 4 Register value.
  *         CN: TIMx Ƚ޵Ĵ 4 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  Tuning: 
  *         EN: specifies the Capture tuning 4 register new value.
  *         CN: ָȽ޵Ĵ4ֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompareTuning4(TIM_TypeDef *TIMx, uint32_t Tuning)
{
  TIMx->CCTR4 = Tuning;
}

/**
  * @name   TIM_SetCompareTuning5
  * @brief  EN: Sets the TIMx Capture tuning 5 Register value.
  *         CN: TIMx Ƚ޵Ĵ 5 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @param  Tuning: 
  *         EN: specifies the Capture tuning 5 register new value.
  *         CN: ָȽ޵Ĵ5ֵ
  *         @range 0x0 - 0xFFFF
  * @retval None
  */
void TIM_SetCompareTuning5(TIM_TypeDef *TIMx, uint32_t Tuning)
{
  TIMx->CCTR5 = Tuning;
}

/**
  * @name   TIM_GetCompareTuning1
  * @brief  EN: Gets the TIMx Capture tuning Register 1 value.
  *         CN: ȡTIMx Ƚ޵Ĵ 1 ֵ
  * @param  TIMx:
  *         EN: where x can be 1, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,15,1617ѡTIMΧ豸
  * @retval EN: Compare tuning 1 Register value.
  *         CN: Ƚ޵1Ĵֵ
  */
uint32_t TIM_GetCompareTuning1(TIM_TypeDef *TIMx)
{
  return TIMx->CCTR1;
}

/**
  * @name   TIM_GetCompareTuning2
  * @brief  EN: Gets the TIMx Capture tuning Register 2 value.
  *         CN: ȡTIMx Ƚ޵Ĵ 2 ֵ
  * @param  TIMx:
  *         EN: where x can be 1, 15 or 16 to select the TIM peripheral.
  *         CN: x1,1516ѡTIMΧ豸
  * @retval EN: Compare tuning 2 Register value.
  *         CN: Ƚ޵2Ĵֵ
  */
uint32_t TIM_GetCompareTuning2(TIM_TypeDef *TIMx)
{
  return TIMx->CCTR2;
}

/**
  * @name   TIM_GetCompareTuning3
  * @brief  EN: Gets the TIMx Capture tuning Register 3 value.
  *         CN: ȡTIMx Ƚ޵Ĵ 3 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @retval EN: Compare tuning 3 Register value.
  *         CN: Ƚ޵3Ĵֵ
  */
uint32_t TIM_GetCompareTuning3(TIM_TypeDef *TIMx)
{
  return TIMx->CCTR3;
}

/**
  * @name   TIM_GetCompareTuning4
  * @brief  EN: Gets the TIMx Capture tuning Register 4 value.
  *         CN: ȡTIMx Ƚ޵Ĵ 4 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @retval EN: Compare tuning 4 Register value.
  *         CN: Ƚ޵4Ĵֵ
  */
uint32_t TIM_GetCompareTuning4(TIM_TypeDef *TIMx)
{
  return TIMx->CCTR4;
}

/**
  * @name   TIM_GetCompareTuning5
  * @brief  EN: Gets the TIMx Capture tuning Register 5 value.
  *         CN: ȡTIMx Ƚ޵Ĵ 5 ֵ
  * @param  TIMx:
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1ѡTIMΧ豸
  * @retval EN: Compare tuning 5 Register value.
  *         CN: Ƚ޵5Ĵֵ
  */
uint32_t TIM_GetCompareTuning5(TIM_TypeDef *TIMx)
{
  return TIMx->CCTR5;
}

/**
  * @name   TIM_BreakTrigOci_Cmd
  * @brief  EN: Configure TIMx_Trig_oci signal trigger source.
  *         CN: TIMx_Trig_ociźŴԴ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx_Trig_oci signal trigger source.
  *         CN:  TIMx_Trig_ociźŴԴ״̬
  * @retval None
  */
void TIM_BreakTrigOci_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
  if (NewState != DISABLE)
  {
    TIMx->BDTR |= TIM_BDTR_BKTRIG_Msk;
  }
  else
  {
    TIMx->BDTR &= (uint32_t)(~((uint32_t)TIM_BDTR_BKTRIG_Msk));
  }
}

/**
  * @name   TIM_OCTuning1_Cmd
  * @brief  EN: Output comparison 1 Tuning enable.
  *         CN: Ƚ1޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 1 Tuning.
  *         CN: TIMx Ƚ1޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning1_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR1 |= TIM_CCMR1_Output_OC1TE_Msk;
    }
    else
    {
      TIMx->CCMR1 &= (uint32_t)(~((uint32_t)TIM_CCMR1_Output_OC1TE_Msk));
    }
}

/**
  * @name   TIM_OCTuning2_Cmd
  * @brief  EN: Output comparison 2 Tuning enable.
  *         CN: Ƚ2޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 to select the TIM peripheral.
  *         CN: x1,2,15,16ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 2 Tuning.
  *         CN: TIMx Ƚ2޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning2_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR1 |= TIM_CCMR1_Output_OC2TE_Msk;
    }
    else
    {
      TIMx->CCMR1 &= (uint32_t)(~((uint32_t)TIM_CCMR1_Output_OC2TE_Msk));
    }
}

/**
  * @name   TIM_OCTuning3_Cmd
  * @brief  EN: Output comparison 3 Tuning enable.
  *         CN: Ƚ3޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2 to select the TIM peripheral.
  *         CN: x1,2ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 3 Tuning.
  *         CN: TIMx Ƚ3޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning3_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR2 |= TIM_CCMR2_Output_OC3TE_Msk;
    }
    else
    {
      TIMx->CCMR2 &= (uint32_t)(~((uint32_t)TIM_CCMR2_Output_OC3TE_Msk));
    }
}

/**
  * @name   TIM_OCTuning4_Cmd
  * @brief  EN: Output comparison 4 Tuning enable.
  *         CN: Ƚ4޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2 to select the TIM peripheral.
  *         CN: x1,2 ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 4 Tuning.
  *         CN: TIMx Ƚ4޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning4_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR2 |= TIM_CCMR2_Output_OC4TE_Msk;
    }
    else
    {
      TIMx->CCMR2 &= (uint32_t)(~((uint32_t)TIM_CCMR2_Output_OC4TE_Msk));
    }
}

/**
  * @name   TIM_OCTuning5_Cmd
  * @brief  EN: Output comparison 5 Tuning enable.
  *         CN: Ƚ5޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1 ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 5 Tuning.
  *         CN: TIMx Ƚ5޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning5_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR3 |= TIM_CCMR3_Output_OC5TE_Msk;
    }
    else
    {
      TIMx->CCMR3 &= (uint32_t)(~((uint32_t)TIM_CCMR3_Output_OC5TE_Msk));
    }
}

/**
  * @name   TIM_OCTuning1Update_Cmd
  * @brief  EN: Output comparison 1 Tuning Updata enable.
  *         CN: Ƚ1޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,2,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 1 Tuning Updata enable.
  *         CN: TIMx Ƚ1޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning1Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR1 |= TIM_CCMR1_Output_OC1TUE_Msk;
    }
    else
    {
      TIMx->CCMR1 &= (uint32_t)(~((uint32_t)TIM_CCMR1_Output_OC1TUE_Msk));
    }
}

/**
  * @name   TIM_OCTuning2Update_Cmd
  * @brief  EN: Output comparison 2 Tuning Updata enable.
  *         CN: Ƚ2޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2, 15, 16 to select the TIM peripheral.
  *         CN: x1,2,15,16ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 2 Tuning Updata enable.
  *         CN: TIMx Ƚ2޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning2Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR1 |= TIM_CCMR1_Output_OC2TUE_Msk;
    }
    else
    {
      TIMx->CCMR1 &= (uint32_t)(~((uint32_t)TIM_CCMR1_Output_OC2TUE_Msk));
    }
}

/**
  * @name   TIM_OCTuning3Update_Cmd
  * @brief  EN: Output comparison 3 Tuning Updata enable.
  *         CN: Ƚ3޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2 to select the TIM peripheral.
  *         CN: x1,2 ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 3 Tuning Updata enable.
  *         CN: TIMx Ƚ3޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning3Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR2 |= TIM_CCMR2_Output_OC3TUE_Msk;
    }
    else
    {
      TIMx->CCMR2 &= (uint32_t)(~((uint32_t)TIM_CCMR2_Output_OC3TUE_Msk));
    }
}

/**
  * @name   TIM_OCTuning4Update_Cmd
  * @brief  EN: Output comparison 4 Tuning Updata enable.
  *         CN: Ƚ4޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1, 2 to select the TIM peripheral.
  *         CN: x1,2 ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 4 Tuning Updata enable.
  *         CN: TIMx Ƚ4޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning4Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR2 |= TIM_CCMR2_Output_OC4TUE_Msk;
    }
    else
    {
      TIMx->CCMR2 &= (uint32_t)(~((uint32_t)TIM_CCMR2_Output_OC4TUE_Msk));
    }
}

/**
  * @name   TIM_OCTuning5Update_Cmd
  * @brief  EN: Output comparison 5 Tuning Updata enable.
  *         CN: Ƚ5޵ʹܡ
  * @param  TIMx: 
  *         EN: where x can be 1 to select the TIM peripheral.
  *         CN: x1 ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Output comparison 5 Tuning Updata enable.
  *         CN: TIMx Ƚ5޵ʹܵ״̬
  * @retval None
  */
void TIM_OCTuning5Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->CCMR3 |= TIM_CCMR3_Output_OC5TUE_Msk;
    }
    else
    {
      TIMx->CCMR3 &= (uint32_t)(~((uint32_t)TIM_CCMR3_Output_OC5TUE_Msk));
    }
}

/**
  * @name   TIM_TrigDelayClear_Cmd
  * @brief  EN: Clear OCiREF during trigger delay period.
  *         CN: ӳڼ OCiREF
  * @param  TIMx: 
  *         EN: where x can be 1, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,15,1617ѡTIMΧ豸
  * @param  NewState: 
  *         EN: New state of the TIMx Clear OCiREF during trigger delay period.
  *         CN: TIMx ӳڼ OCiREF״̬
  * @retval None
  */
void TIM_TrigDelayClear_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState)
{
    if (NewState != DISABLE)
    {
      TIMx->DLAMT |= TIM_DLAMT_DLS_Msk;
    }
    else
    {
      TIMx->DLAMT &= (uint32_t)(~((uint32_t)TIM_DLAMT_DLS_Msk));
    }
}

/**
  * @name   TIM_AntiMisoperateTrig_Config
  * @brief  EN: Anti misoperation trigger configuration.
  *         CN: 󴥷á
  * @param  TIMx: 
  *         EN: where x can be 1, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,15,1617ѡTIMΧ豸
  * @param  AntiMisoperateStart: 
  *         EN: Start time of anti misoperation trigger window.
  *         CN: 󴥷ڿʼʱ䡣
  *         @arg TIM_AntiMisoperateStart_Delay
  *         @arg TIM_AntiMisoperateStart_Immediate
  * @param  AntiMisoperateTime: 
  *         EN: Prevent false triggering window time.
  *         CN: 󴥷ʱ䡣
  *         @range 0x00 - 0xFF
  * @retval None
  */
void TIM_AntiMisoperateTrig_Config(TIM_TypeDef *TIMx, uint32_t AntiMisoperateStart, uint8_t AntiMisoperateTime)
{
  TIMx->DLAMT &= (uint32_t) ~((uint32_t)TIM_DLAMT_AMTDL_Msk | TIM_DLAMT_ATMG_Msk);
  TIMx->DLAMT |= AntiMisoperateStart | (AntiMisoperateTime << TIM_DLAMT_ATMG_Pos);
}

/**
  * @name   TIM_TrigDelay_Config
  * @brief  EN: Trigger delay window time setting.
  *         CN: ӳٴʱá
  * @param  TIMx: 
  *         EN: where x can be 1, 15, 16 or 17 to select the TIM peripheral.
  *         CN: x1,15,1617ѡTIMΧ豸
  * @param  TrigDelayTime: 
  *         EN: Trigger delay window time.
  *         CN: ӳٴʱ䡣
  *         @range 0x00 - 0xFF
  * @retval None
  */
void TIM_TrigDelay_Config(TIM_TypeDef *TIMx, uint8_t TrigDelayTime)
{
  TIMx->DLAMT &= (uint32_t) ~((uint32_t)TIM_DLAMT_DLG_Msk);
  TIMx->DLAMT |= TrigDelayTime;
}
