/**
  ******************************************************************************
  * @file    lcm32f06x_iwdg.c
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   This file provides all the iwdg firmware functions.
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
  /* Includes ------------------------------------------------------------------*/
#include "lcm32f06x_iwdg.h"

/* KR register bit mask */
#define KR_KEY_RELOAD ((uint16_t)0xAAAA)
#define KR_KEY_ENABLE ((uint16_t)0xCCCC)


/**
  * @name   IWDG_WriteAccessCmd
  * @brief  EN: Enables or disables write access to IWDG_PR and IWDG_RLR registers.
  *         CN: ûöIWDG_PRIWDG_RLRĴдʡ
  * @param  IWDG_WriteAccess: 
  *         EN: new state of write access to IWDG_PR and IWDG_RLR registers.
  *         CN: IWDG_PRIWDG_RLRĴдʵ״̬
  *            @arg IWDG_WriteAccess_Enable
  *            @arg IWDG_WriteAccess_Disable
  * @retval None
  */
void IWDG_WriteAccessCmd(uint16_t IWDG_WriteAccess)
{
  /* Check the parameters */
  assert_param(IS_IWDG_WRITE_ACCESS(IWDG_WriteAccess));
  IWDG->KR = IWDG_WriteAccess;
}

/**
  * @name   IWDG_SetPrescaler
  * @brief  EN: Sets IWDG Prescaler value.
  *         CN: IWDGԤƵֵ
  * @param  IWDG_Prescaler: 
  *         EN: specifies the IWDG Prescaler value.
  *         CN: ָIWDGԤƵֵ
  *            @arg IWDG_Prescaler_4
  *            @arg IWDG_Prescaler_8
  *            @arg IWDG_Prescaler_16
  *            @arg IWDG_Prescaler_32
  *            @arg IWDG_Prescaler_64
  *            @arg IWDG_Prescaler_128
  *            @arg IWDG_Prescaler_256
  * @retval None
  */
void IWDG_SetPrescaler(uint8_t IWDG_Prescaler)
{
  /* Check the parameters */
  assert_param(IS_IWDG_PRESCALER(IWDG_Prescaler));
  IWDG->PR = IWDG_Prescaler;
}

/**
  * @name   IWDG_SetReload
  * @brief  EN: Sets IWDG Prescaler value.
  *         CN: IWDGԤƵֵ
  * @param  Reload: 
  *         EN: specifies the IWDG Reload value(0 and 0x0FFF).
  *         CN: ָIWDG¼ֵ(0 and 0x0FFF)
  * @retval None
  */
void IWDG_SetReload(uint16_t Reload)
{
  /* Check the parameters */
  assert_param(IS_IWDG_RELOAD(Reload));
  IWDG->RLR = Reload;
}

/**
  * @name   IWDG_ReloadCounter
  * @brief  EN: Reloads IWDG counter with value defined in the reload register
  *         CN: ¼ؼĴжֵ¼IWDG
  * @retval None
  */
void IWDG_ReloadCounter(void)
{
  IWDG->KR = KR_KEY_RELOAD;
}

/**
  * @name   IWDG_SetWindowValue
  * @brief  EN: Sets the IWDG window value.
  *         CN: IWDGֵ
  * @param  WindowValue: 
  *         EN: specifies the window value to be compared to the downcounter.
  *         CN: ָҪ¼бȽϵĴֵ
  * @retval None
  */
void IWDG_SetWindowValue(uint16_t WindowValue)
{
  /* Check the parameters */
  assert_param(IS_IWDG_WINDOW_VALUE(WindowValue));
  IWDG->WINR = WindowValue;
}

/**
  * @name   IWDG_Enable
  * @brief  EN: Enables IWDG.
  *         CN: IWDG
  * @retval None
  */
void IWDG_Enable(void)
{
  IWDG->KR = KR_KEY_ENABLE;
}

/**
  * @name   IWDG_GetFlagStatus
  * @brief  EN: Checks whether the specified IWDG flag is set or not.
  *         CN: ָIWDG־Ƿá
  * @param  IWDG_FLAG: 
  *         EN: specifies the flag to check.
  *         CN: ָҪı־
  *            @arg IWDG_FLAG_PVU
  *            @arg IWDG_FLAG_RVU
  *            @arg IWDG_FLAG_WVU 
  * @retval FlagStatus
  */
FlagStatus IWDG_GetFlagStatus(uint16_t IWDG_FLAG)
{
  FlagStatus bitstatus = RESET;
  /* Check the parameters */
  assert_param(IS_IWDG_FLAG(IWDG_FLAG));
  if ((IWDG->SR & IWDG_FLAG) != (uint32_t)RESET)
  {
    bitstatus = SET;
  }
  else
  {
    bitstatus = RESET;
  }
  /* Return the flag status */
  return bitstatus;
}

