/**
  ******************************************************************************
  * @file    lcm32f06x_dma.c
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   This file provides all the DMA firmware functions.
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
/* Includes ------------------------------------------------------------------*/
#include "lcm32f06x_dma.h"

/**
  * @name   DMA_DeInit
  * @brief  EN: Deinitializes the DMAy_Channelx channel to their default reset values.
  *         CN:  DMAy_Channelx ͨʼΪĬֵ
  * @param  DMAy_Channelx: 
  *         EN: where y is 1 to select DMA, where x can be 0 or 1 to select DMAy channel.
  *         CN:  y  1 ѡ DMA x  0  1 ѡ DMAy ͨ
  * @retval None
  */
void DMA_DeInit(DMA_Channel_TypeDef *DMAy_Channelx)
{
  /* Check the parameters */
  uint32_t tmpreg;
  uint32_t ch;

  assert_param(IS_DMA_ALL_PERIPH(DMAy_Channelx));

  if (DMAy_Channelx == DMA1_Channel0)
    ch = 0;
  else if (DMAy_Channelx == DMA1_Channel1)
    ch = 1;
  //  else if(DMAy_Channelx==DMA1_Channel5)ch=5;
  //  else if(DMAy_Channelx==DMA1_Channel6)ch=6;//invaild
  else
    ch = 2;                // invaild
  DMA1->DmaCfgReg |= 0x01; //DMAģ
  tmpreg = DMA1->ChEnReg;
  tmpreg |= 1 << (ch + 8);
  tmpreg &= ~(1 << ch);
  DMA1->ChEnReg = tmpreg; //رնӦͨ
  DMAy_Channelx->SAR = 0x00;
  DMAy_Channelx->DAR = 0x00;
  DMAy_Channelx->CTL_L = 0x00; //1burstӦ1signalĬ1burstӦ4signalӦst
  DMAy_Channelx->CTL_H = 0x00; /*Ѻ˶*/
  DMAy_Channelx->CFG_L = 0x00; /*ʹӲź*/
  DMAy_Channelx->CFG_H = 0x04; /*PROTCTL=1*/

  DMA1->RawTfr &= ~(1 << ch);
  DMA1->RawBlock &= ~(1 << ch);
  DMA1->RawSrcTran &= ~(1 << ch);
  DMA1->RawDstTran &= ~(1 << ch);
  DMA1->RawErr &= ~(1 << ch);

  DMA1->StatusTfr &= ~(1 << ch);
  DMA1->StatusBlock &= ~(1 << ch);
  DMA1->StatusSrcTran &= ~(1 << ch);
  DMA1->StatusDstTran &= ~(1 << ch);
  DMA1->StatusErr &= ~(1 << ch);

  DMA1->ClearTfr = 1 << ch;
  DMA1->ClearBlock = 1 << ch;
  DMA1->ClearSrcTran = 1 << ch;
  DMA1->ClearDstTran = 1 << ch;
  DMA1->ClearErr = 1 << ch;
}

/**
  * @name   DMA_Init
  * @brief  EN: Initializes the DMA channel according to the specified.
  *             parameters in the DMA_InitStruct.
  *         CN:  DMA_InitStruct ָĲʼ DMA ͨ
  * @param  DMAy_Channelx: 
  *         EN: where y is 1 to select DMA, where x can be 0 or 1 to select DMAy channel
  *         CN:  y  1 ѡ DMA x  0  1 ѡ DMAy ͨ
  * @param  DMA_InitStruct:
  *         EN: pointer to a DMA_InitStruct structure that contains the 
  *             configuration information for the specified DMA peripheral.
  *         CN: ָ DMA_InitStruct ṹָ룬ýṹָ DMA Ϣ
  * @retval None
  */
void DMA_Init(DMA_Channel_TypeDef *DMAy_Channelx, DMA_InitTypeDef *DMA_InitStruct)
{
  /* Check the parameters */
  assert_param(IS_DMA_ALL_PERIPH(DMAy_Channelx));
  assert_param(IS_DMA_DIR(DMA_InitStruct->DMA_DIR));
  assert_param(IS_DMA_PERIPHERAL_INC_STATE(DMA_InitStruct->DMA_PeripheralInc));
  assert_param(IS_DMA_MEMORY_INC_STATE(DMA_InitStruct->DMA_MemoryInc));
  assert_param(IS_DMA_PERIPHERAL_DATA_SIZE(DMA_InitStruct->DMA_PeripheralDataSize));
  assert_param(IS_DMA_MEMORY_DATA_SIZE(DMA_InitStruct->DMA_MemoryDataSize));
  assert_param(IS_DMA_MODE(DMA_InitStruct->DMA_Mode));
  assert_param(IS_DMA_PRIORITY(DMA_InitStruct->DMA_Priority));
  assert_param(IS_DMA_M2M_STATE(DMA_InitStruct->DMA_M2M));
  assert_param(IS_DMA_MSIZE(DMA_InitStruct->DMA_MSIZE));

  if (DMA_InitStruct->DMA_DIR == DMA_DIR_PeripheralSRC)
  {
    DMAy_Channelx->SAR = DMA_InitStruct->DMA_PeripheralBaseAddr;
    DMAy_Channelx->DAR = DMA_InitStruct->DMA_MemoryBaseAddr;
    DMAy_Channelx->CTL_L_b.SRC_TR_WIDTH = DMA_InitStruct->DMA_PeripheralDataSize;
    DMAy_Channelx->CTL_L_b.DST_TR_WIDTH = DMA_InitStruct->DMA_MemoryDataSize;
    DMAy_Channelx->CTL_L_b.SINC = DMA_InitStruct->DMA_PeripheralInc;
    DMAy_Channelx->CTL_L_b.DINC = DMA_InitStruct->DMA_MemoryInc;
  }
  else
  {
    DMAy_Channelx->DAR = DMA_InitStruct->DMA_PeripheralBaseAddr;
    DMAy_Channelx->SAR = DMA_InitStruct->DMA_MemoryBaseAddr;
    DMAy_Channelx->CTL_L_b.DST_TR_WIDTH = DMA_InitStruct->DMA_PeripheralDataSize;
    DMAy_Channelx->CTL_L_b.SRC_TR_WIDTH = DMA_InitStruct->DMA_MemoryDataSize;
    DMAy_Channelx->CTL_L_b.SINC = DMA_InitStruct->DMA_MemoryInc;
    DMAy_Channelx->CTL_L_b.DINC = DMA_InitStruct->DMA_PeripheralInc;
  }
  DMAy_Channelx->CTL_H_b.BLOCK_TS = DMA_InitStruct->DMA_BufferSize; /*the max vaild value is 4095 @zhang*/
  DMAy_Channelx->CTL_L_b.TT_FC = DMA_InitStruct->DMA_M2M;
  DMAy_Channelx->CTL_L_b.SRC_MSIZE = DMA_InitStruct->DMA_MSIZE;
  DMAy_Channelx->CTL_L_b.DEST_MSIZE = DMA_InitStruct->DMA_MSIZE;
}

/**
  * @name   DMA_StructInit
  * @brief  EN: Fills each DMA_InitStruct member with its default value.
  *         CN: Ĭֵÿ DMA_InitStruct Ա
  * @param  DMA_InitStruct:
  *         EN: pointer to a DMA_InitStruct structure that contains the 
  *             configuration information for the specified DMA peripheral.
  *         CN: ָ DMA_InitStruct ṹָ룬ýṹָ DMA Ϣ
  * @retval None
  */
void DMA_StructInit(DMA_InitTypeDef *DMA_InitStruct)
{
  /*-------------- Reset DMA init structure parameters values ------------------*/
  /* Initialize the DMA_PeripheralBaseAddr member */
  DMA_InitStruct->DMA_PeripheralBaseAddr = 0;
  /* Initialize the DMA_MemoryBaseAddr member */
  DMA_InitStruct->DMA_MemoryBaseAddr = 0;
  /* Initialize the DMA_DIR member */
  DMA_InitStruct->DMA_DIR = DMA_DIR_PeripheralSRC;
  /* Initialize the DMA_BufferSize member */
  DMA_InitStruct->DMA_BufferSize = 0; /*ָǴٸsignal transfer*/
  /* Initialize the DMA_PeripheralInc member */
  DMA_InitStruct->DMA_PeripheralInc = DMA_PeripheralInc_Disable;
  /* Initialize the DMA_MemoryInc member */
  DMA_InitStruct->DMA_MemoryInc = DMA_MemoryInc_Enable;
  /* Initialize the DMA_PeripheralDataSize member */
  DMA_InitStruct->DMA_PeripheralDataSize = DMA_PeripheralDataSize_Byte;
  /* Initialize the DMA_MemoryDataSize member */
  DMA_InitStruct->DMA_MemoryDataSize = DMA_MemoryDataSize_Byte;
  /* Initialize the DMA_M2M member */
  DMA_InitStruct->DMA_M2M = DMA_M2M_Enable;
     /* Initialize the DMA_MSIZE member */
  DMA_InitStruct->DMA_MSIZE = DMA_MSIZE_1;
}

/**
  * @name   DMA_Cmd
  * @brief  EN: Enables or disables the specified DMAy_Channelx.
  *         CN: ûָֹ DMAy_Channelx
  * @param  DMAy_Channelx: 
  *         EN: where y is 1 to select DMA, where x can be 0 or 1 to select DMAy channel.
  *         CN:  y  1 ѡ DMA x  0  1 ѡ DMAy ͨ
  * @param  NewState: 
  *         EN: New state of the DMAy_Channelx.
  *         CN: DMAy_Channelx ״̬
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */
void DMA_Cmd(DMA_Channel_TypeDef *DMAy_Channelx, FunctionalState NewState)
{
  /* Check the parameters */
  uint32_t ch;
  uint32_t tmpreg;
  assert_param(IS_DMA_ALL_PERIPH(DMAy_Channelx));
  assert_param(IS_FUNCTIONAL_STATE(NewState));

  if (DMAy_Channelx == DMA1_Channel0)
    ch = 0;
  else if (DMAy_Channelx == DMA1_Channel1)
    ch = 1;
  else
    ch = 2; // invaild
  tmpreg = DMA1->ChEnReg;
  if (NewState != DISABLE)
  {
    /* Enable the selected DMAy Channelx */
    tmpreg |= 0x101 << ch;
//    DMA1->ChEnReg |= 0x100 << ch;
//    DMA1->ChEnReg |= 0x001 << ch;
  }
  else
  {
    /* Disable the selected DMAy Channelx */
    tmpreg |= 0x100 << ch;
    tmpreg &= ~(1 << ch);
  }
  DMA1->ChEnReg = tmpreg;
  
}

/**
  * @name   DMA_SetCurrDataCounter
  * @brief  EN: Set the current block transfer size.
  *         CN: õǰĿ鴫С
  * @param  DMAy_Channelx: 
  *         EN: where y is 1 to select DMA, where x can be 0 or 1 to select DMAy channel.
  *         CN:  y  1 ѡ DMA x  0  1 ѡ DMAy ͨ
  * @param  DataNumber: 
  *         EN: Block transfer size.
  *         CN: 鴫С
  * @retval None
  * @note   EN: This function can only be used when the DMAy_Channelx is disabled.
  *         CN: ֻ DMAy_Channelx ʱʹá
  */
void DMA_SetCurrDataCounter(DMA_Channel_TypeDef *DMAy_Channelx, uint16_t DataNumber)
{
  /* Check the parameters */
  assert_param(IS_DMA_ALL_PERIPH(DMAy_Channelx));

  /*Block Transfer Size */
  DMAy_Channelx->CTL_H = (0xFFF & DataNumber);
}

/**
  * @name   DMA_GetCurrDataCounter
  * @brief  EN: get the current block transfer size.
  *         CN: ȡǰĿ鴫С
  * @param  DMAy_Channelx: 
  *         EN: where y is 1 to select DMA, where x can be 0 or 1 to select DMAy channel.
  *         CN:  y  1 ѡ DMA x  0  1 ѡ DMAy ͨ
  * @retval EN: The number of data read by the source peripheral
  *         CN: Դȡ
  * @note   EN: The return value after the start of transmission is the total 
  *             number of data read from the source peripheral.
  *         CN: 俪ʼķֵΪԴȡ
  */
uint16_t DMA_GetCurrDataCounter(DMA_Channel_TypeDef *DMAy_Channelx)
{
  /* Check the parameters */
  assert_param(IS_DMA_ALL_PERIPH(DMAy_Channelx));
  /* Return the number of remaining data units for DMAy Channelx */

  return (uint16_t)((DMAy_Channelx->CTL_H) & 0xFFF);
}

/**
  * @name   DMA_ITConfig
  * @brief  EN: Enables or disables the specified DMAy Channelx interrupts.
  *         CN: ûָֹ DMAy ͨ x жϡ
  * @param  DMAy_Channelx: 
  *         EN: where y is 1 to select DMA, where x can be 0 or 1 to select DMAy channel.
  *         CN:  y  1 ѡ DMA x  0  1 ѡ DMAy ͨ
  * @param  DMA_IT: 
  *         EN: Clear or set the corresponding DMAy channel x interrupt.
  *         CN: öӦ DMAy ͨ x жϡ
  *             @arg DMA1_IT_TC0
  *             @arg DMA1_IT_TE0
  *             @arg DMA1_IT_TC1
  *             @arg DMA1_IT_TE1
  * @param  NewState: 
  *         EN: The new state of DMA_IT (SET or RESET).
  *         CN: DMA_IT״̬SETRESET
  * @retval None
  */
void DMA_ITConfig(DMA_Channel_TypeDef *DMAy_Channelx, uint32_t DMA_IT, FunctionalState NewState)
{
  /* Check the parameters */
  assert_param(IS_DMA_ALL_PERIPH(DMAy_Channelx));
  assert_param(IS_DMA_CONFIG_IT(DMA_IT));
  assert_param(IS_FUNCTIONAL_STATE(NewState));
  if (NewState != DISABLE)
  {
    DMAy_Channelx->CTL_L_b.INT_EN = 1;
    /* Enable the selected DMA interrupts */
    if ((DMA_IT & 0xF0) == 0x00)
    {
      DMA1->MaskBlock = (0x101 << (DMA_IT & 0x0F));
    }
    if ((DMA_IT & 0xF0) == 0x10)
    {
      DMA1->MaskErr = 0x101 << (DMA_IT & 0x0F);
    }
  }
  else
  {
    DMAy_Channelx->CTL_L_b.INT_EN = 0;
    /* Disable the selected DMA interrupts */
    if ((DMA_IT & 0xF0) == 0x00)
    {
      DMA1->MaskBlock = (0x100 << (DMA_IT & 0x0F));
    }
    if ((DMA_IT & 0xF0) == 0x10)
    {
      DMA1->MaskErr = 0x100 << (DMA_IT & 0x0F);
    }
  }
}

/**
  * @name   DMA_GetFlagStatus
  * @brief  EN: Get the specified DMAy channel x interrupt status.
  *         CN: ȡָ DMAy ͨ x ж״̬
  * @param  DMAy_FLAG: 
  *         EN: specifies the flag to get.
  *         CN: ָҪȡı־
  *             @arg DMA1_FLAG_TC0
  *             @arg DMA1_FLAG_TE0
  *             @arg DMA1_FLAG_TC1
  *             @arg DMA1_FLAG_TE1
  * @retval FlagStatus
  *         EN: The new state of DMA_FLAG (SET or RESET).
  *         CN: DMA_FLAG״̬SETRESET
  */
FlagStatus DMA_GetFlagStatus(uint32_t DMAy_FLAG)
{
  /* Check the parameters */
  uint32_t tmp;
  assert_param(IS_DMA_GET_FLAG(DMAy_FLAG));
  if ((DMAy_FLAG & 0xF0) == 0x00)
  {
    tmp = (DMA1->RawBlock >> (((uint8_t)DMAy_FLAG) & 0x0F)) & 0x01;
  }
  if ((DMAy_FLAG & 0xF0) == 0x10)
  {
    tmp = (DMA1->RawErr >> (DMAy_FLAG & 0x0F)) & 0x01;
  }
  if (tmp)
    return SET;
  return RESET;
}

/**
  * @name   DMA_ClearFlag
  * @brief  EN: Clears the DMAy Channelx's pending flags.
  *         CN: DMAy ChannelxĹ־
  * @param  DMAy_IT: 
  *         EN: specifies the flag to clear.
  *         CN: ָҪı־
  * @retval None
  */
void DMA_ClearFlag(uint32_t DMAy_FLAG)
{
  /* Check the parameters */
  assert_param(IS_DMA_CLEAR_FLAG(DMAy_FLAG));
  if ((DMAy_FLAG & 0xF0) == 0x00)
  {
    DMA1->ClearBlock |= 1 << (DMAy_FLAG & 0x0F);
  }
  if ((DMAy_FLAG & 0xF0) == 0x10)
  {
    DMA1->ClearErr |= 1 << (DMAy_FLAG & 0x0F);
  }
}

/**
  * @name   DMA_GetITStatus
  * @brief  EN: Checks whether the specified DMAy Channelx interrupt has occurred or not.
  *         CN: ָDMAy ChannelxжǷѷ
  * @param  DMAy_IT: 
  *         EN: specifies the DMA interrupt pending bit to clear.
  *         CN: ָҪDMAжϹλ
  * @retval ITStatus
  *         EN: The new state of DMA_IT (SET or RESET).
  *         CN: DMA_IT״̬SETRESET
  */
ITStatus DMA_GetITStatus(uint32_t DMAy_IT)
{
  /* Check the parameters */
  uint32_t tmp;
  assert_param(IS_DMA_GET_FLAG(DMAy_IT));
  if ((DMAy_IT & 0xF0) == 0x00)
  {
    tmp = (DMA1->StatusBlock >> (DMAy_IT & 0x0F)) & 0x01;
  }
  if ((DMAy_IT & 0xF0) == 0x10)
  {
    tmp = (DMA1->StatusErr >> (DMAy_IT & 0x0F)) & 0x01;
  }
  if (tmp)
    return SET;
  return RESET;
}

/**
  * @name   DMA_ClearITPendingBit
  * @brief  EN: Clears the DMAy Channelx's interrupt pending bits.
  *         CN: DMAy ChannelxжϹλ
  * @param  DMAy_IT: 
  *         EN: specifies the DMA interrupt pending bit to clear.
  *         CN: ָҪDMAжϹλ
  * @note   Clearing the Global interrupt (DMAy_IT_GLx) results in clearing all other
  *         interrupts relative to the same channel (Transfer Complete, Half-transfer
  *         Complete and Transfer Error interrupts: DMAy_IT_TCx, DMAy_IT_HTx and
  *         DMAy_IT_TEx).
  * @retval None
  */
void DMA_ClearITPendingBit(uint32_t DMAy_IT)
{
  /* Check the parameters */
  assert_param(IS_DMA_CLEAR_FLAG(DMAy_IT));
  if ((DMAy_IT & 0xF0) == 0x00)
  {
    DMA1->ClearBlock = 1 << (DMAy_IT & 0x0F);
  }
  if ((DMAy_IT & 0xF0) == 0x10)
  {
    DMA1->ClearErr = 1 << (DMAy_IT & 0x0F);
  }
}

/**
  * @name   DMA_RemapConfig
  * @brief  EN: Map DMA request number to peripheral device.
  *         CN: DMAӳ䵽衣
  * @param  DMAy_Channelx: 
  *         EN: where y is 1 to select DMA, where x can be 0 or 1 to select DMAy channel.
  *         CN:  y  1 ѡ DMA x  0  1 ѡ DMAy ͨ
  * @param  DMA_ReqNum: 
  *         EN: Specify the request number to which DMA should be mapped.
  *         CN: ָDMAҪӳ䵽š
  *             @arg DMA_ReqNum0
  *             @arg DMA_ReqNum1
  *             @arg DMA_ReqNum2
  *             @arg DMA_ReqNum3
  *             @arg DMA_ReqNum4
  *             @arg DMA_ReqNum5
  *             @arg DMA_ReqNum6
  *             @arg DMA_ReqNum7
  * @param  DMA_Request: 
  *         EN: Specify which peripheral to connect the mapping to.
  *         CN: ָӳӵĸ衣
  *             @arg DMA_REQ_TIM1_0
  *             @arg ...
  *             @arg DMA_REQ_SSP0_RX
  * @retval None
  */
void DMA_RemapConfig(DMA_Channel_TypeDef *DMAy_Channelx, uint32_t DMA_ReqNum, uint32_t DMA_Request)
{
  uint32_t tmpreg;
  assert_param(IS_REQ_NUM(DMA_ReqNum));
  assert_param(IS_DMA_REQ(DMA_Request));
  DMAy_Channelx->CFG_H_b.DEST_PER = DMA_ReqNum;
  DMAy_Channelx->CFG_H_b.SRC_PER = DMA_ReqNum;
  tmpreg = SYSCTRL->EDU_CFG0 & ~(0xF << (DMA_ReqNum * 4));
  tmpreg |= DMA_Request << (DMA_ReqNum * 4);
  sysctrl_access();
  SYSCTRL->EDU_CFG0 = tmpreg;
  __dekey();
}


/**
  * @name   DMA_ReEnable_Config
  * @brief  EN: DMA restart. After each DMA completion, the interface needs to be called in DMA non loop mode.
  *         CN: DMA ÿDMA֮ҪýӿڣDMAѭģʽ
  * @param  DMAy_Channelx: 
  *         EN: where y is 1 to select DMA, where x can be 0 or 1 to select DMAy channel.
  *         CN:  y  1 ѡ DMA x  0  1 ѡ DMAy ͨ
  * @param  SAdress: 
  *         EN: DMA source address.
  *         CN: DMA Դַ
  * @param  DAdress: 
  *         EN: DMA target address.
  *         CN: DMA Ŀַ
  * @param  NewState: 
  *         EN: Enable/Disable.
  *         CN: ʹֹܻ
  * @retval None
  */
void DMA_ReEnable_Config(DMA_Channel_TypeDef *DMAy_Channelx, uint32_t SAdress, uint32_t DAdress,FunctionalState NewState)
{
  uint32_t ch;
  uint32_t tmpreg;
  assert_param(IS_DMA_ALL_PERIPH(DMAy_Channelx));
  assert_param(IS_FUNCTIONAL_STATE(NewState));
  
  DMAy_Channelx->DAR = SAdress;
  DMAy_Channelx->SAR = DAdress;
  if (DMAy_Channelx == DMA1_Channel0)
    ch = 0;
  else if (DMAy_Channelx == DMA1_Channel1)
    ch = 1;
  else
    ch = 2; // invaild
  tmpreg = DMA1->ChEnReg;
  if (NewState != DISABLE)
  {
    /* Enable the selected DMAy Channelx */
    tmpreg |= 0x101 << ch;
  }
  else
  {
    /* Disable the selected DMAy Channelx */
    tmpreg |= 0x100 << ch;
    tmpreg &= ~(1 << ch);
  }
  DMA1->ChEnReg = tmpreg;
}

/**
  * @name   DMA_One_Config
  * @brief  EN: DMA single transfer mode.
  *         CN: DMAδģʽ
  * @param  DMAy_Channelx: 
  *         EN: where y is 1 to select DMA, where x can be 0 or 1 to select DMAy channel.
  *         CN:  y  1 ѡ DMA x  0  1 ѡ DMAy ͨ
  * @param  SAdress: 
  *         EN: DMA source address.
  *         CN: DMA Դַ
  * @param  DAdress: 
  *         EN: DMA target address.
  *         CN: DMA Ŀַ
  * @param  Block_size: 
  *         EN: DMA block transfer size.
  *         CN: DMA 鴫С
  * @retval None
  */
void DMA_One_Config(DMA_Channel_TypeDef *DMAy_Channelx, uint32_t SAdress, uint32_t DAdress,uint32_t Block_size)
{
    assert_param(IS_DMA_ALL_PERIPH(DMAy_Channelx));

    
    DMAy_Channelx->SAR = SAdress;
    DMAy_Channelx->DAR = DAdress;

    DMAy_Channelx->CTL_H_b.BLOCK_TS = Block_size;
}
