/**
  ******************************************************************************
  * @file    lcm32f06x_acmp.c
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   This file provides all the ACMP firmware functions.
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "lcm32f06x_acmp.h"

/**
  * @name   ACMP_DeInit
  * @brief  EN: Deinitializes ACMP peripheral registers to their default reset values.
  *         CN:  ACMP ĴʼΪĬֵ
  * @param  None
  * @retval None
  * @note   EN: DAC ACMP and OPA share the same clock enable which can be used once.
  *         CN: DAC ACMP OPA һʱʹܣʹһμɡ
  */
void ACMP_DeInit(void)
{
  RCC_APB0PeriphResetCmd(RCC_APB0Periph_ANACTRL, ENABLE);
  RCC_APB0PeriphResetCmd(RCC_APB0Periph_ANACTRL, DISABLE);
}

/**
  * @name   ACMP0_Init
  * @brief  EN: Initializes the ACMP peripheral according to the specified 
  *             parameters in the ACMP_InitStruct.
  *         CN:  ACMP_InitStruct ָĲʼ ACMP
  * @param  ACMP_InitStruct: 
  *         EN: pointer to a ACMP_InitTypeDef structure that contains the 
  *             configuration information for the specified ACMP peripheral.
  *         CN: ָ ACMP_InitTypeDef ṹָ룬ýṹָ ACMP Ϣ
  * @retval None
  */
void ACMP0_Init(ACMP_InitTypeDef *ACMP_InitStruct)
{
  uint32_t tmpreg = 0;
  uint32_t tmpreg1 = 0;
  uint32_t tmpreg2 = 0;
  tmpreg = ACMP->ACMP0_CSR;
  tmpreg &= ~(0x00007FF8);

  tmpreg1 = ACMP->ACMP0_MUXCSR;
  tmpreg1 &= ~(0x6000FF01);

  tmpreg2 = ACMP->ACMP0_EXTCFG;
  tmpreg2 &= ~(ANACTRL_ACMPX_EXTCFG_EXTDTG_Msk | ANACTRL_ACMPX_EXTCFG_EXTSE_Msk);

  tmpreg |= (((uint8_t)ACMP_InitStruct->ACMP_FILTER_SEL << 23) | (ACMP_InitStruct->ACMP_LPS << 15) | \
             (ACMP_InitStruct->ACMP_FS << 11) | ACMP_InitStruct->ACMP_Delay_time | \
              ACMP_InitStruct->ACMP_Is | (ACMP_InitStruct->ACMP_HYSEN << 2) | (ACMP_InitStruct->ACMP_FREN << 1));

  tmpreg1 |= (ACMP_InitStruct->ACMP_TrigMode | ACMP_InitStruct->ACMP_WorkMode | \
              ACMP_InitStruct->ACMP_P_Select| ACMP_InitStruct->ACMP_N_Select | \
              ((uint8_t)ACMP_InitStruct->ACMP_CNT_CFG << 1));

  tmpreg2 |= ((ACMP_InitStruct->EXTSELECT) | ACMP_InitStruct->TRIG_DELAY_TIME);

  sysctrl_access();
  SYSCTRL->EDU_CFG16 |= (uint32_t) (ACMP_InitStruct->EXTEN_MODE_SEL) | (ACMP_InitStruct->TRIG_SOURCE) | \
                                   (ACMP_InitStruct->CP_EXTSP << 12);
  __dekey();

  ACMP->ACMP0_CSR = tmpreg;
  ACMP->ACMP0_MUXCSR = tmpreg1;
  ACMP->ACMP0_EXTCFG = tmpreg2;
}

/**
  * @name   ACMP1_Init
  * @brief  EN: Initializes the ACMP0 peripheral according to the specified 
  *             parameters in the ACMP_InitStruct.
  *         CN:  ACMP_InitStruct ָĲʼ ACMP0
  * @param  ACMP_InitStruct: 
  *         EN: pointer to a ACMP_InitTypeDef structure that contains the 
  *             configuration information for the specified ACMP peripheral.
  *         CN: ָ ACMP_InitTypeDef ṹָ룬ýṹָ ACMP Ϣ
  * @retval None
  */
void ACMP1_Init(ACMP_InitTypeDef *ACMP_InitStruct)
{
  uint32_t tmpreg = 0;
  uint32_t tmpreg1 = 0;
  uint32_t tmpreg2 = 0;
  tmpreg = ACMP->ACMP1_CSR;
  tmpreg &= ~(0x00007FF8);

  tmpreg1 = ACMP->ACMP1_MUXCSR;
  tmpreg1 &= ~(0x6000FF01);

  tmpreg2 = ACMP->ACMP1_EXTCFG;
  tmpreg2 &= ~(ANACTRL_ACMPX_EXTCFG_EXTDTG_Msk | ANACTRL_ACMPX_EXTCFG_EXTSE_Msk);

  tmpreg |= (((uint8_t)ACMP_InitStruct->ACMP_FILTER_SEL << 23) | (ACMP_InitStruct->ACMP_LPS << 15) | \
             (ACMP_InitStruct->ACMP_FS << 11) | ACMP_InitStruct->ACMP_Delay_time | \
              ACMP_InitStruct->ACMP_Is | (ACMP_InitStruct->ACMP_HYSEN << 2) | (ACMP_InitStruct->ACMP_FREN << 1));

  tmpreg1 |= (ACMP_InitStruct->ACMP_TrigMode | ACMP_InitStruct->ACMP_WorkMode | \
              ACMP_InitStruct->ACMP_P_Select| ACMP_InitStruct->ACMP_N_Select | \
              ((uint8_t)ACMP_InitStruct->ACMP_CNT_CFG << 1));

  tmpreg2 |= ((ACMP_InitStruct->EXTSELECT) | ACMP_InitStruct->TRIG_DELAY_TIME);

  sysctrl_access();
  SYSCTRL->EDU_CFG16 |= (uint32_t) (ACMP_InitStruct->EXTEN_MODE_SEL << 2) | (ACMP_InitStruct->TRIG_SOURCE << 4) | \
                                   (ACMP_InitStruct->CP_EXTSP << 14);
  __dekey();


  ACMP->ACMP1_CSR = tmpreg;
  ACMP->ACMP1_MUXCSR = tmpreg1;
  ACMP->ACMP1_EXTCFG = tmpreg2;
}

/**
  * @name   ACMP_StructInit
  * @brief  EN: Fills each ACMP_InitTypeDef member with its default value.
  *         CN: Ĭֵÿ ACMP_InitTypeDef Ա
  * @param  ACMP_InitStruct: 
  *         EN: pointer to a ACMP_InitTypeDef structure that contains the 
  *             configuration information for the specified ACMP peripheral.
  *         CN: ָ ACMP_InitTypeDef ṹָ룬ýṹָ ACMP Ϣ
  * @retval None
  */
void ACMP_StructInit(ACMP_InitTypeDef *ACMP_InitStruct)
{
  /*--------------- Reset ACMP init structure parameters values -----------------*/
  ACMP_InitStruct->ACMP_TrigMode = ACMP_TrigMode_Software;          //Ƚģʽѡ

  ACMP_InitStruct->ACMP_WorkMode = ACMP_WorkMode_SingleCompar;      //Ƚģʽѡ

  ACMP_InitStruct->ACMP_P_Select = CP_PS_P0;                        //Ƚѡ

  ACMP_InitStruct->ACMP_N_Select = CP_NS_N0;                        //Ƚѡ

  ACMP_InitStruct->ACMP_CNT_CFG = 0;                                //ͨлʱѡ

  ACMP_InitStruct->ACMP_FILTER_SEL = 0;                             //Ƚ˲λѡ

  ACMP_InitStruct->ACMP_LPS = DISABLE;                              //͹ģʽʹ

  ACMP_InitStruct->ACMP_FS = DISABLE;                               //˲ʹ

  ACMP_InitStruct->ACMP_Delay_time = 0;                             //ӳ

  ACMP_InitStruct->ACMP_Is = DISABLE;                               //Ƚ

  ACMP_InitStruct->ACMP_HYSEN = DISABLE;                            //ʹ

  ACMP_InitStruct->ACMP_FREN = DISABLE;                             //Ӧʹ

  ACMP_InitStruct->EXTSELECT = ACMP_EXTSELG_DIS;                    //鴥Դʹ

  ACMP_InitStruct->TRIG_DELAY_TIME = TRIG_DELAY_TIME_0CLK;          //ⲿӳʱ

  ACMP_InitStruct->EXTEN_MODE_SEL = ACMP_EXTEN_RISE;                //ⲿģʽѡ

  ACMP_InitStruct->TRIG_SOURCE = EXTSELG0_TIM1_TRIG_OC1;            //ⲿԴѡ

  ACMP_InitStruct->CP_EXTSP = CP_EXTSP_POS;                         //Դѡ
};

/**
  * @name   ACMP0_SoftWareTrigger
  * @brief  EN: ACMP0 software triggers conversion.
  *         CN: ACMP0 ת
  * @retval None
  */
void ACMP0_SoftWareTrigger(void)
{
  ACMP->ACMP0_MUXCSR |= ANACTRL_ACMPX_MUXCSR_CMP_SEQ_START_Msk;
}

/**
  * @name   ACMP1_SoftWareTrigger
  * @brief  EN: ACMP1 software triggers conversion.
  *         CN: ACMP1 ת
  * @retval None
  */
void ACMP1_SoftWareTrigger(void)
{
  ACMP->ACMP1_MUXCSR |= ANACTRL_ACMPX_MUXCSR_CMP_SEQ_START_Msk;
}

/**
  * @name   GET_ACMP0_SEQ_RESLUT
  * @brief  EN: get the results of continuous compare.
  *         CN: ȡȽϵĽ ACMP0
  * @retval uint32_t
  */
uint32_t GET_ACMP0_SEQ_RESLUT(void)
{
  return ((ACMP->ACMP0_MUXCSR & ANACTRL_ACMPX_MUXCSR_CPOUT_SEQ_Msk) >> ANACTRL_ACMPX_MUXCSR_CPOUT_SEQ_Pos);
}
/**
  * @name   GET_ACMP1_SEQ_RESLUT
  * @brief  EN: get the results of continuous compare.
  *         CN: ȡȽϵĽ ACMP1
  * @retval uint32_t
  */
uint32_t GET_ACMP1_SEQ_RESLUT(void)
{
  return ((ACMP->ACMP1_MUXCSR & ANACTRL_ACMPX_MUXCSR_CPOUT_SEQ_Msk) >> ANACTRL_ACMPX_MUXCSR_CPOUT_SEQ_Pos);
}

/**
  * @name   ACMP0_Cmd
  * @brief  EN: Enables or disables the specified ACMP0 peripheral.
  *         CN: ûָ ACMP0 衣
  * @param  NewState: 
  *         EN: New state of the ACMP0 peripheral.
  *         CN: ACMP0 ״̬
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */
void ACMP0_Cmd(FunctionalState NewState)
{
  if (NewState != DISABLE)
  {
    ACMP->ACMP0_CSR |= (ANACTRL_ACMPX_CSR_CP_EN_Msk);
  }
  else
  {
    ACMP->ACMP0_CSR &= (~(ANACTRL_ACMPX_CSR_CP_EN_Msk));
  }
}

/**
  * @name   ACMP1_Cmd
  * @brief  EN: Enables or disables the specified ACMP1 peripheral.
  *         CN: ûָ ACMP1 衣
  * @param  NewState: 
  *         EN: New state of the ACMP1 peripheral.
  *         CN: ACMP1 ״̬
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */
void ACMP1_Cmd(FunctionalState NewState)
{
  if (NewState != DISABLE)
  {
    ACMP->ACMP1_CSR |= (ANACTRL_ACMPX_CSR_CP_EN_Msk);
  }
  else
  {
    ACMP->ACMP1_CSR &= (~(ANACTRL_ACMPX_CSR_CP_EN_Msk));
  }
}

/**
  * @name   ACMP0_FREN_Cmd
  * @brief  EN: ACMP0 Quick Response Enable.
  *         CN: ACMP0 Ӧʹܡ
  * @param  NewState: 
  *         EN: New state of the ACMP0 Quick Response.
  *         CN: ACMP0 Ӧ״̬
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */
void ACMP0_FREN_Cmd(FunctionalState NewState)
{
  if (NewState != DISABLE)
  {
    ACMP->ACMP0_CSR |= (ANACTRL_ACMPX_CSR_CP_FREN_Msk);
  }
  else
  {
    ACMP->ACMP0_CSR &= (~(ANACTRL_ACMPX_CSR_CP_FREN_Msk));
  }
}

/**
  * @name   ACMP1_FREN_Cmd
  * @brief  EN: ACMP1 Quick Response Enable.
  *         CN: ACMP1 Ӧʹܡ
  * @param  NewState: 
  *         EN: New state of the ACMP1 Quick Response.
  *         CN: ACMP1 Ӧ״̬
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */
void ACMP1_FREN_Cmd(FunctionalState NewState)
{
  if (NewState != DISABLE)
  {
    ACMP->ACMP1_CSR |= (ANACTRL_ACMPX_CSR_CP_FREN_Msk);
  }
  else
  {
    ACMP->ACMP1_CSR &= (~(ANACTRL_ACMPX_CSR_CP_FREN_Msk));
  }
}

/**
  * @name   Acmp0_trim
  * @brief  EN: acmp0 tuning value writing function.
  *         CN: acmp0 ޵ֵд뺯
  * @param  trim: 
  *         EN: ACMP0 adjustment value.
  *         CN: ACMP0 ޵ֵ
  * @retval None
  */
void Acmp0_trim(uint8_t trim)
{
  ACMP->ACMP0_CSR = ((ACMP->ACMP0_CSR & (~ANACTRL_ACMPX_CSR_CP_PF_Msk)) | (trim << ANACTRL_ACMPX_CSR_CP_PF_Pos));
}

/**
  * @name   Acmp1_trim
  * @brief  EN: acmp1 tuning value writing function.
  *         CN: acmp1 ޵ֵд뺯
  * @param  trim: 
  *         EN: ACMP1 adjustment value.
  *         CN: ACMP1 ޵ֵ
  * @retval None
  */
void Acmp1_trim(uint8_t trim)
{
  ACMP->ACMP1_CSR = ((ACMP->ACMP1_CSR & (~ANACTRL_ACMPX_CSR_CP_PF_Msk)) | (trim << ANACTRL_ACMPX_CSR_CP_PF_Pos));
}
/**
  * @name   get_acmp0_trim
  * @brief  EN: Reading ACMP0 tuning values and writing functions.
  *         CN: ȡ ACMP0 ޵ֵ
  * @retval uint8_t
  *         EN: ACMP0 adjustment value.
  *         CN: ACMP0 ޵ֵ
  */
uint8_t get_acmp0_trim(void)
{
  return ((ACMP->ACMP0_CSR & ANACTRL_ACMPX_CSR_CP_PF_Msk) >> ANACTRL_ACMPX_CSR_CP_PF_Pos);
}

/**
  * @name   get_acmp1_trim
  * @brief  EN: Reading ACMP1 tuning values and writing functions.
  *         CN: ȡ ACMP1 ޵ֵ
  * @retval uint8_t
  *         EN: ACMP1 adjustment value.
  *         CN: ACMP1 ޵ֵ
  */
uint8_t get_acmp1_trim(void)
{
  return ((ACMP->ACMP1_CSR & ANACTRL_ACMPX_CSR_CP_PF_Msk) >> ANACTRL_ACMPX_CSR_CP_PF_Pos);
}


/**
  * @name   ACMP0_TrimInit
  * @brief  EN: ACMP0 calibration function.
  *         CN: Ƚ0У׼
  * @retval None
  */
void ACMP0_TrimInit(void)
{
  uint8_t cptrim = 0x00;
  unsigned char tmpregs = 0;
  unsigned char flag = 0;

  tmpregs = ((ACMP->ACMP0_CSR) >> 28) & 0x01;

  if (tmpregs == 0)
  {
    while (flag != 1)
    {
      Acmp0_trim(cptrim);
      tmpregs = ((ACMP->ACMP0_CSR) >> 28) & 0x01;
      if (tmpregs == 1)
      {
        flag = 1;
      }
      else if (cptrim == 0x1f)
      {
        flag = 1;
      }
      else
      {
        cptrim++;
      }
    }
  }
}

/**
  * @name   ACMP1_TrimInit
  * @brief  EN: ACMP1 calibration function.
  *         CN: Ƚ1У׼
  * @retval None
  */
void ACMP1_TrimInit(void)
{
  uint8_t cptrim = 0x00;
  unsigned char tmpregs = 0;
  unsigned char flag = 0;

  tmpregs = ((ACMP->ACMP1_CSR) >> 28) & 0x01;

  if (tmpregs == 0)
  {
    while (flag != 1)
    {
      Acmp1_trim(cptrim);
      tmpregs = ((ACMP->ACMP1_CSR) >> 28) & 0x01;
      if (tmpregs == 1)
      {
        flag = 1;
      }
      else if (cptrim == 0x1f)
      {
        flag = 1;
      }
      else
      {
        cptrim++;
      }
    }
  }
}

/**
  * @name   Hallmid_IP1_Config
  * @brief  EN: Hallmid IP1 config.
  *         CN: Hallmid IP1 á
  * @param  HALL_IP1: 
  *         EN: Hallmid IP1 Channel selection.
  *         CN: Hallmid IP1 ͨѡ
  *             @arg HALL_IP1_ACMP0_P4
  *             @arg HALL_IP1_ACMP0_P0
  * @param  NewState: 
  *         EN: New state of the Hallmid IP1.
  *         CN: Hallmid IP1 ״̬
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */
void Hallmid_IP1_Config(uint32_t HALL_IP1, FunctionalState NewState)
{
  assert_param(IS_HALL_IP1(HALL_IP1));

  if (NewState == ENABLE)
  {
    ANACTRL->HALL_CR_b.HALL_IP1_SEL = HALL_IP1;
    ANACTRL->HALL_CR_b.HALL_SW1 = 1;
  }
  else
  {
    ANACTRL->HALL_CR_b.HALL_SW1 = 0;
  }
} 

/**
  * @name   Hallmid_IP2_Config
  * @brief  EN: Hallmid IP2 config.
  *         CN: Hallmid IP2 á
  * @param  HALL_IP1: 
  *         EN: Hallmid IP2 Channel selection.
  *         CN: Hallmid IP2 ͨѡ
  *             @arg HALL_IP2_ACMP0_P5
  *             @arg HALL_IP2_ACMP0_P1
  * @param  NewState: 
  *         EN: New state of the Hallmid IP2.
  *         CN: Hallmid IP2 ״̬
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */

void Hallmid_IP2_Config(uint32_t HALL_IP2, FunctionalState NewState)
{
  assert_param(IS_HALL_IP2(HALL_IP2));
  
  if (NewState == ENABLE)
  {
    ANACTRL->HALL_CR_b.HALL_IP2_SEL = HALL_IP2;
    ANACTRL->HALL_CR_b.HALL_SW2 = 1;
  }
  else
  {
    ANACTRL->HALL_CR_b.HALL_SW2 = 0;
  }
}

/**
  * @name   Hallmid_IP3_Config
  * @brief  EN: Hallmid IP3 config.
  *         CN: Hallmid IP3 á
  * @param  HALL_IP1: 
  *         EN: Hallmid IP3 Channel selection.
  *         CN: Hallmid IP3 ͨѡ
  *             @arg HALL_IP3_ACMP0_P6
  *             @arg HALL_IP3_ACMP0_P2
  * @param  NewState: 
  *         EN: New state of the Hallmid IP3.
  *         CN: Hallmid IP3 ״̬
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */
void Hallmid_IP3_Config(uint32_t HALL_IP3, FunctionalState NewState)
{
  assert_param(IS_HALL_IP3(HALL_IP3));

  if (NewState == ENABLE)
  {
    ANACTRL->HALL_CR_b.HALL_IP3_SEL = HALL_IP3;
    ANACTRL->HALL_CR_b.HALL_SW3 = 1;
  }
  else
  {
    ANACTRL->HALL_CR_b.HALL_SW3 = 0;
  }
}


/**
  * @name   Hallmid_Cmd
  * @brief  EN: Hallmid enable. 
  *         CN: Hallmid ʹܡ
  * @param  NewState: 
  *         EN: New state of the Hallmid.
  *         CN: Hallmid ״̬
  *             @arg ENABLE
  *             @arg DISABLE
  * @retval None
  */
void Hallmid_Cmd(FunctionalState NewState)
{
  if (NewState == ENABLE)
  {
    ANACTRL->HALL_CR_b.HALL_MID_EN = 1;
  }
  else
  {
    ANACTRL->HALL_CR_b.HALL_MID_EN = 0;
  }
}

/**
  * @name   ACMP_ANA_CVREFCfg
  * @brief  EN: ACMP internal reference voltage configuration.
  *         CN: ACMPڲοѹá
  * @param  NewState: 
  *         EN: Enable/disable comparator internal voltage CVREF.
  *         CN: / Ƚڲѹ CVREF
  *             @arg ENABLE
  *             @arg DISABLE
  * @param  VRH_SEL: 
  *         EN: Selection of voltage divider resistance range.
  *         CN: ѹ跶Χѡ
  *             @arg VRH_SEL_VDDA_0
  *             @arg VRH_SEL_VDDA_1
  *             @arg VRH_SEL_VDDA_2
  *             @arg VRH_SEL_VDDA_3
  *             @arg VRH_SEL_ADC_VREF_0
  *             @arg VRH_SEL_ADC_VREF_1
  *             @arg VRH_SEL_ADC_VREF_2
  *             @arg VRH_SEL_ADC_VREF_3
  * @param  RDS_SEL: 
  *         EN: Selection of voltage divider.
  *         CN: ѹѡ
  *             @range 0x0 - 0xF
  * @retval None
  */
void ACMP_ANA_CVREFCfg(FunctionalState NewState, uint32_t VRH_SEL,uint8_t RDS_SEL)
{
  uint32_t temprg = 0;
  assert_param(IS_VRH_SEL(VRH_SEL));

  if (NewState == ENABLE)
  {
    ANACTRL->ANA_CTRL_b.CVREFEN = 1;
  }
  else
  {
    ANACTRL->ANA_CTRL_b.CVREFEN = 0;
  }
  temprg = ANACTRL->ANA_CTRL;
  temprg &= ~(0xFE000000);
  temprg |= (VRH_SEL | (RDS_SEL << ANACTRL_ANA_CTRL_RDS_Pos));
  ANACTRL->ANA_CTRL = temprg;
}


/**
  * @name   ACMP0_BlankingCfg
  * @brief  EN: ACMP0 Hidden Source Selection.
  *         CN: ACMP0 Դѡ
  * @param  BlankingSource: 
  *         EN: Select the hidden source, and the parameters can be or operated on.
  *         CN: ѡԴԻ
  *             @arg Blanking_Software
  *             @arg Blanking_TIM2_TRIG_OC4
  *             @arg Blanking_TIM15_TRIG_OC2
  *             @arg Blanking_TIM15_TRIG_OC1
  *             @arg Blanking_TIM1_TRIG_OC5
  *             @arg Blanking_TIM1_TRIG_OC3
  *             @arg Blanking_TIM1_TRIG_OC2
  *             @arg Blanking_TIM1_TRIG_OC1
  * @param  NewState: 
  *         EN: Enable/disable hidden source selection.
  *         CN: ʹ/ʧԴѡ
  *             @arg ENABLE
  *             @arg DISABLE

  * @retval None
  */
void ACMP0_BlankingCfg(uint32_t BlankingSource, FunctionalState NewState)
{
  if (NewState != DISABLE)
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG15 |= BlankingSource;
    __dekey();
  }
  else
  {
      sysctrl_access();
      SYSCTRL->EDU_CFG15 &= (~BlankingSource);
      __dekey();
  }
}

/**
  * @name   ACMP1_BlankingCfg
  * @brief  EN: ACMP1 Hidden Source Selection.
  *         CN: ACMP1 Դѡ
  * @param  BlankingSource: 
  *         EN: Select the hidden source, and the parameters can be or operated on.
  *         CN: ѡԴԻ
  *             @arg Blanking_Software
  *             @arg Blanking_TIM2_TRIG_OC4
  *             @arg Blanking_TIM15_TRIG_OC2
  *             @arg Blanking_TIM15_TRIG_OC1
  *             @arg Blanking_TIM1_TRIG_OC5
  *             @arg Blanking_TIM1_TRIG_OC3
  *             @arg Blanking_TIM1_TRIG_OC2
  *             @arg Blanking_TIM1_TRIG_OC1
  * @param  NewState: 
  *         EN: Enable/disable hidden source selection.
  *         CN: ʹ/ʧԴѡ
  *             @arg ENABLE
  *             @arg DISABLE

  * @retval None
  */
void ACMP1_BlankingCfg(uint32_t BlankingSource, FunctionalState NewState)
{
  if (NewState != DISABLE)
  {
    sysctrl_access();
    SYSCTRL->EDU_CFG15 |= (BlankingSource << 8);
    __dekey();
  }
  else
  {
      sysctrl_access();
      SYSCTRL->EDU_CFG15 &= (~(BlankingSource << 8));
      __dekey();
  }
}
/**
  * @name   ACMP0_PollModeCfg
  * @brief  EN: ACMP0 polling mode configuration.
  *         CN: ACMP0 ѯģʽá
  * @param  Poll_Mode: 
  *         EN: Positive and negative selection of channel polling.
  *         CN: ͨѯѡ
  *             @arg POLL_MODE_POSITIVE
  *             @arg POLL_MODE_NEGATIVE
  * @param  Poll_ChannelNum: 
  *         EN: Polling comparison channel selection.
  *         CN: ѯȽͨѡ
  *             @arg POLL_CHANNEL_NUM_1
  *             @arg POLL_CHANNEL_NUM_2
  *             @arg POLL_CHANNEL_NUM_3
  *             @arg POLL_CHANNEL_NUM_4
  * @param  Poll_OutputType: 
  *         EN: ACMP Selection of external output results.
  *         CN: ACMP ѡ
  *             @arg POLL_OUTTYPE_Normal
  *             @arg POLL_OUTTYPE_SEQ0
  *             @arg POLL_OUTTYPE_SEQ1
  *             @arg POLL_OUTTYPE_SEQ2
  *             @arg POLL_OUTTYPE_SEQ3
  *             @arg POLL_OUTTYPE_SEQ4
  *             @arg POLL_OUTTYPE_SEQ5
  *             @arg POLL_OUTTYPE_SEQ6
  *             @arg POLL_OUTTYPE_SEQ7
  * @retval None
  */
void ACMP0_PollModeCfg(uint32_t Poll_Mode, uint32_t Poll_ChannelNum, uint32_t Poll_OutputType)
{
  ACMP->ACMP0_CHNLSEL &= ~(ANACTRL_ACMPX_CHNLSEL_OUT_TYPE_Msk | \
                           ANACTRL_ACMPX_CHNLSEL_CHNL_NUM_Msk | \
                           ANACTRL_ACMPX_CHNLSEL_POLL_PS_Msk | \
                           ANACTRL_ACMPX_CHNLSEL_CHNLSEL_OUT_Msk);
  ACMP->ACMP0_CHNLSEL |= Poll_Mode | Poll_ChannelNum | Poll_OutputType;
}

/**
  * @name   ACMP1_PollModeCfg
  * @brief  EN: ACMP1 polling mode configuration.
  *         CN: ACMP1 ѯģʽá
  * @param  Poll_Mode: 
  *         EN: Positive and negative selection of channel polling.
  *         CN: ͨѯѡ
  *             @arg POLL_MODE_POSITIVE
  *             @arg POLL_MODE_NEGATIVE
  * @param  Poll_ChannelNum: 
  *         EN: Polling comparison channel selection.
  *         CN: ѯȽͨѡ
  *             @arg POLL_CHANNEL_NUM_1
  *             @arg POLL_CHANNEL_NUM_2
  *             @arg POLL_CHANNEL_NUM_3
  *             @arg POLL_CHANNEL_NUM_4
  * @param  Poll_OutputType: 
  *         EN: ACMP Selection of external output results.
  *         CN: ACMP ѡ
  *             @arg POLL_OUTTYPE_Normal
  *             @arg POLL_OUTTYPE_SEQ0
  *             @arg POLL_OUTTYPE_SEQ1
  *             @arg POLL_OUTTYPE_SEQ2
  *             @arg POLL_OUTTYPE_SEQ3
  *             @arg POLL_OUTTYPE_SEQ4
  *             @arg POLL_OUTTYPE_SEQ5
  *             @arg POLL_OUTTYPE_SEQ6
  *             @arg POLL_OUTTYPE_SEQ7
  * @retval None
  */
void ACMP1_PollModeCfg(uint32_t Poll_Mode, uint32_t Poll_ChannelNum, uint32_t Poll_OutputType)
{
  ACMP->ACMP1_CHNLSEL &= ~(ANACTRL_ACMPX_CHNLSEL_OUT_TYPE_Msk | \
                           ANACTRL_ACMPX_CHNLSEL_CHNL_NUM_Msk | \
                           ANACTRL_ACMPX_CHNLSEL_POLL_PS_Msk | \
                           ANACTRL_ACMPX_CHNLSEL_CHNLSEL_OUT_Msk);
  ACMP->ACMP1_CHNLSEL |= Poll_Mode | Poll_ChannelNum | Poll_OutputType;
}

/**
  * @name   ACMP0_PollChannelSelxCfg
  * @brief  EN: ACMP0 Polling comparison channel position x channel selection selection.
  *         CN: ACMP0 ѯȽͨλ x ͨѡѡ
  * @param  Poll_Position: 
  *         EN: Polling comparison channel position x selection.
  *         CN: ѯȽͨλxѡ
  *             @arg POLL_POSITION_0
  *             @arg POLL_POSITION_1
  *             @arg POLL_POSITION_2
  *             @arg POLL_POSITION_3
  * @param  Poll_ChannelSel: 
  *         EN: ACMP Polling comparison channel selection.
  *         CN: ACMP ѯȽͨѡ
  *             @arg POLL_CHANNEL_0
  *             @arg POLL_CHANNEL_1
  *             @arg POLL_CHANNEL_2
  *             @arg POLL_CHANNEL_3
  *             @arg POLL_CHANNEL_4
  *             @arg POLL_CHANNEL_5
  *             @arg POLL_CHANNEL_6
  *             @arg POLL_CHANNEL_7
  * @retval None
  */
void ACMP0_PollChannelSelxCfg(uint32_t Poll_Position, uint32_t Poll_ChannelSel)
{
  ACMP->ACMP0_CHNLSEL &= ~(Poll_ChannelSel << Poll_Position);
  ACMP->ACMP0_CHNLSEL |= (Poll_ChannelSel << Poll_Position);
}

/**
  * @name   ACMP1_PollChannelSelxCfg
  * @brief  EN: ACMP1 Polling comparison channel position x channel selection selection.
  *         CN: ACMP1 ѯȽͨλ x ͨѡѡ
  * @param  Poll_Position: 
  *         EN: Polling comparison channel position x selection.
  *         CN: ѯȽͨλxѡ
  *             @arg POLL_POSITION_0
  *             @arg POLL_POSITION_1
  *             @arg POLL_POSITION_2
  *             @arg POLL_POSITION_3
  * @param  Poll_ChannelSel: 
  *         EN: ACMP Polling comparison channel selection.
  *         CN: ACMP ѯȽͨѡ
  *             @arg POLL_CHANNEL_0
  *             @arg POLL_CHANNEL_1
  *             @arg POLL_CHANNEL_2
  *             @arg POLL_CHANNEL_3
  *             @arg POLL_CHANNEL_4
  *             @arg POLL_CHANNEL_5
  *             @arg POLL_CHANNEL_6
  *             @arg POLL_CHANNEL_7
  * @retval None
  */
void ACMP1_PollChannelSelxCfg(uint32_t Poll_Position, uint32_t Poll_ChannelSel)
{
  ACMP->ACMP1_CHNLSEL &= ~(Poll_ChannelSel << Poll_Position);
  ACMP->ACMP1_CHNLSEL |= (Poll_ChannelSel << Poll_Position);
}

