/**
  ******************************************************************************
  * @file    lcm32f06x_I2cEmulation.c 
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   This file provides all the I2C software firmware functions.
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */

#include "lcm32f06x.h"
#include "lcm32f06x_gpio.h"
#include "lcm32f06x_i2c.h"

#define AT24C01   127
#define AT24C02   255
#define AT24C04   511
#define AT24C08   1023
#define AT24C16   2047
#define AT24C32   4095
#define AT24C64   8191
#define AT24C128  16383
#define AT24C256  32767

#define IIC_SCL_H   GPIO_SetBits(GPIOA, GPIO_Pin_9)
#define IIC_SCL_L   GPIO_ResetBits(GPIOA, GPIO_Pin_9)
#define IIC_SDA_H   GPIO_SetBits(GPIOA, GPIO_Pin_10)
#define IIC_SDA_L   GPIO_ResetBits(GPIOA, GPIO_Pin_10)
#define READ_SDA    GPIO_ReadInputDataBit(GPIOA, GPIO_Pin_10)

#define SDA_IN()   \
    {    \
        GPIOA->MODER &= 0xFFCFFFFF; \
    }
#define SDA_OUT()  \
    {    \
        GPIOA->MODER |= 0x00100000; \
    }
void delay_us(int i)
{
    while (i--)
    {
        __NOP(); __NOP(); __NOP(); __NOP(); __NOP();
    }
}

/**
  * @name   IIC_Start
  * @brief  EN: IIC Start,when CLK is high,DATA change form high to low.
  *         CN: IIC CLKΪʱDATAӸ߱Ϊ͡
  * @retval None
  */
void IIC_Start(void)
{
    SDA_OUT(); // sda /Line output
    IIC_SCL_H;
    IIC_SDA_H;
    delay_us(5);
    IIC_SDA_L;
    delay_us(5);
}
/**
  * @name   IIC_Stop
  * @brief  EN: IIC stop,when CLK is high DATA change form low to high.
  *         CN: IIC ֹͣIICֹͣCLKΪʱDATAӵͱΪߡ
  * @retval None
  */
void IIC_Stop(void)
{
    SDA_OUT(); // sda /Line output
    IIC_SDA_L;
    IIC_SCL_L;
    delay_us(5);
    IIC_SCL_H;
    IIC_SDA_H; //I2C߽ź /Send the I2C bus end signal
    delay_us(5);
}


/**
  * @name   IIC_Wait_Ack
  * @brief  EN: Wait for the response signal to arrive.
  *         CN: ȴӦźŵ
  * @retval EN: 1:Failure to receive a reply;   0:Response received successfully.
  *         CN: 1:Ӧʧ;   0:Ӧɹ
  */
uint8_t IIC_Wait_Ack(void)
{
    uint8_t ucErrTime = 0;
    SDA_IN(); // SDAΪ /Set to input
    IIC_SDA_H;
    delay_us(5);
    IIC_SCL_H;
    delay_us(5);
    while (READ_SDA)
    {
        ucErrTime++;
        if (ucErrTime > 250)
        {
            IIC_Stop();
            return 1;
        }
    }
    IIC_SCL_L; //ʱ0 /Clock output 0
    return 0;
}

/**
  * @name   IIC_Ack
  * @brief  EN: Generating an ACK response.
  *         CN: ACKӦ
  * @retval None
  */
void IIC_Ack(void)
{
    IIC_SCL_L;
    SDA_OUT();
    IIC_SDA_L;
    delay_us(5);
    IIC_SCL_H;
    delay_us(5);
    IIC_SCL_L;
}

/**
  * @name   IIC_NAck
  * @brief  EN: No ACK response is generated.
  *         CN: ACKӦ
  * @retval None
  */
void IIC_NAck(void)
{
    IIC_SCL_L;
    SDA_OUT();
    IIC_SDA_H;
    delay_us(5);
    IIC_SCL_H;
    delay_us(5);
    IIC_SCL_L;
}

/**
  * @name   IIC_Send_Byte
  * @brief  EN: IIC sends byte data.
  *         CN: IIC ֽݡ
  * @retval None
  */
void IIC_Send_Byte(uint8_t txd)
{
    uint8_t t = 0;
    SDA_OUT();
    IIC_SCL_L; //ʱӿʼݴ /Lower the clock to start data transmission
    for (t = 0; t < 8; t++)
    {
        if (txd & 0x80)
            IIC_SDA_H;
        else
            IIC_SDA_L;
        txd <<= 1;
        delay_us(5);
        IIC_SCL_H;
        delay_us(5);
        IIC_SCL_L;
        delay_us(5);
    }
}

/**
  * @name   IIC_Read_Byte
  * @brief  EN: Read 1 byte, ACK =1, send ACK, ack=0, send nACK.
  *         CN: 1ֽڣack=1ʱACKack=0nACK
  * @retval uint8_t
  *         EN: Data read.
  *         CN: ݡ
  */
uint8_t IIC_Read_Byte(uint8_t ack)
{
    uint8_t receive = 0;
    uint8_t i = 0;
    SDA_IN(); // SDAΪ/Set to input
    for (i = 0; i < 8; i++)
    {
        IIC_SCL_L;
        delay_us(5);
        IIC_SCL_H;
        receive <<= 1;
        if (READ_SDA)
            receive++;
        delay_us(5);
    }
    if (!ack)
        IIC_NAck(); //nACK /Send nACK
    else
        IIC_Ack(); //ACK /Send ACK
    return receive;
}

/**
  * @name   IIC_Init
  * @brief  EN: Simulate IIC initialization.
  *         CN: ģIICʼ
  * @retval None
  */
void IIC_Init(void)
{
    IIC_SCL_H;
    IIC_SDA_H;
    delay_us(5000);
}

/**
  * @name   AT24CXX_ReadOneByte
  * @brief  EN: Read a piece of data at AT24CXX at the address specified.
  *         CN: AT24CXXַָһݡ
  * @param  ReadAddr:
  *         EN: Address to start reading.
  *         CN: ʼĵַ
  * @retval uint8_t
  *         EN: Data read.
  *         CN: ݡ
  */
uint8_t AT24CXX_ReadOneByte(uint16_t ReadAddr)
{
    uint8_t temp = 0;
    IIC_Start();
    if (AT24C02 > AT24C16)
    {
        IIC_Send_Byte(0XA0); //д/Sending Write Commands
        IIC_Wait_Ack();
        IIC_Send_Byte(ReadAddr >> 8); //͸ߵַ/Send high address
        IIC_Wait_Ack();
    }
    else
        IIC_Send_Byte(0XA0 + ((ReadAddr / 256) << 1)); //ַ0XA0,д/Sending device address 0XA0 writes data

    IIC_Wait_Ack();
    IIC_Send_Byte(ReadAddr % 256); //͵͵ַ /Send low address
    IIC_Wait_Ack();
    IIC_Start();
    IIC_Send_Byte(0XA1); //ģʽ /Enter receive mode
    IIC_Wait_Ack();
    temp = IIC_Read_Byte(0);
    IIC_Stop(); //һֹͣ /Produces a stop condition
    return temp;
}

/**
  * @name   AT24CXX_WriteOneByte
  * @brief  EN: Writes a piece of data at AT24CXX at the address specified.
  *         CN: AT24CXXַָдһݡ
  * @param  WriteAddr:
  *         EN: Destination address of data to be written.
  *         CN: дݵĿĵַ
  * @param  DataToWrite:
  *         EN: The data to be written.
  *         CN: Ҫдݡ
  * @retval None
  */
void AT24CXX_WriteOneByte(uint16_t WriteAddr, uint8_t DataToWrite)
{
    IIC_Start();
    if (AT24C02 > AT24C16)
    {
        IIC_Send_Byte(0XA0); //д/Sending Write Commands
        IIC_Wait_Ack();
        IIC_Wait_Ack();
        IIC_Send_Byte(WriteAddr >> 8); //͸ߵַ/Send high address
    }
    else
    {
        IIC_Send_Byte(0XA0 + ((WriteAddr / 256) << 1)); //ַ0XA0,д/Send device address 0XA0 and write data
    }
    IIC_Wait_Ack();
    IIC_Send_Byte(WriteAddr % 256); //͵͵ַ /Send low address
    IIC_Wait_Ack();
    IIC_Send_Byte(DataToWrite); //ֽ /Send byte
    IIC_Wait_Ack();
    IIC_Stop(); //һֹͣ /Produces a stop condition
    delay_us(1000);
}

/**
  * @name   AT24CXX_WriteLenByte
  * @brief  EN: The specified address in AT24CXX starts writing data of length Len.
  *         CN: AT24CXXַָʼд볤ΪLenݡ
  * @param  WriteAddr:
  *         EN: Destination address of data to be written.
  *         CN: дݵĿĵַ
  * @param  DataToWrite:
  *         EN: The data to be written.
  *         CN: Ҫдݡ
  * @param  Len:
  *         EN: The length of data to be written is 2,4.
  *         CN: Ҫдݵĳ2,4
  * @retval None
  */
void AT24CXX_WriteLenByte(uint16_t WriteAddr, uint32_t DataToWrite, uint8_t Len)
{
    uint8_t t;
    for (t = 0; t < Len; t++)
    {
        AT24CXX_WriteOneByte(WriteAddr + t, (DataToWrite >> (8 * t)) & 0xff);
    }
}

/**
  * @name   AT24CXX_ReadLenByte
  * @brief  EN: The specified address in AT24CXX starts reading data of length Len.
  *         CN: AT24CXXַָʼΪLenݡ
  * @param  ReadAddr:
  *         EN: Address to start reading.
  *         CN: ʼĵַ
  * @param  Len:
  *         EN: The length of the data to be read is 2,4.
  *         CN: Ҫݵĳ2,4
  * @retval uint32_t
  *         EN: Data read.
  *         CN: ݡ
  */
uint32_t AT24CXX_ReadLenByte(uint16_t ReadAddr, uint8_t Len)
{
    uint8_t t;
    uint32_t temp = 0;
    for (t = 0; t < Len; t++)
    {
        temp <<= 8;
        temp += AT24CXX_ReadOneByte(ReadAddr + Len - t - 1);
    }
    return temp;
}

/**
  * @name   AT24CXX_Check
  * @brief  EN: Check whether the AT24CXX is normal(The last address of 24XX (255) is used here to store the token word.If using other 24C series, this address should be changed).
  *         CN: AT24CXXǷ(24XXһַ(255)洢־.24Cϵ,ַҪ޸)
  * @retval uint8_t
  *         EN: Status of AT24CXX.
  *         CN: AT24CXX ״̬
  */
uint8_t AT24CXX_Check(void)
{
    uint8_t temp;
    temp = AT24CXX_ReadOneByte(255); //ÿοдAT24CXX Avoid writing AT24CXX every time you boot up
    if (temp == 0X55)
        return 0;
    else //ųһγʼ Exclude the first initialization
    {
        AT24CXX_WriteOneByte(255, 0X55);
        temp = AT24CXX_ReadOneByte(255);
        if (temp == 0X55)
            return 0;
    }
    return 1;
}

/**
  * @name   AT24CXX_Read
  * @brief  EN: The specified address in AT24CXX starts to read a specified amount of data.
  *         CN: AT24CXXַָʼָݡ
  * @param  ReadAddr:
  *         EN: Address to start reading for 24c02  is 0~255.
  *         CN: ʼĵַ 24c02Ϊ0~255
  * @param  pBuffer:
  *         EN: Data array starting address.
  *         CN: ׵ַ
  * @param  NumToRead:
  *         EN: The number of data to be read out.
  *         CN: Ҫݵĸ,4
  * @retval None
  */
void AT24CXX_Read(uint16_t ReadAddr, uint8_t *pBuffer, uint16_t NumToRead)
{
    while (NumToRead)
    {
        *pBuffer++ = AT24CXX_ReadOneByte(ReadAddr++);
        NumToRead--;
    }
}

/**
  * @name   AT24CXX_Write
  * @brief  EN: The specified address in AT24CXX starts writing a specified amount of data.
  *         CN: AT24CXXַָʼдָݡ
  * @param  WriteAddr:
  *         EN: Address to start writing for 24c02 is 0~255.
  *         CN: ʼдĵַ 24c02Ϊ0~255
  * @param  pBuffer:
  *         EN: Data array starting address.
  *         CN: ׵ַ
  * @param  NumToWrite:
  *         EN: The number of data to be written.
  *         CN: Ҫдݵĸ
  * @retval None
  */
void AT24CXX_Write(uint16_t WriteAddr, uint8_t *pBuffer, uint16_t NumToWrite)
{
    while (NumToWrite--)
    {
        AT24CXX_WriteOneByte(WriteAddr, *pBuffer);
        WriteAddr++;
        pBuffer++;
    }
}
