/**
  ******************************************************************************
  * @file    lcm32f06x_tim.h 
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   Header for tim.c module
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __lCM32F06X_TIM_H__
#define __lCM32F06X_TIM_H__

/* Includes ------------------------------------------------------------------*/
#include "lcm32f06x.h"

#ifdef __cplusplus
  extern "C" {
#endif

typedef struct
{
  uint32_t TIM_Prescaler;        /*!< EN:Prescaler value.
                                      CN:ԤƵֵ
                                      @Register:TIMx_PSC @Bits:[15:0]PSC
                                      @range 0x0000 - 0xFFFF
                                      @default 0 */

  uint32_t TIM_CounterMode;      /*!< EN:Counter mode selection.
                                      CN:ģʽѡ
                                      @Register:TIMx_CR1 @Bits:[6:5]CMS [4]DIR
                                      @default TIM_CounterMode_Up */

  uint32_t TIM_Period;           /*!< EN:Auto-reload value.
                                      CN:ԶװؼĴֵ
                                      @Register:TIMx_ARR @Bits:[15:0]ARR
                                      @range 0x0000 - 0xFFFF
                                      @default 0xFFFF */

  uint32_t TIM_ClockDivision;    /*!< EN:The clock division ratio between the timer clock (CK_INT) frequency and the 
                                         dead-time/sampling clock (CK_DTS).
                                      CN:CK_INT ʱӺ/ʱӣCK_DTSķƵϵ
                                      @Register:TIMx_CR1 @Bits:[9:8]CKD
                                      @default TIM_CKD_DIV1 */

  uint8_t TIM_RepetitionCounter; /*!< EN:Repetition counter value
                                      CN:ظֵ
                                      @Register:TIMx_RCR @Bits:[7:0]REP
                                      @range 0x00 - 0xFF
                                      @default 0 
                                      @note This parameter is valid only for TIM1 */
} TIM_TimeBaseInitTypeDef;

typedef struct
{
  uint32_t TIM_OCMode;        /*!< EN: TIM Output Compare mode selection.
                                   CN: TIMȽģʽѡ
                                   @Register:TIMx_CCMRx @Bits:[6:4]OCxM
                                   @default TIM_OCMode_Timing */

  uint32_t TIM_OutputState;   /*!< EN: Specifies the TIM Output Compare state.
                                   CN: ָTIMȽ״̬
                                   @Register:TIMx_CCER @Bit:[0]CCxE
                                   @default TIM_OutputState_Disable*/

  uint32_t TIM_OutputNState;  /*!< EN: Specifies the TIM complementary Output Compare state.
                                   CN: ָTIMȽ״̬
                                   @Register:TIMx_CCER @Bit:[2]CCxNE
                                   @default TIM_OutputNState_Disable*/

  uint32_t TIM_Pulse;         /*!< EN: Specifies the pulse value to be loaded into the Capture Compare Register. 
                                   CN: ָҪصȽϼĴеֵ
                                   @Register:TIMx_CCRx @Bits:[15:0]CCRx
                                   @range  0x0000 - 0xFFFF  
                                   @default 0*/

  uint32_t TIM_OCPolarity;    /*!< EN: Specifies the output polarity.
                                   CN: ָԡ
                                   @Register:TIMx_CCER @Bit:[1]CCxP
                                   @default TIM_OCPolarity_High*/

  uint32_t TIM_OCNPolarity;   /*!< EN: Specifies the complementary output polarity.
                                   CN: ָԡ
                                   @Register:TIMx_CCER @Bit:[3]CCxNP
                                   @default TIM_OCNPolarity_High*/

  uint32_t TIM_OCIdleState;   /*!< EN: Specifies the TIM Output Compare pin state during Idle state.
                                   CN: ָ״̬TIMȽŵ״̬
                                   @Register:TIMx_CR2 @Bit:[8]OISx
                                   @default TIM_OCIdleState_Reset */

  uint32_t TIM_OCNIdleState; /*!< EN: Specifies the TIM Output Compare pin state during Idle state.
                                  CN: ָ״̬TIMȽŵ״̬
                                  @Register:TIMx_CR2 @Bit:[9]OISxN
                                  @default TIM_OCNIdleState_Reset */
} TIM_OCInitTypeDef;

typedef struct
{
  uint32_t TIM_Channel;      /*!< EN: Specifies the TIM channel.
                                  CN: ָTIM ͨ
                                  @default TIM_Channel_1 */

  uint32_t TIM_ICPolarity;   /*!< EN: Specifies the active edge of the input signal.
                                  CN: ָźŵļԡ
                                  @Register:TIMx_CCER @Bit:[1]CCxP
                                  @default TIM_ICPolarity_Rising */

  uint32_t TIM_ICSelection;  /*!< EN: Specifies the input.
                                  CN: ָ롣
                                  @Register:TIMx_CCMRx @Bits:[1:0]CCxS
                                  @default TIM_ICSelection_DirectTI */

  uint32_t TIM_ICPrescaler;  /*!< EN: Specifies the Input Capture Prescaler.
                                  CN: ָ벶ԤƵ
                                  @Register:TIMx_CCMRx @Bits:[3:2]ICxPSC
                                  @default TIM_ICPSC_DIV1 */

  uint32_t TIM_ICFilter;     /*!< EN: Specifies the TIM channel.
                                  CN: ָ벶˲ֵ
                                  @Register:TIMx_CCMRx @Bits:[7:4]ICxF
                                  @range 0x0 - 0xf
                                  @default 0 */
} TIM_ICInitTypeDef;

typedef struct
{

  uint32_t TIM_OSSRState;        /*!< EN: Specifies the Off-State selection used in Run mode.
                                      CN: ָģʽʹõĹر״̬ѡ
                                      @Register:TIMx_BDTR @Bit:[11]OSSR
                                      @default TIM_OSSRState_Disable */

  uint32_t TIM_OSSIState;        /*!< EN: Specifies the Off-State used in Idle state.
                                      CN: ָڿ״̬ʹõĹر״̬
                                      @Register:TIMx_BDTR @Bit:[10]OSSI
                                      @default TIM_OSSIState_Disable */

  uint32_t TIM_LOCKLevel;        /*!< EN: Specifies the LOCK level parameters.
                                      CN: ָLOCKĲ
                                      @Register:TIMx_BDTR @Bits:[9:8]LOCK
                                      @default TIM_LOCKLevel_OFF */

  uint32_t TIM_DeadTime;         /*!< EN: Specifies the delay time between the switching-off and the switching-on of the outputs.
                                      CN: ָĹرպʹ֮ӳʱ䡣
                                      @Register:TIMx_BDTR @Bits:[7:0]DTG
                                      @range 0x00 - 0xFF
                                      @default 0 */

  uint16_t TIM_Break;           /*!< EN: Specifies whether the TIM Break1 input is enabled or not.
                                      CN: ָTIM Break Ƿá
                                      @Register:TIMx_BDTR @Bit:[12]BKE
                                      @default TIM_Break_Disable */

  uint16_t TIM_BreakPolarity;   /*!< EN: Specifies the TIM Break2 Input pin polarity.
                                      CN: ָTIM Break żԡ
                                      @Register:TIMx_BDTR @Bit:[13]BKP
                                      @default TIM_BreakPolarity_Low */

  uint16_t TIM_AutomaticOutput; /*!< EN: Specifies whether the TIM Automatic1 Output feature is enabled or not.
                                      CN: ָTIMԶ1Ƿá
                                      @Register:TIMx_BDTR @Bit:[14]AOE
                                      @default TIM_AutomaticOutput_Disable */

  uint16_t TIM_Break2;           /*!< EN: Specifies whether the TIM Break input is enabled or not.
                                      CN: ָTIM Break2 Ƿá
                                      @Register:TIMx_BDTR @Bit:[24]BK2E
                                      @default TIM_Break_Disable */

  uint16_t TIM_Break2Polarity;   /*!< EN: Specifies the TIM Break Input pin polarity.
                                      CN: ָTIM Break2 żԡ
                                      @Register:TIMx_BDTR @Bit:[25]BK2P
                                      @default TIM_BreakPolarity_Low */

  uint16_t TIM_AutomaticOutput2; /*!< EN: Specifies whether the TIM Automatic2 Output feature is enabled or not.
                                      CN: ָTIMԶ2Ƿá
                                      @Register:TIMx_BDTR @Bit:[26]AO2E
                                      @default TIM_AutomaticOutput_Disable */

} TIM_BDTRInitTypeDef;


/* @defgroup TIM_CounterMode */
#define TIM_CounterMode_Up                    ((uint32_t)0x00000000) /*@EN: The counter counts up mode @CN:ϼģʽ*/
#define TIM_CounterMode_Down                  ((uint32_t)0x00000010) /*@EN: The counter counts down mode @CN:¼ģʽ*/
#define TIM_CounterMode_CenterAligned1        ((uint32_t)0x00000020) /*@EN:Center-aligned mode 1 @CN:ģʽ 1*/
#define TIM_CounterMode_CenterAligned2        ((uint32_t)0x00000040) /*@EN:Center-aligned mode 2 @CN:ģʽ 2*/
#define TIM_CounterMode_CenterAligned3        ((uint32_t)0x00000060) /*@EN:Center-aligned mode 3 @CN:ģʽ 3*/
#define IS_TIM_COUNTER_MODE(MODE)             (((MODE) == TIM_CounterMode_Up) ||             \
                                               ((MODE) == TIM_CounterMode_Down) ||           \
                                               ((MODE) == TIM_CounterMode_CenterAligned1) || \
                                               ((MODE) == TIM_CounterMode_CenterAligned2) || \
                                               ((MODE) == TIM_CounterMode_CenterAligned3))
/* @end */

/* @defgroup TIM_ClockDivision */
#define TIM_CKD_DIV1                          ((uint32_t)0x00000000) /*@EN:1 division frequency @CN:1Ƶ*/
#define TIM_CKD_DIV2                          ((uint32_t)0x00000100) /*@EN:2 division frequency @CN:2Ƶ*/
#define TIM_CKD_DIV4                          ((uint32_t)0x00000200) /*@EN:4 division frequency @CN:4Ƶ*/
#define IS_TIM_CKD_DIV(DIV)                   (((DIV) == TIM_CKD_DIV1) || \
                                               ((DIV) == TIM_CKD_DIV2) || \
                                               ((DIV) == TIM_CKD_DIV4))
/* @end */

/* @defgroup TIM_OCMode */
#define TIM_OCMode_Timing                     ((uint32_t)0x00000000) /*@EN:Frozen @CN:*/
#define TIM_OCMode_Active                     ((uint32_t)0x00000010) /*@EN:Set channel1 to active level on match @CN:ƥʱͨ1ΪЧƽ*/
#define TIM_OCMode_Inactive                   ((uint32_t)0x00000020) /*@EN:Set channel1 to inactive level on match @CN:ƥʱͨ1ΪЧƽ*/
#define TIM_OCMode_Toggle                     ((uint32_t)0x00000030) /*@EN:Toggle @CN: ת*/
#define TIM_OCMode_PWM1                       ((uint32_t)0x00000060) /*@EN:PWM mode1 @CN:PWMģʽ1*/
#define TIM_OCMode_PWM2                       ((uint32_t)0x00000070) /*@EN:PWM mode2 @CN:PWMģʽ2*/
#define IS_TIM_OC_MODE(MODE)                  (((MODE) == TIM_OCMode_Timing) ||   \
                                               ((MODE) == TIM_OCMode_Active) ||   \
                                               ((MODE) == TIM_OCMode_Inactive) || \
                                               ((MODE) == TIM_OCMode_Toggle) ||   \
                                               ((MODE) == TIM_OCMode_PWM1) ||     \
                                               ((MODE) == TIM_OCMode_PWM2))
#define IS_TIM_OCM(MODE)                      (((MODE) == TIM_OCMode_Timing) ||       \
                                               ((MODE) == TIM_OCMode_Active) ||       \
                                               ((MODE) == TIM_OCMode_Inactive) ||     \
                                               ((MODE) == TIM_OCMode_Toggle) ||       \
                                               ((MODE) == TIM_OCMode_PWM1) ||         \
                                               ((MODE) == TIM_OCMode_PWM2) ||         \
                                               ((MODE) == TIM_ForcedAction_Active) || \
                                               ((MODE) == TIM_ForcedAction_InActive))
/* @end */

/* @defgroup TIM_OutputState */
#define TIM_OutputState_Disable               ((uint32_t)0x00000000) /*@EN:Disable @CN:ʧ*/
#define TIM_OutputState_Enable                ((uint32_t)0x00000001) /*@EN:Enable @CN:ʹ*/
#define IS_TIM_OUTPUT_STATE(STATE)            (((STATE) == TIM_OutputState_Disable) || \
                                               ((STATE) == TIM_OutputState_Enable))
/* @end */

/* @defgroup TIM_OutputNState */
#define TIM_OutputNState_Disable              ((uint32_t)0x00000000) /*@EN:Disable @CN:ʧ*/
#define TIM_OutputNState_Enable               ((uint32_t)0x00000004) /*@EN:Enable @CN:ʹ*/
#define IS_TIM_OUTPUTN_STATE(STATE)           (((STATE) == TIM_OutputNState_Disable) || \
                                               ((STATE) == TIM_OutputNState_Enable))
/* @end */

/* @defgroup TIM_OCPolarity */
#define TIM_OCPolarity_High                   ((uint32_t)0x00000000) /*@EN:High @CN:ߵƽ*/
#define TIM_OCPolarity_Low                    ((uint32_t)0x00000002) /*@EN:Low @CN:͵ƽ*/
#define IS_TIM_OC_POLARITY(POLARITY)          (((POLARITY) == TIM_OCPolarity_High) || \
                                               ((POLARITY) == TIM_OCPolarity_Low))
/* @end */

/* @defgroup TIM_OCNPolarity */
#define TIM_OCNPolarity_High                  ((uint32_t)0x00000000) /*@EN:High @CN:ߵƽ*/
#define TIM_OCNPolarity_Low                   ((uint32_t)0x00000008) /*@EN:Low @CN:͵ƽ*/
#define IS_TIM_OCN_POLARITY(POLARITY)         (((POLARITY) == TIM_OCNPolarity_High) || \
                                               ((POLARITY) == TIM_OCNPolarity_Low))
/* @end */

/* @defgroup TIM_OCIdleState */
#define TIM_OCIdleState_Set                   ((uint32_t)0x00000100) /*@EN:Set @CN:λ*/
#define TIM_OCIdleState_Reset                 ((uint32_t)0x00000000) /*@EN:Reset @CN:λ*/
#define IS_TIM_OCIDLE_STATE(STATE)            (((STATE) == TIM_OCIdleState_Set) || \
                                               ((STATE) == TIM_OCIdleState_Reset))
/* @end */

/* @defgroup TIM_OCNIdleState */
#define TIM_OCNIdleState_Set                  ((uint32_t)0x00000200) /*@EN:Set @CN:λ*/
#define TIM_OCNIdleState_Reset                ((uint32_t)0x00000000) /*@EN:Reset @CN:λ*/
#define IS_TIM_OCNIDLE_STATE(STATE)           (((STATE) == TIM_OCNIdleState_Set) || \
                                               ((STATE) == TIM_OCNIdleState_Reset))
/* @end */

/* @defgroup TIM_Channel */
#define TIM_Channel_1                         ((uint32_t)0x00000000) /*@EN: TIM Channel 1 @CN: TIM ͨ1 */
#define TIM_Channel_2                         ((uint32_t)0x00000004) /*@EN: TIM Channel 2 @CN: TIM ͨ2 */
#define TIM_Channel_3                         ((uint32_t)0x00000008) /*@EN: TIM Channel 3 @CN: TIM ͨ3 */
#define TIM_Channel_4                         ((uint32_t)0x0000000C) /*@EN: TIM Channel 4 @CN: TIM ͨ4 */
#define TIM_Channel_5                         ((uint32_t)0x00000010) /*@EN: TIM Channel 5 @CN: TIM ͨ5 */

#define IS_TIM_CHANNEL(CHANNEL)               (((CHANNEL) == TIM_Channel_1) || \
                                               ((CHANNEL) == TIM_Channel_2) || \
                                               ((CHANNEL) == TIM_Channel_3) || \
                                               ((CHANNEL) == TIM_Channel_4) || \
                                               ((CHANNEL) == TIM_Channel_5))
#define IS_TIM_COMPLEMENTARY_CHANNEL(CHANNEL) (((CHANNEL) == TIM_Channel_1) || \
                                               ((CHANNEL) == TIM_Channel_2) || \
                                               ((CHANNEL) == TIM_Channel_3) || \
                                               ((CHANNEL) == TIM_Channel_4))
#define IS_TIM_PWMI_CHANNEL(CHANNEL)          (((CHANNEL) == TIM_Channel_1) || \
                                               ((CHANNEL) == TIM_Channel_2))
/* @end */

/* @defgroup TIM_ICPolarity */
#define TIM_ICPolarity_Rising                 ((uint32_t)0x00000000) /*@EN:Noninverted, it is active on TIiFP1 rising edge @CN:źŲ࣬TIiFP1 Ч*/
#define TIM_ICPolarity_Falling                ((uint32_t)0x00000002) /*@EN:Inverted, it is active on TIiFP1 falling edge @CN:źŷ࣬TIiFP1 ½Ч*/
#define TIM_ICPolarity_BothEdge               ((uint32_t)0x0000000A) /*@EN:Noninverted, it is active on both edges of TIiFP1 @CN:źŲ࣬TIiFP1 غ½ؾЧ*/
#define IS_TIM_IC_POLARITY(POLARITY)          (((POLARITY) == TIM_ICPolarity_Rising) ||  \
                                               ((POLARITY) == TIM_ICPolarity_Falling) || \
                                               ((POLARITY) == TIM_ICPolarity_BothEdge))
/* @end */

/* @defgroup TIM_ICSelection */
#define TIM_ICSelection_DirectTI              ((uint32_t)0x00000001) /*@EN:Direct mapping @CN:ֱӳ*/
#define TIM_ICSelection_IndirectTI            ((uint32_t)0x00000002) /*@EN:Indirect  mapping @CN:ӳ*/
#define TIM_ICSelection_TRC                   ((uint32_t)0x00000003) /*@EN:Map to TRC @CN:ӳTRC*/
#define IS_TIM_IC_SELECTION(SELECTION)        (((SELECTION) == TIM_ICSelection_DirectTI) ||   \
                                               ((SELECTION) == TIM_ICSelection_IndirectTI) || \
                                               ((SELECTION) == TIM_ICSelection_TRC))
/* @end */

/* @defgroup TIM_ICPrescaler */
#define TIM_ICPSC_DIV1                        ((uint32_t)0x00000000) /*@EN:Capture performed each time an edge is detected on the capture input @CN:ÿ 1 ЧΪһ */
#define TIM_ICPSC_DIV2                        ((uint32_t)0x00000004) /*@EN:Capture performed once every 2 events @CN:ÿ 2 ЧΪһ*/
#define TIM_ICPSC_DIV4                        ((uint32_t)0x00000008) /*@EN:Capture performed once every 4 events @CN:ÿ 4 ЧΪһ*/
#define TIM_ICPSC_DIV8                        ((uint32_t)0x0000000C) /*@EN:Capture performed once every 8 events @CN:ÿ 8 ЧΪһ*/
#define IS_TIM_IC_PRESCALER(PRESCALER)        (((PRESCALER) == TIM_ICPSC_DIV1) || \
                                               ((PRESCALER) == TIM_ICPSC_DIV2) || \
                                               ((PRESCALER) == TIM_ICPSC_DIV4) || \
                                               ((PRESCALER) == TIM_ICPSC_DIV8))
/* @end */

/* @defgroup TIM_OSSRState */
#define TIM_OSSRState_Enable                  ((uint32_t)0x00000800) /*@EN:Enable @CN:ʹ*/
#define TIM_OSSRState_Disable                 ((uint32_t)0x00000000) /*@EN:Disable @CN:ʧ*/
#define IS_TIM_OSSR_STATE(STATE)              (((STATE) == TIM_OSSRState_Enable) || \
                                               ((STATE) == TIM_OSSRState_Disable))
/* @end */

/* @defgroup TIM_OSSIState */
#define TIM_OSSIState_Enable                  ((uint32_t)0x00000400) /*@EN:Enable @CN:ʹ*/
#define TIM_OSSIState_Disable                 ((uint32_t)0x00000000) /*@EN:Disable @CN:ʧ*/
#define IS_TIM_OSSI_STATE(STATE)              (((STATE) == TIM_OSSIState_Enable) || \
                                               ((STATE) == TIM_OSSIState_Disable))
/* @end */

/* @defgroup TIM_LOCKLevel */
#define TIM_LOCKLevel_OFF                     ((uint32_t)0x00000000) /*@EN:LOCK OFF - No bit is write protected @CN:رգĴд*/
#define TIM_LOCKLevel_1                       ((uint32_t)0x00000100) /*@EN:LOCK Level 1 @CN: 1*/
#define TIM_LOCKLevel_2                       ((uint32_t)0x00000200) /*@EN:LOCK Level 2 @CN: 2*/
#define TIM_LOCKLevel_3                       ((uint32_t)0x00000300) /*@EN:LOCK Level 3 @CN: 3*/
#define IS_TIM_LOCK_LEVEL(LEVEL)              (((LEVEL) == TIM_LOCKLevel_OFF) || \
                                               ((LEVEL) == TIM_LOCKLevel_1) ||   \
                                               ((LEVEL) == TIM_LOCKLevel_2) ||   \
                                               ((LEVEL) == TIM_LOCKLevel_3))
/* @end */

/* @defgroup TIM_Break1 or TIM_Break2 */
#define TIM_Break_Enable                      ((uint32_t)0x00001000) /*@EN:Enable @CN:ʹ*/
#define TIM_Break_Disable                     ((uint32_t)0x00000000) /*@EN:Disable @CN:ʧ*/
#define IS_TIM_BREAK_STATE(STATE)             (((STATE) == TIM_Break_Enable) || \
                                               ((STATE) == TIM_Break_Disable))
/* @end */

/* @defgroup TIM_Break1Polarity or TIM_Break2Polarity */
#define TIM_BreakPolarity_Low                 ((uint32_t)0x00000000) /*@EN:Low @CN:͵ƽ*/
#define TIM_BreakPolarity_High                ((uint32_t)0x00002000) /*@EN:High @CN:ߵƽ*/
#define IS_TIM_BREAK_POLARITY(POLARITY)       (((POLARITY) == TIM_BreakPolarity_Low) || \
                                               ((POLARITY) == TIM_BreakPolarity_High))
/* @end */

/* @defgroup TIM_AutomaticOutput1 or TIM_AutomaticOutput2 */
#define TIM_AutomaticOutput_Enable            ((uint32_t)0x00004000) /*@EN:Enable @CN:ʹ*/
#define TIM_AutomaticOutput_Disable           ((uint32_t)0x00000000) /*@EN:Disable @CN:ʧ*/
#define IS_TIM_AUTOMATIC_OUTPUT_STATE(STATE)  (((STATE) == TIM_AutomaticOutput_Enable) || \
                                               ((STATE) == TIM_AutomaticOutput_Disable))
/* @end */

/* @defgroup TIM_PSCReloadMode */
#define TIM_PSCReloadMode_Update              ((uint32_t)0x00000000) /*@EN:No action @CN:޶*/
#define TIM_PSCReloadMode_Immediate           ((uint32_t)0x00000001) /*@EN:Reinitialize the counter and generate an update event @CN:³ʼһ¼*/
#define IS_TIM_PRESCALER_RELOAD(RELOAD)       (((RELOAD) == TIM_PSCReloadMode_Update) || \
                                               ((RELOAD) == TIM_PSCReloadMode_Immediate))
/* @end */

/* @defgroup TIM_UpdateSource */
#define TIM_UpdateSource_Global               ((uint32_t)0x00000000) /*@EN: Source of update is the counter overflow/underflow or the setting of UG bit, or an update generation through the slave mode controller.
                                                                       @CN: µԴǼ/UGλãͨģʽɸ */
#define TIM_UpdateSource_Regular              ((uint32_t)0x00000001) /*@EN: Source of update is counter overflow/underflow @CN: µԴǼ/ */
#define IS_TIM_UPDATE_SOURCE(SOURCE)          (((SOURCE) == TIM_UpdateSource_Global) || \
                                               ((SOURCE) == TIM_UpdateSource_Regular))
/* @end */

/* @defgroup TIM_OPMode */
#define TIM_OPMode_Single                     ((uint32_t)0x00000008) /*@EN: Counter stops when the next update event occurs @CN: ڷһθ¼ʱֹͣ */
#define TIM_OPMode_Repetitive                 ((uint32_t)0x00000000) /*@EN: When an update event occurs, the counter does not stop @CN: ڷ¼ʱֹͣ */
#define IS_TIM_OPM_MODE(MODE)                 (((MODE) == TIM_OPMode_Single) || \
                                               ((MODE) == TIM_OPMode_Repetitive))
/* @end */

/* @defgroup TIM_ForcedAction */
#define TIM_ForcedAction_Active               ((uint32_t)0x00000050) /*@EN: Force active level on OCiREF @CN: OCiREFϵǿƼ */
#define TIM_ForcedAction_InActive             ((uint32_t)0x00000040) /*@EN: Force inactive level on OCiREF. @CN: ǿOCiREFڷǻ */
#define IS_TIM_FORCED_ACTION(ACTION)          (((ACTION) == TIM_ForcedAction_Active) || \
                                               ((ACTION) == TIM_ForcedAction_InActive))
/* @end */

/* @defgroup TIM_OCPreload */
#define TIM_OCPreload_Enable                  ((uint32_t)0x00000008) /*@EN: TIM output compare Preload enable @CN: TIMȽԤװ */
#define TIM_OCPreload_Disable                 ((uint32_t)0x00000000) /*@EN: TIM output compare Preload disable @CN: TIMȽԤװؽ */
#define IS_TIM_OCPRELOAD_STATE(STATE)         (((STATE) == TIM_OCPreload_Enable) || \
                                               ((STATE) == TIM_OCPreload_Disable))
/* @end */

/* @defgroup TIM_OCFast */
#define TIM_OCFast_Enable                     ((uint32_t)0x00000004) /*@EN: TIM output compare fast enable @CN: TIMȽϿ */
#define TIM_OCFast_Disable                    ((uint32_t)0x00000000) /*@EN: TIM output compare fast disable @CN: TIMȽϿٽ */
#define IS_TIM_OCFAST_STATE(STATE)            (((STATE) == TIM_OCFast_Enable) || \
                                               ((STATE) == TIM_OCFast_Disable))
/* @end */

/* @defgroup TIM_OCClear */
#define TIM_OCClear_Enable                    ((uint32_t)0x00000080) /*@EN: TIM Output clear enable @CN: TIMʹ */
#define TIM_OCClear_Disable                   ((uint32_t)0x00000000) /*@EN: TIM Output clear disable @CN: TIM */
#define IS_TIM_OCCLEAR_STATE(STATE)           (((STATE) == TIM_OCClear_Enable) || \
                                               ((STATE) == TIM_OCClear_Disable))
/* @end */

/* @defgroup TIM_OCReferenceClear */
#define TIM_OCReferenceClear_ETRF             ((uint32_t)0x00000008) /*@EN: The internal OCreference clear input is connected to ETRF. @CN: ڲOCreferenceӵETRF */
#define TIM_OCReferenceClear_OCREFCLR         ((uint32_t)0x00000000) /*@EN: The internal OCreference clear input is connected to OCREF_CLR input @CN: ڲOCreferenceӵOCREF_CLR */
#define TIM_OCREFERENCECECLEAR_SOURCE(SOURCE) (((SOURCE) == TIM_OCReferenceClear_ETRF) || \
                                               ((SOURCE) == TIM_OCReferenceClear_OCREFCLR))
/* @end */

/* @defgroup TIM_CCx */
#define TIM_CCx_Enable                        ((uint32_t)0x00000001) /*@EN: Capture/Compare Enable @CN: /Ƚ ʹ */
#define TIM_CCx_Disable                       ((uint32_t)0x00000000) /*@EN: Capture/Compare Disable @CN: /Ƚ ʧ */
#define IS_TIM_CCX(CCX)                       (((CCX) == TIM_CCx_Enable) || \
                                               ((CCX) == TIM_CCx_Disable))
/* @end */

/* @defgroup TIM_CCxN */
#define TIM_CCxN_Enable                       ((uint32_t)0x00000004) /*@EN: Capture/Compare Complementary Enable @CN: /Ƚ  ʹ */
#define TIM_CCxN_Disable                      ((uint32_t)0x00000000) /*@EN: Capture/Compare Complementary Disable @CN: /Ƚ  ʧ */
#define IS_TIM_CCXN(CCXN)                     (((CCXN) == TIM_CCxN_Enable) || \
                                               ((CCXN) == TIM_CCxN_Disable))
/* @end */

/* @defgroup TIM_EventSource */
#define TIM_EventSource_Update                ((uint32_t)0x00000001) /*@EN: Generate update event @CN: ¼ */
#define TIM_EventSource_CC1                   ((uint32_t)0x00000002) /*@EN: Generate comparison 1 events @CN: Ƚ 1 ¼ */
#define TIM_EventSource_CC2                   ((uint32_t)0x00000004) /*@EN: Generate comparison 2 events @CN: Ƚ 2 ¼ */
#define TIM_EventSource_CC3                   ((uint32_t)0x00000008) /*@EN: Generate comparison 3 events @CN: Ƚ 3 ¼ */
#define TIM_EventSource_CC4                   ((uint32_t)0x00000010) /*@EN: Generate comparison 4 events @CN: Ƚ 4 ¼ */
#define TIM_EventSource_COM                   ((uint32_t)0x00000020) /*@EN: Generate COM event @CN:  ȽϿƸ ¼ */
#define TIM_EventSource_Trigger               ((uint32_t)0x00000040) /*@EN: Generate triger event @CN:  ¼ */
#define TIM_EventSource_Break                 ((uint32_t)0x00000080) /*@EN: Generate brake 1 event @CN: ɲ 1 ¼ */
#define TIM_EventSource_Break2                ((uint32_t)0x00000100) /*@EN: Generate brake 2 event @CN: ɲ 2 ¼ */
#define IS_TIM_EVENT_SOURCE(SOURCE)           ((((SOURCE) & (uint32_t)0x0000FF00) == 0x00000000) && ((SOURCE) != 0x00000000))
/* @end */

/* @defgroup TIM_FLAG */
#define TIM_FLAG_Update                       ((uint32_t)0x00000001) /*@EN: TIM update Flag @CN: TIM жϱ־ */
#define TIM_FLAG_CC1                          ((uint32_t)0x00000002) /*@EN: TIM Capture Compare 1 Flag @CN: TIM Ƚͨ1жϱ־ */
#define TIM_FLAG_CC2                          ((uint32_t)0x00000004) /*@EN: TIM Capture Compare 2 Flag @CN: TIM Ƚͨ2жϱ־ */
#define TIM_FLAG_CC3                          ((uint32_t)0x00000008) /*@EN: TIM Capture Compare 3 Flag @CN: TIM Ƚͨ3жϱ־ */
#define TIM_FLAG_CC4                          ((uint32_t)0x00000010) /*@EN: TIM Capture Compare 4 Flag @CN: TIM Ƚͨ4жϱ־ */
#define TIM_FLAG_COM                          ((uint32_t)0x00000020) /*@EN: TIM COM Flag @CN: TIM COMж */
#define TIM_FLAG_Trigger                      ((uint32_t)0x00000040) /*@EN: TIM Trigger Flag @CN: TIM жϱ־ */
#define TIM_FLAG_Break                        ((uint32_t)0x00000080) /*@EN: TIM Break Flag @CN: TIM ɲжϱ־ */
#define TIM_FLAG_Break2                       ((uint32_t)0x00000100) /*@EN: TIM Break2 Flag @CN: TIM ɲ2жϱ־ */
#define TIM_FLAG_CC1OF                        ((uint32_t)0x00000200) /*@EN: TIM Capture Compare 1 overcapture Flag @CN: Ƚͨ1Ƚϱ־ */
#define TIM_FLAG_CC2OF                        ((uint32_t)0x00000400) /*@EN: TIM Capture Compare 2 overcapture Flag @CN: Ƚͨ2Ƚϱ־ */
#define TIM_FLAG_CC3OF                        ((uint32_t)0x00000800) /*@EN: TIM Capture Compare 3 overcapture Flag @CN: Ƚͨ3Ƚϱ־ */
#define TIM_FLAG_CC4OF                        ((uint32_t)0x00001000) /*@EN: TIM Capture Compare 4 overcapture Flag @CN: Ƚͨ4Ƚϱ־ */
#define TIM_FLAG_CC5OF                        ((uint32_t)0x00002000) /*@EN: TIM Capture Compare 4 overcapture Flag @CN: Ƚͨ5Ƚϱ־ */
#define TIM_FLAG_CC5                          ((uint32_t)0x00010000) /*@EN: TIM Capture Compare 5 Flag @CN: TIM Ƚͨ5жϱ־ */
#define IS_TIM_GET_FLAG(FLAG)                 (((FLAG) == TIM_FLAG_Update) ||  \
                                               ((FLAG) == TIM_FLAG_CC1) ||     \
                                               ((FLAG) == TIM_FLAG_CC2) ||     \
                                               ((FLAG) == TIM_FLAG_CC3) ||     \
                                               ((FLAG) == TIM_FLAG_CC4) ||     \
                                               ((FLAG) == TIM_FLAG_CC5) ||     \
                                               ((FLAG) == TIM_FLAG_COM) ||     \
                                               ((FLAG) == TIM_FLAG_Trigger) || \
                                               ((FLAG) == TIM_FLAG_Break) ||   \
                                               ((FLAG) == TIM_FLAG_Break2) ||  \
                                               ((FLAG) == TIM_FLAG_CC1OF) ||   \
                                               ((FLAG) == TIM_FLAG_CC2OF) ||   \
                                               ((FLAG) == TIM_FLAG_CC3OF) ||   \
                                               ((FLAG) == TIM_FLAG_CC4OF) ||   \
                                               ((FLAG) == TIM_FLAG_CC5OF)) 

#define IS_TIM_CLEAR_FLAG(TIM_FLAG)           ((((TIM_FLAG) & (uint32_t)0x0000E100) == 0x00000000) && ((TIM_FLAG) != 0x00000000))
/* @end */

/* @defgroup TIM_IT */
#define TIM_IT_Update                         ((uint32_t)0x00000001) /*@EN: TIM1 update Interrupt source @CN: TIM1жԴ */
#define TIM_IT_CC1                            ((uint32_t)0x00000002) /*@EN: TIM Capture Compare 1 Interrupt source @CN: TIMȽ1жԴ */
#define TIM_IT_CC2                            ((uint32_t)0x00000004) /*@EN: TIM Capture Compare 2 Interrupt source @CN: TIMȽ2жԴ */
#define TIM_IT_CC3                            ((uint32_t)0x00000008) /*@EN: TIM Capture Compare 3 Interrupt source @CN: TIMȽ3жԴ */
#define TIM_IT_CC4                            ((uint32_t)0x00000010) /*@EN: TIM Capture Compare 4 Interrupt source @CN: TIMȽ4жԴ */
#define TIM_IT_COM                            ((uint32_t)0x00000020) /*@EN: TIM COM Interrupt source @CN: TIM COM жԴ */
#define TIM_IT_Trigger                        ((uint32_t)0x00000040) /*@EN: TIM Trigger Interrupt source @CN: TIMжԴ */
#define TIM_IT_Break                          ((uint32_t)0x00000080) /*@EN: TIM Break Interrupt source @CN: TIM break жԴ */
#define TIM_IT_DMA                            ((uint32_t)0x00000100) /*@EN: DMA request interrupt source for TIM updates @CN: TIM   dma   жԴ */
#define TIM_IT_CC5                            ((uint32_t)0x00010000) /*@EN: TIM Capture Compare 5 Interrupt source @CN: TIMȽ5жԴ */
#define IS_TIM_IT(IT)                         ((((IT) & (uint32_t)0x0000FF00) == 0x00000000) && ((IT) != 0x00000000))
#define IS_TIM_GET_IT(IT)                     (((IT) == TIM_IT_Update) ||  \
                                               ((IT) == TIM_IT_CC1) ||     \
                                               ((IT) == TIM_IT_CC2) ||     \
                                               ((IT) == TIM_IT_CC3) ||     \
                                               ((IT) == TIM_IT_CC4) ||     \
                                               ((IT) == TIM_IT_CC5) ||     \
                                               ((IT) == TIM_IT_COM) ||     \
                                               ((IT) == TIM_IT_Trigger) || \
                                               ((IT) == TIM_IT_Break) ||   \
                                               ((IT) == TIM_IT_DMA) ||     \)
/* @end */

/* @defgroup TIM_DMABurstLength */
#define TIM_DMABurstLength_1Transfer          ((uint32_t)0x00000000) /*@EN: DMA continuous transmission length of 1 byte @CN: DMA䳤 1ֽ */
#define TIM_DMABurstLength_2Transfers         ((uint32_t)0x00000100) /*@EN: DMA continuous transmission length of 2 byte @CN: DMA䳤 2ֽ */
#define TIM_DMABurstLength_3Transfers         ((uint32_t)0x00000200) /*@EN: DMA continuous transmission length of 3 byte @CN: DMA䳤 3ֽ */
#define TIM_DMABurstLength_4Transfers         ((uint32_t)0x00000300) /*@EN: DMA continuous transmission length of 4 byte @CN: DMA䳤 4ֽ */
#define TIM_DMABurstLength_5Transfers         ((uint32_t)0x00000400) /*@EN: DMA continuous transmission length of 5 byte @CN: DMA䳤 5ֽ */
#define TIM_DMABurstLength_6Transfers         ((uint32_t)0x00000500) /*@EN: DMA continuous transmission length of 6 byte @CN: DMA䳤 6ֽ */
#define TIM_DMABurstLength_7Transfers         ((uint32_t)0x00000600) /*@EN: DMA continuous transmission length of 7 byte @CN: DMA䳤 7ֽ */
#define TIM_DMABurstLength_8Transfers         ((uint32_t)0x00000700) /*@EN: DMA continuous transmission length of 8 byte @CN: DMA䳤 8ֽ */
#define TIM_DMABurstLength_9Transfers         ((uint32_t)0x00000800) /*@EN: DMA continuous transmission length of 9 byte @CN: DMA䳤 9ֽ */
#define TIM_DMABurstLength_10Transfers        ((uint32_t)0x00000900) /*@EN: DMA continuous transmission length of 10 byte @CN: DMA䳤 10ֽ */
#define TIM_DMABurstLength_11Transfers        ((uint32_t)0x00000A00) /*@EN: DMA continuous transmission length of 11 byte @CN: DMA䳤 11ֽ */
#define TIM_DMABurstLength_12Transfers        ((uint32_t)0x00000B00) /*@EN: DMA continuous transmission length of 12 byte @CN: DMA䳤 12ֽ */
#define TIM_DMABurstLength_13Transfers        ((uint32_t)0x00000C00) /*@EN: DMA continuous transmission length of 13 byte @CN: DMA䳤 13ֽ */
#define TIM_DMABurstLength_14Transfers        ((uint32_t)0x00000D00) /*@EN: DMA continuous transmission length of 14 byte @CN: DMA䳤 14ֽ */
#define TIM_DMABurstLength_15Transfers        ((uint32_t)0x00000E00) /*@EN: DMA continuous transmission length of 15 byte @CN: DMA䳤 15ֽ */
#define TIM_DMABurstLength_16Transfers        ((uint32_t)0x00000F00) /*@EN: DMA continuous transmission length of 16 byte @CN: DMA䳤 16ֽ */
#define TIM_DMABurstLength_17Transfers        ((uint32_t)0x00001000) /*@EN: DMA continuous transmission length of 17 byte @CN: DMA䳤 17ֽ */
#define TIM_DMABurstLength_18Transfers        ((uint32_t)0x00001100) /*@EN: DMA continuous transmission length of 18 byte @CN: DMA䳤 18ֽ */
#define IS_TIM_DMA_LENGTH(LENGTH)             (((LENGTH) == TIM_DMABurstLength_1Transfer) ||   \
                                               ((LENGTH) == TIM_DMABurstLength_2Transfers) ||  \
                                               ((LENGTH) == TIM_DMABurstLength_3Transfers) ||  \
                                               ((LENGTH) == TIM_DMABurstLength_4Transfers) ||  \
                                               ((LENGTH) == TIM_DMABurstLength_5Transfers) ||  \
                                               ((LENGTH) == TIM_DMABurstLength_6Transfers) ||  \
                                               ((LENGTH) == TIM_DMABurstLength_7Transfers) ||  \
                                               ((LENGTH) == TIM_DMABurstLength_8Transfers) ||  \
                                               ((LENGTH) == TIM_DMABurstLength_9Transfers) ||  \
                                               ((LENGTH) == TIM_DMABurstLength_10Transfers) || \
                                               ((LENGTH) == TIM_DMABurstLength_11Transfers) || \
                                               ((LENGTH) == TIM_DMABurstLength_12Transfers) || \
                                               ((LENGTH) == TIM_DMABurstLength_13Transfers) || \
                                               ((LENGTH) == TIM_DMABurstLength_14Transfers) || \
                                               ((LENGTH) == TIM_DMABurstLength_15Transfers) || \
                                               ((LENGTH) == TIM_DMABurstLength_16Transfers) || \
                                               ((LENGTH) == TIM_DMABurstLength_17Transfers) || \
                                               ((LENGTH) == TIM_DMABurstLength_18Transfers))
/* @end */

/* @defgroup TIM_DMABase */
#define TIM_DMABase_CR1                       ((uint32_t)0x00000000) /*@EN: DMA base address CR1   @CN: DMA ַ CR1   */
#define TIM_DMABase_CR2                       ((uint32_t)0x00000001) /*@EN: DMA base address CR2   @CN: DMA ַ CR2   */
#define TIM_DMABase_SMCR                      ((uint32_t)0x00000002) /*@EN: DMA base address SMCR  @CN: DMA ַ SMCR  */
#define TIM_DMABase_DIER                      ((uint32_t)0x00000003) /*@EN: DMA base address DIER  @CN: DMA ַ DIER  */
#define TIM_DMABase_SR                        ((uint32_t)0x00000004) /*@EN: DMA base address SR    @CN: DMA ַ SR    */
#define TIM_DMABase_EGR                       ((uint32_t)0x00000005) /*@EN: DMA base address EGR   @CN: DMA ַ EGR   */
#define TIM_DMABase_CCMR1                     ((uint32_t)0x00000006) /*@EN: DMA base address CCMR1 @CN: DMA ַ CCMR1 */
#define TIM_DMABase_CCMR2                     ((uint32_t)0x00000007) /*@EN: DMA base address CCMR2 @CN: DMA ַ CCMR2 */
#define TIM_DMABase_CCER                      ((uint32_t)0x00000008) /*@EN: DMA base address CCER  @CN: DMA ַ CCER  */
#define TIM_DMABase_CNT                       ((uint32_t)0x00000009) /*@EN: DMA base address CNT   @CN: DMA ַ CNT   */
#define TIM_DMABase_PSC                       ((uint32_t)0x0000000A) /*@EN: DMA base address PSC   @CN: DMA ַ PSC   */
#define TIM_DMABase_ARR                       ((uint32_t)0x0000000B) /*@EN: DMA base address ARR   @CN: DMA ַ ARR   */
#define TIM_DMABase_RCR                       ((uint32_t)0x0000000C) /*@EN: DMA base address RCR   @CN: DMA ַ RCR   */
#define TIM_DMABase_CCR1                      ((uint32_t)0x0000000D) /*@EN: DMA base address CCR1  @CN: DMA ַ CCR1  */
#define TIM_DMABase_CCR2                      ((uint32_t)0x0000000E) /*@EN: DMA base address CCR2  @CN: DMA ַ CCR2  */
#define TIM_DMABase_CCR3                      ((uint32_t)0x0000000F) /*@EN: DMA base address CCR3  @CN: DMA ַ CCR3  */
#define TIM_DMABase_CCR4                      ((uint32_t)0x00000010) /*@EN: DMA base address CCR4  @CN: DMA ַ CCR4  */
#define TIM_DMABase_BDTR                      ((uint32_t)0x00000011) /*@EN: DMA base address BDTR  @CN: DMA ַ BDTR  */
#define TIM_DMABase_DCR                       ((uint32_t)0x00000012) /*@EN: DMA base address DCR   @CN: DMA ַ DCR   */
#define TIM_DMABase_OR                        ((uint32_t)0x00000013) /*@EN: DMA base address OR    @CN: DMA ַ OR    */
#define TIM_DMABase_CCMR3                     ((uint32_t)0x00000014) /*@EN: DMA base address CCMR3 @CN: DMA ַ CCMR3 */
#define IS_TIM_DMA_BASE(BASE)                 (((BASE) == TIM_DMABase_CR1) ||   \
                                               ((BASE) == TIM_DMABase_CR2) ||   \
                                               ((BASE) == TIM_DMABase_SMCR) ||  \
                                               ((BASE) == TIM_DMABase_DIER) ||  \
                                               ((BASE) == TIM_DMABase_SR) ||    \
                                               ((BASE) == TIM_DMABase_EGR) ||   \
                                               ((BASE) == TIM_DMABase_CCMR1) || \
                                               ((BASE) == TIM_DMABase_CCMR2) || \
                                               ((BASE) == TIM_DMABase_CCMR3) || \
                                               ((BASE) == TIM_DMABase_CCER) ||  \
                                               ((BASE) == TIM_DMABase_CNT) ||   \
                                               ((BASE) == TIM_DMABase_PSC) ||   \
                                               ((BASE) == TIM_DMABase_ARR) ||   \
                                               ((BASE) == TIM_DMABase_RCR) ||   \
                                               ((BASE) == TIM_DMABase_CCR1) ||  \
                                               ((BASE) == TIM_DMABase_CCR2) ||  \
                                               ((BASE) == TIM_DMABase_CCR3) ||  \
                                               ((BASE) == TIM_DMABase_CCR4) ||  \
                                               ((BASE) == TIM_DMABase_BDTR) ||  \
                                               ((BASE) == TIM_DMABase_DCR) ||   \
                                               ((BASE) == TIM_DMABase_OR))
/* @end */
 
/* @defgroup TIM_DMASource */
#define TIM_DMA_Update                        ((uint32_t)0x00000100) /*@EN: TIM update Interrupt source @CN: TIMжԴ */
#define TIM_DMA_CC1                           ((uint32_t)0x00000200) /*@EN: TIM Capture Compare 1 DMA source @CN: TIMȽ1 DMAԴ */
#define TIM_DMA_CC2                           ((uint32_t)0x00000400) /*@EN: TIM Capture Compare 2 DMA source @CN: TIMȽ2 DMAԴ */
#define TIM_DMA_CC3                           ((uint32_t)0x00000800) /*@EN: TIM Capture Compare 3 DMA source @CN: TIMȽ3 DMAԴ */
#define TIM_DMA_CC4                           ((uint32_t)0x00001000) /*@EN: TIM Capture Compare 4 DMA source @CN: TIMȽ4 DMAԴ */
#define TIM_DMA_COM                           ((uint32_t)0x00002000) /*@EN: TIM COM DMA source @CN: TIM COM DMAԴ */
#define TIM_DMA_Trigger                       ((uint32_t)0x00004000) /*@EN: TIM Trigger DMA source @CN: TIMDMAԴ */
#define TIM_DMA_CC5                           ((uint32_t)0x00040000) /*@EN: TIM Capture Compare 4 DMA source @CN: TIMȽ5 DMAԴ */
#define IS_TIM_DMA_SOURCE(SOURCE)             ((((SOURCE) & (uint32_t)0x000900FF) == 0x00000000) && ((SOURCE) != 0x00000000))
/* @end */

/* @defgroup TIM_TIxExternalCLKSource */
#define TIM_TIxExternalCLK1Source_TI1ED       ((uint32_t)0x00000040) /*@EN: TI1 Edge Detector @CN: ڲ ITR0*/
#define TIM_TIxExternalCLK1Source_TI1         ((uint32_t)0x00000050) /*@EN: Filtered Timer Input 1 @CN: ڲ ITR0*/
#define TIM_TIxExternalCLK1Source_TI2         ((uint32_t)0x00000060) /*@EN: Filtered Timer Input 2 @CN: ڲ ITR0*/
/* @end */

/* @defgroup TIM_InputTriggerSource */
#define TIM_TS_ITR0                           ((uint32_t)0x00000000) /*@EN: Internal Trigger 0 @CN: ڲ ITR0*/
#define TIM_TS_ITR1                           ((uint32_t)0x00000010) /*@EN: Internal Trigger 1 @CN: ڲ ITR1*/
#define TIM_TS_ITR2                           ((uint32_t)0x00000020) /*@EN: Internal Trigger 2 @CN: ڲ ITR2*/
#define TIM_TS_ITR3                           ((uint32_t)0x00000030) /*@EN: Internal Trigger 3 @CN: ڲ ITR3*/
#define TIM_TS_TI1F_ED                        ((uint32_t)0x00000040) /*@EN: Edge detector of TI1 @CN: TI1 ıؼ*/
#define TIM_TS_TI1FP1                         ((uint32_t)0x00000050) /*@EN: Timer input 1 after filtering @CN: ˲Ķʱ 1*/
#define TIM_TS_TI2FP2                         ((uint32_t)0x00000060) /*@EN: Timer input 2 after filtering @CN: ˲Ķʱ 2*/
#define TIM_TS_ETRF                           ((uint32_t)0x00000070) /*@EN: External trigger input @CN: ⲿ*/
#define IS_TIM_TRIGGER_SELECTION(SELECTION)   (((SELECTION) == TIM_TS_ITR0) ||    \
                                               ((SELECTION) == TIM_TS_ITR1) ||    \
                                               ((SELECTION) == TIM_TS_ITR2) ||    \
                                               ((SELECTION) == TIM_TS_ITR3) ||    \
                                               ((SELECTION) == TIM_TS_TI1F_ED) || \
                                               ((SELECTION) == TIM_TS_TI1FP1) ||  \
                                               ((SELECTION) == TIM_TS_TI2FP2) ||  \
                                               ((SELECTION) == TIM_TS_ETRF))
/* @end */

/* @defgroup TIM_TRGOSource */
#define TIM_TRGOSource_Reset                  ((uint32_t)0x00000000) /*@EN: Reset (TRGO). @CN: λ (TRGO)*/
#define TIM_TRGOSource_Enable                 ((uint32_t)0x00000010) /*@EN: Enable (TRGO). @CN: ʹ (TRGO)*/
#define TIM_TRGOSource_Update                 ((uint32_t)0x00000020) /*@EN: Update (TRGO). @CN:  (TRGO)*/
#define TIM_TRGOSource_OC1                    ((uint32_t)0x00000030) /*@EN: Compare pulses (TRGO). @CN: Ƚ (TRGO)*/
#define TIM_TRGOSource_OC1Ref                 ((uint32_t)0x00000040) /*@EN: Comparison - OC1REF signal used as trigger output (TRGO). @CN: Ƚ - OC1REF źű (TRGO)*/
#define TIM_TRGOSource_OC2Ref                 ((uint32_t)0x00000050) /*@EN: Comparison - OC2REF signal used as trigger output (TRGO). @CN: Ƚ - OC2REF źű (TRGO)*/
#define TIM_TRGOSource_OC3Ref                 ((uint32_t)0x00000060) /*@EN: Comparison - OC3REF signal used as trigger output (TRGO). @CN: Ƚ - OC3REF źű (TRGO)*/
#define TIM_TRGOSource_OC4Ref                 ((uint32_t)0x00000070) /*@EN: Comparison - OC4REF signal used as trigger output (TRGO). @CN: Ƚ - OC4REF źű (TRGO)*/
#define IS_TIM_TRGO_SOURCE(SOURCE)            (((SOURCE) == TIM_TRGOSource_Reset) ||  \
                                               ((SOURCE) == TIM_TRGOSource_Enable) || \
                                               ((SOURCE) == TIM_TRGOSource_Update) || \
                                               ((SOURCE) == TIM_TRGOSource_OC1) ||    \
                                               ((SOURCE) == TIM_TRGOSource_OC1Ref) || \
                                               ((SOURCE) == TIM_TRGOSource_OC2Ref) || \
                                               ((SOURCE) == TIM_TRGOSource_OC3Ref) || \
                                               ((SOURCE) == TIM_TRGOSource_OC4Ref))
/* @end */

/* @defgroup TIM_SlaveMode */
#define TIM_SlaveMode_Reset                   ((uint32_t)0x00000004) /*@EN: Reset Mode @CN: λģʽ */
#define TIM_SlaveMode_Gated                   ((uint32_t)0x00000005) /*@EN: Gated Mode @CN: ſģʽ */
#define TIM_SlaveMode_Trigger                 ((uint32_t)0x00000006) /*@EN: Trigger Mode @CN: ģʽ */
#define TIM_SlaveMode_External1               ((uint32_t)0x00000007) /*@EN: External1 Mode @CN: ⲿʱģʽ1*/
#define IS_TIM_SLAVE_MODE(MODE)               (((MODE) == TIM_SlaveMode_Reset) ||   \
                                               ((MODE) == TIM_SlaveMode_Gated) ||   \
                                               ((MODE) == TIM_SlaveMode_Trigger) || \
                                               ((MODE) == TIM_SlaveMode_External1))
/* @end */

/* @defgroup TIM_MasterSlaveMode */
#define TIM_MasterSlaveMode_Enable            ((uint32_t)0x00000080) /*@EN: synchronization between the current timer and its slaves (through TRGO). @CN: ǰʱʱ֮ͬͨTRGO */
#define TIM_MasterSlaveMode_Disable           ((uint32_t)0x00000000) /*@EN: No effect (default). @CN: ãĬϣ */
#define IS_TIM_MSM_STATE(STATE)               (((STATE) == TIM_MasterSlaveMode_Enable) || \
                                               ((STATE) == TIM_MasterSlaveMode_Disable))
/* @end */

/* @defgroup TIM_ExtTRGPrescaler */
#define TIM_ExtTRGPSC_OFF                     ((uint32_t)0x00000000) /*@EN: ETRP Prescaler OFF. @CN: ETRPԤƵرա */
#define TIM_ExtTRGPSC_DIV2                    ((uint32_t)0x00001000) /*@EN: ETRP frequency divided by 2. @CN: ETRPƵʳ2 */
#define TIM_ExtTRGPSC_DIV4                    ((uint32_t)0x00002000) /*@EN: ETRP frequency divided by 4. @CN: ETRPƵʳ4 */
#define TIM_ExtTRGPSC_DIV8                    ((uint32_t)0x00003000) /*@EN: ETRP frequency divided by 8. @CN: ETRPƵʳ8 */
#define IS_TIM_EXT_PRESCALER(PRESCALER)       (((PRESCALER) == TIM_ExtTRGPSC_OFF) ||  \
                                               ((PRESCALER) == TIM_ExtTRGPSC_DIV2) || \
                                               ((PRESCALER) == TIM_ExtTRGPSC_DIV4) || \
                                               ((PRESCALER) == TIM_ExtTRGPSC_DIV8))
/* @end */

/* @defgroup TIM_ExtTRGPolarity */

#define TIM_ExtTRGPolarity_Inverted           ((uint32_t)0x00008000) /*@EN: ETR inversion, which means low level or effective falling edge. @CN: ETR ࣬͵ƽ½Ч */
#define TIM_ExtTRGPolarity_NonInverted        ((uint32_t)0x00000000) /*@EN: ETR does not invert, meaning high level or rising edge is effective. @CN: ETR ࣬ߵƽЧ */
#define IS_TIM_EXT_POLARITY(POLARITY)         (((POLARITY) == TIM_ExtTRGPolarity_Inverted) || \
                                               ((POLARITY) == TIM_ExtTRGPolarity_NonInverted))
/* @end */
 
/* @defgroup TIM_EncoderMode */
#define TIM_EncoderMode_TI1                   ((uint32_t)0x00000001) /*@EN: Counter counts on TI1FP1 edge depending on TI2FP2 level. @CN:TI1FP1ԵϵļȡTI2FP2ƽ*/
#define TIM_EncoderMode_TI2                   ((uint32_t)0x00000002) /*@EN: Counter counts on TI2FP2 edge depending on TI1FP1 level. @CN:TI2FP2ԵϵļȡTI1FP1ƽ*/
#define TIM_EncoderMode_TI12                  ((uint32_t)0x00000003) /*@EN: Counter counts on both TI1FP1 and TI2FP2 edges depending on the level of the other input. @CN:TI1FP1TI2FP2ϵļȡһĵƽ*/
#define IS_TIM_ENCODER_MODE(MODE)             (((MODE) == TIM_EncoderMode_TI1) || \
                                               ((MODE) == TIM_EncoderMode_TI2) || \
                                               ((MODE) == TIM_EncoderMode_TI12))
/* @end */

#define TIM_CCMR1_CC1S_0                      ((uint32_t)0x0001) /*@EN: Define the direction of the channel @CN: ͨķ */
#define TIM_CCMR1_CC1S_1                      ((uint32_t)0x0002) /*@EN: Define the direction of the channel @CN: ͨķ */
#define TIM_CCMR1_CC2S_0                      ((uint32_t)0x0100) /*@EN: Define the direction of the channel @CN: ͨķ */
#define TIM_CCMR1_CC2S_1                      ((uint32_t)0x0200) /*@EN: Define the direction of the channel @CN: ͨķ */

/* @defgroup TIM1_INPUT */
#define TIM1_INPUT_ETR                        ((uint8_t)0x04)  /*@EN: External trigger input @CN: ⲿ */
#define IS_TIM1_INPUT_TI(TIx)                 ((TIx == TIM1_INPUT_ETR))
/* @end */

/* @defgroup TIM1_IntputSource */
#define TIM1_IntputSource_IO                  ((uint32_t)0x00000000)  /*@EN: Input trigger source: GPIO @CN: 봥ԴGPIO */
#define TIM1_IntputSource_TIM2_TRGO           ((uint32_t)0x00000001)  /*@EN: Input trigger source: TIM2 TRGO @CN: 봥ԴTIM2 TRGO */
#define TIM1_IntputSource_ACMP0_OUTPUT        ((uint32_t)0x00000002)  /*@EN: Input trigger source: ACMP0 OUT @CN: 봥ԴACMP0 OUT */
#define TIM1_IntputSource_ACMP1_OUTPUT        ((uint32_t)0x00000003)  /*@EN: Input trigger source: ACMP1 OUT @CN: 봥ԴACMP1 OUT */
#define TIM1_IntputSource_TIM15_TRIG_OC1      ((uint32_t)0x00000005)  /*@EN: Input trigger source: TIM15 TRIG OC1 @CN: 봥ԴTIM15 TRIG OC1 */
#define TIM1_IntputSource_TIM16_TRIG_OC1      ((uint32_t)0x00000006)  /*@EN: Input trigger source: TIM16 TRIG OC1 @CN: 봥ԴTIM16 TRIG OC1 */
#define IS_TIM1_IntputSource(INx)             ((INx == TIM1_IntputSource_IO) ||             \
                                               (INx == TIM1_IntputSource_TIM2_TRGO) ||            \
                                               (INx == TIM1_IntputSource_ACMP0_OUTPUT) ||   \
                                               (INx == TIM1_IntputSource_ACMP1_OUTPUT) ||   \
                                               (INx == TIM1_IntputSource_TIM15_TRIG_OC1) ||   \
                                               (INx == TIM1_IntputSource_TIM16_TRIG_OC1))
/* @end */

/* @defgroup TIM2_INPUT */
#define TIM2_INPUT_TI1                        ((uint8_t)0x00)  /*@EN: External Input 1 @CN: ⲿ1 */
#define TIM2_INPUT_TI2                        ((uint8_t)0x01)  /*@EN: External Input 2 @CN: ⲿ2 */
#define TIM2_INPUT_TI3                        ((uint8_t)0x02)  /*@EN: External Input 3 @CN: ⲿ3 */
#define TIM2_INPUT_TI4                        ((uint8_t)0x03)  /*@EN: External Input 4 @CN: ⲿ4 */
#define TIM2_INPUT_ETR                        ((uint8_t)0x04)  /*@EN: External trigger input @CN: ⲿ */
#define IS_TIM2_INPUT_TI(TIx)                 ((TIx == TIM2_INPUT_TI1) || \
                                               (TIx == TIM2_INPUT_TI2) || \
                                               (TIx == TIM2_INPUT_TI3) || \
                                               (TIx == TIM2_INPUT_TI4) || \
                                               (TIx == TIM2_INPUT_ETR))
/* @end */

/* @defgroup TIM2_IntputSource */
#define TIM2_IntputSource_IO                  ((uint32_t)0x00000000)  /*@EN: Input trigger source: GPIO @CN: 봥ԴGPIO */
#define TIM2_IntputSource_AWD                 ((uint32_t)0x00000001)  /*@EN: Input trigger source: AWD @CN: 봥ԴAWD */
#define TIM2_IntputSource_ACMP0_OUTPUT        ((uint32_t)0x00000002)  /*@EN: Input trigger source: ACMP0 OUTPUT   @CN: 봥ԴACMP0 OUTPUT   */
#define TIM2_IntputSource_ACMP1_OUTPUT        ((uint32_t)0x00000003)  /*@EN: Input trigger source: ACMP1 OUTPUT   @CN: 봥ԴACMP1 OUTPUT   */
#define TIM2_IntputSource_TIM1_TRIG_OC1       ((uint32_t)0x00000004)  /*@EN: Input trigger source: TIM1 TRIG OC1  @CN: 봥ԴTIM1 TRIG OC1  */
#define TIM2_IntputSource_TIM1_TRIG_OC5       ((uint32_t)0x00000005)  /*@EN: Input trigger source: TIM1 TRIG OC5  @CN: 봥ԴTIM1 TRIG OC5  */
#define TIM2_IntputSource_TIM15_TRIG_OC1      ((uint32_t)0x00000006)  /*@EN: Input trigger source: TIM15 TRIG OC1 @CN: 봥ԴTIM15 TRIG OC1 */
#define TIM2_IntputSource_TIM16_TRIG_OC1      ((uint32_t)0x00000007)  /*@EN: Input trigger source: TIM16 TRIG OC1 @CN: 봥ԴTIM16 TRIG OC1 */
#define IS_TIM2_IntputSource(INx)             ((INx == TIM2_IntputSource_IO) ||             \
                                               (INx == TIM2_IntputSource_AWD) ||            \
                                               (INx == TIM2_IntputSource_ACMP0_OUTPUT) ||   \
                                               (INx == TIM2_IntputSource_ACMP1_OUTPUT) ||   \
                                               (INx == TIM2_IntputSource_TIM1_TRIG_OC1) ||  \
                                               (INx == TIM2_IntputSource_TIM1_TRIG_OC5) ||  \
                                               (INx == TIM2_IntputSource_TIM16_TRIG_OC1) || \
                                               (INx == TIM2_IntputSource_TIM16_TRIG_OC1))
/* @end */

/* @defgroup TIM15_INPUT */
#define TIM15_INPUT_TI1                       ((uint8_t)0x00)  /*@EN: External Input 1 @CN: ⲿ1 */
#define TIM15_INPUT_TI2                       ((uint8_t)0x01)  /*@EN: External Input 2 @CN: ⲿ2 */
#define TIM15_INPUT_ETR                       ((uint8_t)0x02)  /*@EN: External trigger input @CN: ⲿ */
#define IS_TIM15_INPUT_TI(TIx)                ((TIx == TIM15_INPUT_TI1) || \
                                               (TIx == TIM15_INPUT_TI2) || \
                                               (TIx == TIM15_INPUT_ETR))
/* @end */

/* @defgroup TIM15IntputSource */
#define TIM15_IntputSource_IO                 ((uint32_t)0x00000000)  /*@EN: Input trigger source: GPIO @CN: 봥ԴGPIO */
#define TIM15_IntputSource_AWD0_TIM2_TRGO     ((uint32_t)0x00000001)  /*@EN: Input trigger source: AWD0 TIM2 TRGO     @CN: 봥ԴAWD0 TIM2 TRGO     */
#define TIM15_IntputSource_AWD1               ((uint32_t)0x00000002)  /*@EN: Input trigger source: AWD1               @CN: 봥ԴAWD1               */
#define TIM15_IntputSource_MCO                ((uint32_t)0x00000002)  /*@EN: Input trigger source: MCO                @CN: 봥ԴMCO                */
#define TIM15_IntputSource_ACMP0_ACMP1_OUTPUT ((uint32_t)0x00000003)  /*@EN: Input trigger source: ACMP0 ACMP1 OUTPUT @CN: 봥ԴACMP0 ACMP1 OUTPUT */
#define TIM15_IntputSource_ACMP1_OUTPUT       ((uint32_t)0x00000004)  /*@EN: Input trigger source: ACMP1 OUTPUT       @CN: 봥ԴACMP1 OUTPUT       */
#define TIM15_IntputSource_TIM1_TRIG_OC1      ((uint32_t)0x00000005)  /*@EN: Input trigger source: TIM1 TRIG_OC1      @CN: 봥ԴTIM1 TRIG_OC1      */
#define TIM15_IntputSource_TIM1_TRIG_OC5      ((uint32_t)0x00000006)  /*@EN: Input trigger source: TIM1 TRIG_OC5      @CN: 봥ԴTIM1 TRIG_OC5      */
#define TIM15_IntputSource_TIM2_TRIG_OC1      ((uint32_t)0x00000007)  /*@EN: Input trigger source: TIM2 TRIG_OC1      @CN: 봥ԴTIM2 TRIG_OC1      */
#define IS_TIM15_IntputSource(INx)            ((INx == TIM15_IntputSource_IO) ||             \
                                               (INx == TIM15_IntputSource_AWD0_TIM2_TRGO) || \
                                               (INx == TIM15_IntputSource_AWD1) || \
                                               (INx == TIM15_IntputSource_MCO) ||   \
                                               (INx == TIM15_IntputSource_ACMP0_ACMP1_OUTPUT) ||   \
                                               (INx == TIM15_IntputSource_ACMP1_OUTPUT) ||  \
                                               (INx == TIM15_IntputSource_TIM1_TRIG_OC1) ||  \
                                               (INx == TIM15_IntputSource_TIM1_TRIG_OC5) || \
                                               (INx == TIM15_IntputSource_TIM2_TRIG_OC1))
/* @end */

/* @defgroup TIM16_INPUT */
#define TIM16_INPUT_TI1                       ((uint8_t)0x00)  /*@EN: External Input 1 @CN: ⲿ1 */
#define TIM16_INPUT_TI2                       ((uint8_t)0x01)  /*@EN: External Input 2 @CN: ⲿ2 */
#define TIM16_INPUT_ETR                       ((uint8_t)0x02)  /*@EN: External trigger input @CN: ⲿ */
#define IS_TIM16_INPUT_TI(TIx)                ((TIx == TIM16_INPUT_TI1) || \
                                               (TIx == TIM16_INPUT_TI2) || \
                                               (TIx == TIM16_INPUT_ETR))
/* @end */

/* @defgroup TIM16IntputSource */
#define TIM16_IntputSource_IO                 ((uint32_t)0x00000000)  /*@EN: Input trigger source: GPIO @CN: 봥ԴGPIO */
#define TIM16_IntputSource_AWD0_1_TIM2_TRGO   ((uint32_t)0x00000001)  /*@EN: Input trigger source: AWD0 1 TIM2 TRGO @CN: 봥ԴAWD0 1 TIM2 TRGO */
#define TIM16_IntputSource_ACMP0_OUTPUT       ((uint32_t)0x00000002)  /*@EN: Input trigger source: ACMP0 OUTPUT     @CN: 봥ԴACMP0 OUTPUT     */
#define TIM16_IntputSource_ACMP1_OUTPUT       ((uint32_t)0x00000003)  /*@EN: Input trigger source: ACMP1 OUTPUT     @CN: 봥ԴACMP1 OUTPUT     */
#define TIM16_IntputSource_TIM2_TRGO          ((uint32_t)0x00000004)  /*@EN: Input trigger source: TIM2 TRGO        @CN: 봥ԴTIM2 TRGO        */
#define TIM16_IntputSource_TIM1_TRGO          ((uint32_t)0x00000005)  /*@EN: Input trigger source: TIM1 TRGO        @CN: 봥ԴTIM1 TRGO        */
#define TIM16_IntputSource_TIM1_TRIG_OC5      ((uint32_t)0x00000006)  /*@EN: Input trigger source: TIM1 TRIG_OC5    @CN: 봥ԴTIM1 TRIG_OC5    */
#define TIM16_IntputSource_TIM15_TRGO         ((uint32_t)0x00000007)  /*@EN: Input trigger source: TIM15 TRGO       @CN: 봥ԴTIM15 TRGO       */
#define IS_TIM16_IntputSource(INx)            ((INx == TIM16_IntputSource_IO) ||              \
                                               (INx == TIM16_IntputSource_AWD0_1_TIM2_TRGO) || \
                                               (INx == TIM16_IntputSource_ACMP0_OUTPUT) ||   \
                                               (INx == TIM16_IntputSource_ACMP1_OUTPUT) ||   \
                                               (INx == TIM16_IntputSource_TIM2_TRGO) ||  \
                                               (INx == TIM16_IntputSource_TIM1_TRGO) ||  \
                                               (INx == TIM16_IntputSource_TIM1_TRIG_OC5) || \
                                               (INx == TIM16_IntputSource_TIM15_TRGO))
/* @end */

/* @defgroup TIM17_INPUT */
#define TIM17_INPUT_TI1                       ((uint8_t)0x00)  /*@EN: External Input 1 @CN: ⲿ1 */
#define TIM17_INPUT_ETR                       ((uint8_t)0x01)  /*@EN: External trigger input @CN: ⲿ */
#define IS_TIM17_INPUT_TI(TIx)                ((TIx == TIM17_INPUT_TI1) || \
                                               (TIx == TIM17_INPUT_ETR))
/* @end */

/* @defgroup TIM17IntputSource */
#define TIM17_IntputSource_IO                 ((uint32_t)0x00000000)  /*@EN: Input trigger source: GPIO @CN: 봥ԴGPIO */
#define TIM17_IntputSource_AWD1_TIM2_TRGO     ((uint32_t)0x00000001)  /*@EN: Input trigger source: AWD1 TIM2 TRGO @CN: 봥ԴAWD1 TIM2 TRGO */
#define TIM17_IntputSource_ACMP0_OUTPUT       ((uint32_t)0x00000002)  /*@EN: Input trigger source: ACMP0 OUTPUT   @CN: 봥ԴACMP0 OUTPUT   */
#define TIM17_IntputSource_ACMP1_OUTPUT       ((uint32_t)0x00000003)  /*@EN: Input trigger source: ACMP1 OUTPUT   @CN: 봥ԴACMP1 OUTPUT   */
#define TIM17_IntputSource_TIM2_TRGO          ((uint32_t)0x00000004)  /*@EN: Input trigger source: TIM2 TRGO      @CN: 봥ԴTIM2 TRGO      */
#define TIM17_IntputSource_TIM1_TRGO          ((uint32_t)0x00000005)  /*@EN: Input trigger source: TIM1 TRGO      @CN: 봥ԴTIM1 TRGO      */
#define TIM17_IntputSource_TIM1_TRIG_OC5      ((uint32_t)0x00000006)  /*@EN: Input trigger source: TIM1 TRIG_OC5  @CN: 봥ԴTIM1 TRIG_OC5  */
#define TIM17_IntputSource_TIM15_TRGO         ((uint32_t)0x00000007)  /*@EN: Input trigger source: TIM15 TRGO     @CN: 봥ԴTIM15 TRGO     */
#define IS_TIM17_IntputSource(INx)            ((INx == TIM17_IntputSource_IO) ||              \
                                               (INx == TIM17_IntputSource_AWD1_TIM2_TRGO) || \
                                               (INx == TIM17_IntputSource_ACMP0_OUTPUT) ||   \
                                               (INx == TIM17_IntputSource_ACMP1_OUTPUT) ||   \
                                               (INx == TIM17_IntputSource_TIM2_TRGO) ||  \
                                               (INx == TIM17_IntputSource_TIM1_TRGO) ||  \
                                               (INx == TIM17_IntputSource_TIM1_TRIG_OC5) || \
                                               (INx == TIM17_IntputSource_TIM15_TRGO))
/* @end */

/* @defgroup TIM1_BreakSource */
// TIM1 break1
#define TIM1_Break1Source_ACMP0_OUTPUT        ((uint32_t)0x00000100)  /*@EN: Break 1 Input trigger source: ACMP0 OUTPUT  @CN: ɲ1봥ԴACMP0 OUTPUT */
#define TIM1_Break1Source_ACMP1_OUTPUT        ((uint32_t)0x00000200)  /*@EN: Break 1 Input trigger source: ACMP1 OUTPUT  @CN: ɲ1봥ԴACMP1 OUTPUT */
#define TIM1_Break1Source_TIM2_TRGO           ((uint32_t)0x00000400)  /*@EN: Break 1 Input trigger source: TIM2 TRGO     @CN: ɲ1봥ԴTIM2 TRGO    */
#define TIM1_Break1Source_TIM15_TRGO          ((uint32_t)0x00000800)  /*@EN: Break 1 Input trigger source: TIM15 TRGO    @CN: ɲ1봥ԴTIM15 TRGO   */
#define TIM1_Break1Source_AWD0                ((uint32_t)0x00001000)  /*@EN: Break 1 Input trigger source: AWD0          @CN: ɲ1봥ԴAWD0         */
#define TIM1_Break1Source_AWD1                ((uint32_t)0x00002000)  /*@EN: Break 1 Input trigger source: AWD1          @CN: ɲ1봥ԴAWD1         */
#define TIM1_Break1Source_IO                  ((uint32_t)0x00004000)  /*@EN: Break 1 Input trigger source: GPIO @CN: ɲ1봥ԴGPIO */
// TIM1 break2
#define TIM1_Break2Source_ACMP0_OUTPUT        ((uint32_t)0x00010000)  /*@EN: Break 2 Input trigger source: ACMP0 OUTPUT   @CN: ɲ2봥ԴACMP0 OUTPUT   */
#define TIM1_Break2Source_ACMP1_OUTPUT        ((uint32_t)0x00020000)  /*@EN: Break 2 Input trigger source: ACMP1 OUTPUT   @CN: ɲ2봥ԴACMP1 OUTPUT   */
#define TIM1_Break2Source_TIM17_TRIG_OC1      ((uint32_t)0x00040000)  /*@EN: Break 2 Input trigger source: TIM17 TRIG OC1 @CN: ɲ2봥ԴTIM17 TRIG OC1 */
#define TIM1_Break2Source_TIM16_TRIG_OC1      ((uint32_t)0x00080000)  /*@EN: Break 2 Input trigger source: TIM16 TRIG OC1 @CN: ɲ2봥ԴTIM16 TRIG OC1 */
#define TIM1_Break2Source_AWD0                ((uint32_t)0x00100000)  /*@EN: Break 2 Input trigger source: AWD0           @CN: ɲ2봥ԴAWD0           */
#define TIM1_Break2Source_AWD1                ((uint32_t)0x00200000)  /*@EN: Break 2 Input trigger source: AWD1           @CN: ɲ2봥ԴAWD1           */
#define IS_TIM1_BreakSource(Source)           ((Source == TIM1_Break1Source_ACMP0_OUTPUT) ||    \
                                               (Source == TIM1_Break1Source_ACMP1_OUTPUT) ||    \
                                               (Source == TIM1_Break1Source_TIM2_TRGO) ||    \
                                               (Source == TIM1_Break1Source_TIM15_TRGO) ||       \
                                               (Source == TIM1_Break1Source_AWD0) ||      \
                                               (Source == TIM1_Break1Source_AWD1) ||             \
                                               (Source == TIM1_Break1Source_IO) ||              \
                                               (Source == TIM1_Break2Source_ACMP0_OUTPUT) ||   \
                                               (Source == TIM1_Break2Source_ACMP1_OUTPUT) ||   \
                                               (Source == TIM1_Break2Source_TIM17_TRIG_OC1) ||   \
                                               (Source == TIM1_Break2Source_TIM16_TRIG_OC1) || \
                                               (Source == TIM1_Break2Source_AWD0) || \
                                               (Source == TIM1_Break2Source_AWD1))
/* @end */

/* @defgroup TIM1_OCREF_CLR_Source */
#define TIM1_OCREF_CLR_Source_ACMP0_OUTPUT    ((uint32_t)0x01000000)  /*@EN: External event clearing Input trigger source: ACMP0 OUTPUT @CN: ⲿ¼봥ԴACMP0 OUTPUT */
#define TIM1_OCREF_CLR_Source_ACMP1_OUTPUT    ((uint32_t)0x02000000)  /*@EN: External event clearing Input trigger source: ACMP1 OUTPUT @CN: ⲿ¼봥ԴACMP1 OUTPUT */
#define TIM1_OCREF_CLR_Source_TIM2_TRGO       ((uint32_t)0x04000000)  /*@EN: External event clearing Input trigger source: TIM2 TRGO    @CN: ⲿ¼봥ԴTIM2 TRGO    */
#define TIM1_OCREF_CLR_Source_TIM15_TRGO      ((uint32_t)0x08000000)  /*@EN: External event clearing Input trigger source: TIM15 TRGO   @CN: ⲿ¼봥ԴTIM15 TRGO   */
#define TIM1_OCREF_CLR_Source_AWD0            ((uint32_t)0x10000000)  /*@EN: External event clearing Input trigger source: AWD0         @CN: ⲿ¼봥ԴAWD0         */
#define TIM1_OCREF_CLR_Source_AWD1            ((uint32_t)0x20000000)  /*@EN: External event clearing Input trigger source: AWD1         @CN: ⲿ¼봥ԴAWD1         */
#define IS_TIM1_OCREF_CLR_Source(Source)      ((Source == TIM1_OCREF_CLR_Source_ACMP0_OUTPUT) || \
                                               (Source == TIM1_OCREF_CLR_Source_ACMP1_OUTPUT) || \
                                               (Source == TIM1_OCREF_CLR_Source_TIM2_TRGO) || \
                                               (Source == TIM1_OCREF_CLR_Source_TIM15_TRGO) ||    \
                                               (Source == TIM1_OCREF_CLR_Source_AWD0) ||   \
                                               (Source == TIM1_OCREF_CLR_Source_AWD1))
/* @end */

/* @defgroup TIM17_BreakSource */
#define TIM17_Break1Source_ACMP0_OUTPUT       ((uint32_t)0x01000000)  /*@EN: Break 1 Input trigger source:ACMP0 OUTPUT @CN: ɲ1봥ԴACMP0 OUTPUT */
#define TIM17_Break1Source_ACMP1_OUTPUT       ((uint32_t)0x02000000)  /*@EN: Break 1 Input trigger source:ACMP1 OUTPUT @CN: ɲ1봥ԴACMP1 OUTPUT */
#define TIM17_Break1Source_TIM2_TRGO          ((uint32_t)0x04000000)  /*@EN: Break 1 Input trigger source:TIM2 TRGO    @CN: ɲ1봥ԴTIM2 TRGO    */
#define TIM17_Break1Source_TIM1_TRGO          ((uint32_t)0x08000000)  /*@EN: Break 1 Input trigger source:TIM1 TRGO    @CN: ɲ1봥ԴTIM1 TRGO    */
#define TIM17_Break1Source_AWD0               ((uint32_t)0x10000000)  /*@EN: Break 1 Input trigger source:AWD0         @CN: ɲ1봥ԴAWD0         */
#define TIM17_Break1Source_AWD1               ((uint32_t)0x20000000)  /*@EN: Break 1 Input trigger source:AWD1         @CN: ɲ1봥ԴAWD1         */
#define TIM17_Break1Source_IO                 ((uint32_t)0x40000000)  /*@EN: Break 1 Input trigger source:GPIO         @CN: ɲ1봥ԴGPIO         */
#define IS_TIM17_BreakSource(Source)          ((Source == TIM17_Break1Source_ACMP0_OUTPUT) || \
                                                (Source == TIM17_Break1Source_ACMP1_OUTPUT) || \
                                                (Source == TIM17_Break1Source_TIM2_TRGO) || \
                                                (Source == TIM17_Break1Source_TIM1_TRGO) || \
                                                (Source == TIM17_Break1Source_AWD0) || \
                                                (Source == TIM17_Break1Source_AWD1) || \
                                                (Source == TIM17_Break1Source_IO))
/* @end */

/* @defgroup TIM15_BreakSource */
// TIM15 break1
#define TIM15_Break1Source_ACMP0_OUTPUT       ((uint32_t)0x00000100)  /*@EN: Break 1 Input trigger source:ACMP0 OUTPUT @CN: ɲ1봥ԴACMP0 OUTPUT */
#define TIM15_Break1Source_ACMP1_OUTPUT       ((uint32_t)0x00000200)  /*@EN: Break 1 Input trigger source:ACMP1 OUTPUT @CN: ɲ1봥ԴACMP1 OUTPUT */
#define TIM15_Break1Source_TIM2_TRGO          ((uint32_t)0x00000400)  /*@EN: Break 1 Input trigger source:TIM2 TRGO    @CN: ɲ1봥ԴTIM2 TRGO    */
#define TIM15_Break1Source_TIM1_TRGO          ((uint32_t)0x00000800)  /*@EN: Break 1 Input trigger source:TIM1 TRGO    @CN: ɲ1봥ԴTIM1 TRGO    */
#define TIM15_Break1Source_AWD0               ((uint32_t)0x00001000)  /*@EN: Break 1 Input trigger source:AWD0         @CN: ɲ1봥ԴAWD0         */
#define TIM15_Break1Source_AWD1               ((uint32_t)0x00002000)  /*@EN: Break 1 Input trigger source:AWD1         @CN: ɲ1봥ԴAWD1         */
#define TIM15_Break1Source_IO                 ((uint32_t)0x00004000)  /*@EN: Break 1 Input trigger source:IO           @CN: ɲ1봥ԴIO           */
// TIM15 break2
#define TIM15_Break2Source_ACMP0_OUTPUT       ((uint32_t)0x00010000)  /*@EN: Break 2 Input trigger source:ACMP0 OUTPUT   @CN: ɲ2봥ԴACMP0 OUTPUT   */
#define TIM15_Break2Source_ACMP1_OUTPUT       ((uint32_t)0x00020000)  /*@EN: Break 2 Input trigger source:ACMP1 OUTPUT   @CN: ɲ2봥ԴACMP1 OUTPUT   */
#define TIM15_Break2Source_TIM17_TRIG_OC1     ((uint32_t)0x00040000)  /*@EN: Break 2 Input trigger source:TIM17 TRIG OC1 @CN: ɲ2봥ԴTIM17 TRIG OC1 */
#define TIM15_Break2Source_TIM16_TRIG_OC1     ((uint32_t)0x00080000)  /*@EN: Break 2 Input trigger source:TIM16 TRIG OC1 @CN: ɲ2봥ԴTIM16 TRIG OC1 */
#define TIM15_Break2Source_AWD0               ((uint32_t)0x00100000)  /*@EN: Break 2 Input trigger source:AWD0           @CN: ɲ2봥ԴAWD0           */
#define TIM15_Break2Source_AWD1               ((uint32_t)0x00200000)  /*@EN: Break 2 Input trigger source:AWD1           @CN: ɲ2봥ԴAWD1           */
#define IS_TIM15_BreakSource(Source)          ((Source == TIM15_Break1Source_ACMP0_OUTPUT) ||    \
                                                (Source == TIM15_Break1Source_ACMP1_OUTPUT) ||    \
                                                (Source == TIM15_Break1Source_TIM2_TRGO) ||    \
                                                (Source == TIM15_Break1Source_TIM1_TRGO) ||       \
                                                (Source == TIM15_Break1Source_AWD0) ||      \
                                                (Source == TIM15_Break1Source_AWD1) ||             \
                                                (Source == TIM15_Break1Source_IO) ||              \
                                                (Source == TIM15_Break2Source_ACMP0_OUTPUT) ||   \
                                                (Source == TIM15_Break2Source_ACMP1_OUTPUT) ||   \
                                                (Source == TIM15_Break2Source_TIM17_TRIG_OC1) ||   \
                                                (Source == TIM15_Break2Source_TIM16_TRIG_OC1) || \
                                                (Source == TIM15_Break2Source_AWD0) || \
                                                (Source == TIM15_Break2Source_AWD1))
/* @end */

/* @defgroup TIM16_BreakSource */
// TIM16 break1
#define TIM16_Break1Source_ACMP0_OUTPUT       ((uint32_t)0x00000100)  /*@EN: Break 1 Input trigger source:ACMP0 OUTPUT @CN: ɲ1봥ԴACMP0 OUTPUT */
#define TIM16_Break1Source_ACMP1_OUTPUT       ((uint32_t)0x00000200)  /*@EN: Break 1 Input trigger source:ACMP1 OUTPUT @CN: ɲ1봥ԴACMP1 OUTPUT */
#define TIM16_Break1Source_TIM2_TRGO          ((uint32_t)0x00000400)  /*@EN: Break 1 Input trigger source:TIM2 TRGO    @CN: ɲ1봥ԴTIM2 TRGO    */
#define TIM16_Break1Source_TIM1_TRGO          ((uint32_t)0x00000800)  /*@EN: Break 1 Input trigger source:TIM1 TRGO    @CN: ɲ1봥ԴTIM1 TRGO    */
#define TIM16_Break1Source_AWD0               ((uint32_t)0x00001000)  /*@EN: Break 1 Input trigger source:AWD0         @CN: ɲ1봥ԴAWD0         */
#define TIM16_Break1Source_AWD1               ((uint32_t)0x00002000)  /*@EN: Break 1 Input trigger source:AWD1         @CN: ɲ1봥ԴAWD1         */
#define TIM16_Break1Source_IO                 ((uint32_t)0x00004000)  /*@EN: Break 1 Input trigger source:IO           @CN: ɲ1봥ԴIO           */
// TIM16 break2
#define TIM16_Break2Source_ACMP0_OUTPUT       ((uint32_t)0x00010000)  /*@EN: Break 2 Input trigger source:ACMP0 OUTPUT   @CN: ɲ2봥ԴACMP0 OUTPUT   */
#define TIM16_Break2Source_ACMP1_OUTPUT       ((uint32_t)0x00020000)  /*@EN: Break 2 Input trigger source:ACMP1 OUTPUT   @CN: ɲ2봥ԴACMP1 OUTPUT   */
#define TIM16_Break2Source_TIM17_TRIG_OC1     ((uint32_t)0x00040000)  /*@EN: Break 2 Input trigger source:TIM17_TRIG OC1 @CN: ɲ2봥ԴTIM17 TRIG OC1 */
#define TIM16_Break2Source_TIM15_TRGO         ((uint32_t)0x00080000)  /*@EN: Break 2 Input trigger source:TIM15 TRGO     @CN: ɲ2봥ԴTIM15 TRGO     */
#define TIM16_Break2Source_AWD0               ((uint32_t)0x00100000)  /*@EN: Break 2 Input trigger source:AWD0           @CN: ɲ2봥ԴAWD0           */
#define TIM16_Break2Source_AWD1               ((uint32_t)0x00200000)  /*@EN: Break 2 Input trigger source:AWD1           @CN: ɲ2봥ԴAWD1           */
#define IS_TIM16_BreakSource(Source)          ((Source == TIM16_Break1Source_ACMP0_OUTPUT) ||    \
                                                (Source == TIM16_Break1Source_ACMP1_OUTPUT) ||    \
                                                (Source == TIM16_Break1Source_TIM2_TRGO) ||    \
                                                (Source == TIM16_Break1Source_TIM1_TRGO) ||       \
                                                (Source == TIM16_Break1Source_AWD0) ||      \
                                                (Source == TIM16_Break1Source_AWD1) ||             \
                                                (Source == TIM16_Break1Source_IO) ||              \
                                                (Source == TIM16_Break2Source_ACMP0_OUTPUT) ||   \
                                                (Source == TIM16_Break2Source_ACMP1_OUTPUT) ||   \
                                                (Source == TIM16_Break2Source_TIM17_TRIG_OC1) ||   \
                                                (Source == TIM16_Break2Source_TIM15_TRGO) || \
                                                (Source == TIM16_Break2Source_AWD0) || \
                                                (Source == TIM16_Break2Source_AWD1))
/* @end */

/* @defgroup TIM2_BreakSource */
#define TIM2_Break1Source_ACMP0_OUTPUT        ((uint32_t)0x01000000)  /*@EN: Break 1 Input trigger source:ACMP0 OUTPUT @CN: ɲ1봥ԴACMP0 OUTPUT */
#define TIM2_Break1Source_ACMP1_OUTPUT        ((uint32_t)0x02000000)  /*@EN: Break 1 Input trigger source:ACMP1 OUTPUT @CN: ɲ1봥ԴACMP1 OUTPUT */
#define TIM2_Break1Source_TIM15_TRGO          ((uint32_t)0x04000000)  /*@EN: Break 1 Input trigger source:TIM15 TRGO   @CN: ɲ1봥ԴTIM15 TRGO   */
#define TIM2_Break1Source_TIM1_TRGO           ((uint32_t)0x08000000)  /*@EN: Break 1 Input trigger source:TIM1 TRGO    @CN: ɲ1봥ԴTIM1 TRGO    */
#define TIM2_Break1Source_AWD0                ((uint32_t)0x10000000)  /*@EN: Break 1 Input trigger source:AWD0         @CN: ɲ1봥ԴAWD0         */
#define TIM2_Break1Source_AWD1                ((uint32_t)0x20000000)  /*@EN: Break 1 Input trigger source:AWD1         @CN: ɲ1봥ԴAWD1         */
#define TIM2_Break1Source_IO                  ((uint32_t)0x40000000)  /*@EN: Break 1 Input trigger source:IO           @CN: ɲ1봥ԴIO           */
#define IS_TIM2_BreakSource(Source)           ((Source == TIM2_Break1Source_ACMP0_OUTPUT) || \
                                               (Source == TIM2_Break1Source_ACMP1_OUTPUT) || \
                                               (Source == TIM2_Break1Source_TIM15_TRGO) || \
                                               (Source == TIM2_Break1Source_TIM1_TRGO) || \
                                               (Source == TIM2_Break1Source_AWD0) || \
                                               (Source == TIM2_Break1Source_AWD1) || \
                                               (Source == TIM2_Break1Source_IO))
/* @end */

/* @defgroup TIM2_OCREF_CLR_Source */
#define TIM2_OCREF_CLR_Source_ACMP0_OUTPUT    ((uint32_t)0x01000000)  /*@EN: External event clearing Input trigger source: ACMP0 OUTPUT @CN: ⲿ¼봥ԴACMP0 OUTPUT */
#define TIM2_OCREF_CLR_Source_ACMP1_OUTPUT    ((uint32_t)0x02000000)  /*@EN: External event clearing Input trigger source: ACMP1 OUTPUT @CN: ⲿ¼봥ԴACMP1 OUTPUT */
#define TIM2_OCREF_CLR_Source_TIM1_TRGO       ((uint32_t)0x04000000)  /*@EN: External event clearing Input trigger source: TIM1 TRGO    @CN: ⲿ¼봥ԴTIM1 TRGO    */
#define TIM2_OCREF_CLR_Source_TIM15_TRGO      ((uint32_t)0x08000000)  /*@EN: External event clearing Input trigger source: TIM15 TRGO   @CN: ⲿ¼봥ԴTIM15 TRGO   */
#define TIM2_OCREF_CLR_Source_AWD0            ((uint32_t)0x10000000)  /*@EN: External event clearing Input trigger source: AWD0         @CN: ⲿ¼봥ԴAWD0         */
#define TIM2_OCREF_CLR_Source_AWD1            ((uint32_t)0x20000000)  /*@EN: External event clearing Input trigger source: AWD1         @CN: ⲿ¼봥ԴAWD1         */
#define IS_TIM2_OCREF_CLR_Source(Source)      ((Source == TIM2_OCREF_CLR_Source_ACMP0_OUTPUT) || \
                                               (Source == TIM2_OCREF_CLR_Source_ACMP1_OUTPUT) || \
                                               (Source == TIM2_OCREF_CLR_Source_TIM1_TRGO) || \
                                               (Source == TIM2_OCREF_CLR_Source_TIM15_TRGO) ||    \
                                               (Source == TIM2_OCREF_CLR_Source_AWD0) ||   \
                                               (Source == TIM2_OCREF_CLR_Source_AWD1))
/* @end */

/* @defgroup TIM_InputLevel */
#define TIM_InputLevel_High                   ((uint8_t)0x00)  /*@EN: The effective level of the trigger source is high @CN: ԴЧƽǸ */
#define TIM_InputLevel_Low                    ((uint8_t)0x01)  /*@EN: The effective level of the trigger source is low  @CN: ԴЧƽǵ */
#define IS_TIM_InputLevel(Level)              (Level == TIM_InputLevel_High) || (Level == TIM_InputLevel_Low)
/* @end */

/* @defgroup PWM_Num */
#define TIM1_PWM1                             ((uint8_t)0x00)  /*@EN: TIM1 PWM1 @CN: TIM1 PWM1 */
#define TIM1_PWM2                             ((uint8_t)0x01)  /*@EN: TIM1 PWM2 @CN: TIM1 PWM2 */
#define TIM1_PWM3                             ((uint8_t)0x02)  /*@EN: TIM1 PWM3 @CN: TIM1 PWM3 */
#define TIM1_PWM4                             ((uint8_t)0x03)  /*@EN: TIM1 PWM4 @CN: TIM1 PWM4 */
#define TIM1_PWM5                             ((uint8_t)0x04)  /*@EN: TIM1 PWM5 @CN: TIM1 PWM5 */
#define TIM1_PWM6                             ((uint8_t)0x05)  /*@EN: TIM1 PWM6 @CN: TIM1 PWM6 */
#define TIM1_PWM7                             ((uint8_t)0x06)  /*@EN: TIM1 PWM7 @CN: TIM1 PWM7 */
#define TIM1_PWM8                             ((uint8_t)0x07)  /*@EN: TIM1 PWM8 @CN: TIM1 PWM8 */
/* @end */

/* @defgroup OCx_Sel */
#define TIM1_PWMx_OC1                         ((uint8_t)0x00)  /*@EN: TIM1 PWM output mapping OC1  @CN: TIM1 PWM ӳ OC1 */
#define TIM1_PWMx_OC2                         ((uint8_t)0x01)  /*@EN: TIM1 PWM output mapping OC2  @CN: TIM1 PWM ӳ OC2 */
#define TIM1_PWMx_OC3                         ((uint8_t)0x02)  /*@EN: TIM1 PWM output mapping OC3  @CN: TIM1 PWM ӳ OC3 */
#define TIM1_PWMx_OC4                         ((uint8_t)0x03)  /*@EN: TIM1 PWM output mapping OC4  @CN: TIM1 PWM ӳ OC4 */
#define TIM1_PWMx_OC1N                        ((uint8_t)0x04)  /*@EN: TIM1 PWM output mapping OC1N @CN: TIM1 PWM ӳ OC1N */
#define TIM1_PWMx_OC2N                        ((uint8_t)0x05)  /*@EN: TIM1 PWM output mapping OC2N @CN: TIM1 PWM ӳ OC2N */
#define TIM1_PWMx_OC3N                        ((uint8_t)0x06)  /*@EN: TIM1 PWM output mapping OC3N @CN: TIM1 PWM ӳ OC3N */
#define TIM1_PWMx_OC4N                        ((uint8_t)0x07)  /*@EN: TIM1 PWM output mapping OC4N @CN: TIM1 PWM ӳ OC4N */
/* @end */

/* @defgroup OCx_Mode */
#define TIM1_OCx_Mode_UP_CCRx                 ((uint32_t)0x00000001)  /*@EN: Counter up, CCRx serves as the comparison value for CCx channel @CN: ڼϼУʹCCRxΪCCxͨıȽֵ */
#define TIM1_OCx_Mode_DOWN_CCRx               ((uint32_t)0x00000002)  /*@EN: Counter down, CCRx serves as the comparison value for CCx channel @CN: ڼ¼УʹCCRxΪCCxͨıȽֵ */
#define TIM1_OCx_Mode_UP_CCORx                ((uint32_t)0x00000004)  /*@EN: Counter up, CCORx serves as the comparison value for CCx channel @CN: ڼϼУʹCCORxΪCCxͨıȽֵ */
#define TIM1_OCx_Mode_DOWN_CCORx              ((uint32_t)0x00000008)  /*@EN: Counter down, CCORx serves as the comparison value for CCx channel @CN: ڼ¼УʹCCORxΪCCxͨıȽֵ */
/* @end */

/* @defgroup AntiMisoperateStart */
#define TIM_AntiMisoperateStart_Delay         ((uint32_t)0x00000000)  /*@EN: Starting from delayed trigger input @CN: ӳٵĴ뿪ʼ */
#define TIM_AntiMisoperateStart_Immediate     ((uint32_t)0x00010000)  /*@EN: Starting from triggering input (default) @CN: Ӵ뿪ʼĬϣ */
/* @end */

/* ========================================================================== */
/* ================             TIM  REGISTER BIT               ============= */
/* ========================================================================== */

/* ==============================  CR1  ===================================== */
#define TIM_CR1_AS_Pos                           (24UL)                       /*!< AS (Bit 24)                        */
#define TIM_CR1_AS_Msk                           (0x01000000UL)               /*!< AS (Bitfield-Mask: 0x01)           */
#define TIM_CR1_OCF_Pos                          (23UL)                       /*!< OCF (Bit 23)                       */
#define TIM_CR1_OCF_Msk                          (0x00800000UL)               /*!< OCF (Bitfield-Mask: 0x01)          */
#define TIM_CR1_CENCE_Pos                        (22UL)                       /*!< CENCE (Bit 22)                     */
#define TIM_CR1_CENCE_Msk                        (0x00400000UL)               /*!< CENCE (Bitfield-Mask: 0x01)        */
#define TIM_CR1_TI4E_Pos                         (21UL)                       /*!< TI4E (Bit 21)                      */
#define TIM_CR1_TI4E_Msk                         (0x00200000UL)               /*!< TI4E (Bitfield-Mask: 0x01)         */
#define TIM_CR1_TI3E_Pos                         (20UL)                       /*!< TI3E (Bit 20)                      */
#define TIM_CR1_TI3E_Msk                         (0x00100000UL)               /*!< TI3E (Bitfield-Mask: 0x01)         */
#define TIM_CR1_TI2E_Pos                         (19UL)                       /*!< TI2E (Bit 19)                      */
#define TIM_CR1_TI2E_Msk                         (0x00080000UL)               /*!< TI2E (Bitfield-Mask: 0x01)         */
#define TIM_CR1_TI1E_Pos                         (18UL)                       /*!< TI1E (Bit 18)                      */
#define TIM_CR1_TI1E_Msk                         (0x00040000UL)               /*!< TI1E (Bitfield-Mask: 0x01)         */
#define TIM_CR1_ETRE_Pos                         (17UL)                       /*!< ETRE (Bit 17)                      */
#define TIM_CR1_ETRE_Msk                         (0x00020000UL)               /*!< ETRE (Bitfield-Mask: 0x01)         */
#define TIM_CR1_FTE_Pos                          (16UL)                       /*!< FTE (Bit 16)                      */
#define TIM_CR1_FTE_Msk                          (0x00010000UL)               /*!< FTE (Bitfield-Mask: 0x01)         */
#define TIM_CR1_CKD_Pos                          (8UL)                        /*!< CKD (Bit 8)                        */
#define TIM_CR1_CKD_Msk                          (0x00000300UL)               /*!< CKD (Bitfield-Mask: 0x03)          */
#define TIM_CR1_ARPE_Pos                         (7UL)                        /*!< ARPE (Bit 7)                       */
#define TIM_CR1_ARPE_Msk                         (0x00000080UL)               /*!< ARPE (Bitfield-Mask: 0x01)         */
#define TIM_CR1_CMS_Pos                          (5UL)                        /*!< CMS (Bit 5)                        */
#define TIM_CR1_CMS_Msk                          (0x00000060UL)               /*!< CMS (Bitfield-Mask: 0x03)          */
#define TIM_CR1_DIR_Pos                          (4UL)                        /*!< DIR (Bit 4)                        */
#define TIM_CR1_DIR_Msk                          (0x00000010UL)               /*!< DIR (Bitfield-Mask: 0x01)          */
#define TIM_CR1_OPM_Pos                          (3UL)                        /*!< OPM (Bit 3)                        */
#define TIM_CR1_OPM_Msk                          (0x00000008UL)               /*!< OPM (Bitfield-Mask: 0x01)          */
#define TIM_CR1_URS_Pos                          (2UL)                        /*!< URS (Bit 2)                        */
#define TIM_CR1_URS_Msk                          (0x00000004UL)               /*!< URS (Bitfield-Mask: 0x01)          */
#define TIM_CR1_UDIS_Pos                         (1UL)                        /*!< UDIS (Bit 1)                       */
#define TIM_CR1_UDIS_Msk                         (0x00000002UL)               /*!< UDIS (Bitfield-Mask: 0x01)         */
#define TIM_CR1_CEN_Pos                          (0UL)                        /*!< CEN (Bit 0)                        */
#define TIM_CR1_CEN_Msk                          (0x00000001UL)               /*!< CEN (Bitfield-Mask: 0x01)          */
/* =============================  CR2  ====================================== */
#define TIM_CR2_OIS5_Pos                         (16UL)                       /*!< OIS5 (Bit 16)                      */
#define TIM_CR2_OIS5_Msk                         (0x00010000UL)               /*!< OIS5 (Bitfield-Mask: 0x01)         */
#define TIM_CR2_OIS4N_Pos                        (15UL)                       /*!< OIS4N (Bit 15)                     */
#define TIM_CR2_OIS4N_Msk                        (0x00008000UL)               /*!< OIS4N (Bitfield-Mask: 0x01)        */
#define TIM_CR2_OIS4_Pos                         (14UL)                       /*!< OIS4 (Bit 14)                      */
#define TIM_CR2_OIS4_Msk                         (0x00004000UL)               /*!< OIS4 (Bitfield-Mask: 0x01)         */
#define TIM_CR2_OIS3N_Pos                        (13UL)                       /*!< OIS3N (Bit 13)                     */
#define TIM_CR2_OIS3N_Msk                        (0x00002000UL)               /*!< OIS3N (Bitfield-Mask: 0x01)        */
#define TIM_CR2_OIS3_Pos                         (12UL)                       /*!< OIS3 (Bit 12)                      */
#define TIM_CR2_OIS3_Msk                         (0x00001000UL)               /*!< OIS3 (Bitfield-Mask: 0x01)         */
#define TIM_CR2_OIS2N_Pos                        (11UL)                       /*!< OIS2N (Bit 11)                     */
#define TIM_CR2_OIS2N_Msk                        (0x00000800UL)               /*!< OIS2N (Bitfield-Mask: 0x01)        */
#define TIM_CR2_OIS2_Pos                         (10UL)                       /*!< OIS2 (Bit 10)                      */
#define TIM_CR2_OIS2_Msk                         (0x00000400UL)               /*!< OIS2 (Bitfield-Mask: 0x01)         */
#define TIM_CR2_OIS1N_Pos                        (9UL)                        /*!< OIS1N (Bit 9)                      */
#define TIM_CR2_OIS1N_Msk                        (0x00000200UL)               /*!< OIS1N (Bitfield-Mask: 0x01)        */
#define TIM_CR2_OIS1_Pos                         (8UL)                        /*!< OIS1 (Bit 8)                       */
#define TIM_CR2_OIS1_Msk                         (0x00000100UL)               /*!< OIS1 (Bitfield-Mask: 0x01)         */
#define TIM_CR2_TI1S_Pos                         (7UL)                        /*!< TI1S (Bit 7)                       */
#define TIM_CR2_TI1S_Msk                         (0x00000080UL)               /*!< TI1S (Bitfield-Mask: 0x01)         */
#define TIM_CR2_MMS_Pos                          (4UL)                        /*!< MMS (Bit 4)                        */
#define TIM_CR2_MMS_Msk                          (0x00000070UL)               /*!< MMS (Bitfield-Mask: 0x07)          */
#define TIM_CR2_CCDS_Pos                         (3UL)                        /*!< CCDS (Bit 3)                       */
#define TIM_CR2_CCDS_Msk                         (0x00000008UL)               /*!< CCDS (Bitfield-Mask: 0x01)         */
#define TIM_CR2_CCUS_Pos                         (2UL)                        /*!< CCUS (Bit 2)                       */
#define TIM_CR2_CCUS_Msk                         (0x00000004UL)               /*!< CCUS (Bitfield-Mask: 0x01)         */
#define TIM_CR2_CCPC_Pos                         (0UL)                        /*!< CCPC (Bit 0)                       */
#define TIM_CR2_CCPC_Msk                         (0x00000001UL)               /*!< CCPC (Bitfield-Mask: 0x01) */
/* ============================  SMCR  ====================================== */
#define TIM_SMCR_OCCP_Pos                        (17UL)                       /*!< OCCP (Bit 17)                      */
#define TIM_SMCR_OCCP_Msk                        (0x00020000UL)               /*!< OCCP (Bitfield-Mask: 0x01)         */
#define TIM_SMCR_SMS3_Pos                        (16UL)                       /*!< SMS3 (Bit 16)                      */
#define TIM_SMCR_SMS3_Msk                        (0x00010000UL)               /*!< SMS3 (Bitfield-Mask: 0x01)         */
#define TIM_SMCR_ETP_Pos                         (15UL)                       /*!< ETP (Bit 15)                       */
#define TIM_SMCR_ETP_Msk                         (0x00008000UL)               /*!< ETP (Bitfield-Mask: 0x01)          */
#define TIM_SMCR_ECE_Pos                         (14UL)                       /*!< ECE (Bit 14)                       */
#define TIM_SMCR_ECE_Msk                         (0x00004000UL)               /*!< ECE (Bitfield-Mask: 0x01)          */
#define TIM_SMCR_ETPS_Pos                        (12UL)                       /*!< ETPS (Bit 12)                      */
#define TIM_SMCR_ETPS_Msk                        (0x00003000UL)               /*!< ETPS (Bitfield-Mask: 0x03)         */
#define TIM_SMCR_ETF_Pos                         (8UL)                        /*!< ETF (Bit 8)                        */
#define TIM_SMCR_ETF_Msk                         (0x00000f00UL)               /*!< ETF (Bitfield-Mask: 0x0f)          */
#define TIM_SMCR_MSM_Pos                         (7UL)                        /*!< MSM (Bit 7)                        */
#define TIM_SMCR_MSM_Msk                         (0x00000080UL)               /*!< MSM (Bitfield-Mask: 0x01)          */
#define TIM_SMCR_TS_Pos                          (4UL)                        /*!< TS (Bit 4)                         */
#define TIM_SMCR_TS_Msk                          (0x00000070UL)               /*!< TS (Bitfield-Mask: 0x07)           */
#define TIM_SMCR_OCCS_Pos                        (3UL)                        /*!< OCCS (Bit 3)                       */
#define TIM_SMCR_OCCS_Msk                        (0x00000080UL)               /*!< OCCS (Bitfield-Mask: 0x01)         */
#define TIM_SMCR_SMS_Pos                         (0UL)                        /*!< SMS (Bit 0)                        */
#define TIM_SMCR_SMS_Msk                         (0x00000007UL)               /*!< SMS (Bitfield-Mask: 0x07)          */
/* ============================  DIER  ====================================== */
#define TIM_DIER_CC5DE_Pos                       (18UL)                       /*!< CC5DE (Bit 18)                       */
#define TIM_DIER_CC5DE_Msk                       (0x00040000UL)               /*!< CC5DE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_CC5IE_Pos                       (16UL)                       /*!< CC5IE (Bit 16)                       */
#define TIM_DIER_CC5IE_Msk                       (0x00010000UL)               /*!< CC5IE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_TDE_Pos                         (14UL)                       /*!< TDE (Bit 14)                         */
#define TIM_DIER_TDE_Msk                         (0x00004000UL)               /*!< TDE (Bitfield-Mask: 0x01)            */
#define TIM_DIER_COMDE_Pos                       (13UL)                       /*!< COMDE (Bit 13)                       */
#define TIM_DIER_COMDE_Msk                       (0x00002000UL)               /*!< COMDE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_CC4DE_Pos                       (12UL)                       /*!< CC4DE (Bit 12)                       */
#define TIM_DIER_CC4DE_Msk                       (0x00001000UL)               /*!< CC4DE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_CC3DE_Pos                       (11UL)                       /*!< CC3DE (Bit 11)                       */
#define TIM_DIER_CC3DE_Msk                       (0x00000800UL)               /*!< CC3DE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_CC2DE_Pos                       (10UL)                       /*!< CC2DE (Bit 10)                       */
#define TIM_DIER_CC2DE_Msk                       (0x00000400UL)               /*!< CC2DE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_CC1DE_Pos                       (9UL)                        /*!< CC1DE (Bit 9)                        */
#define TIM_DIER_CC1DE_Msk                       (0x00000200UL)               /*!< CC1DE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_UDE_Pos                         (8UL)                        /*!< UDE (Bit 8)                          */
#define TIM_DIER_UDE_Msk                         (0x00000100UL)               /*!< UDE (Bitfield-Mask: 0x01)            */
#define TIM_DIER_TIE_Pos                         (6UL)                        /*!< TIE (Bit 6)                          */
#define TIM_DIER_TIE_Msk                         (0x00000040UL)               /*!< TIE (Bitfield-Mask: 0x01)            */
#define TIM_DIER_CC4IE_Pos                       (4UL)                        /*!< CC4IE (Bit 4)                        */
#define TIM_DIER_CC4IE_Msk                       (0x00000010UL)               /*!< CC4IE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_CC3IE_Pos                       (3UL)                        /*!< CC3IE (Bit 3)                        */
#define TIM_DIER_CC3IE_Msk                       (0x00000008UL)               /*!< CC3IE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_CC2IE_Pos                       (2UL)                        /*!< CC2IE (Bit 2)                        */
#define TIM_DIER_CC2IE_Msk                       (0x00000004UL)               /*!< CC2IE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_CC1IE_Pos                       (1UL)                        /*!< CC1IE (Bit 1)                        */
#define TIM_DIER_CC1IE_Msk                       (0x00000002UL)               /*!< CC1IE (Bitfield-Mask: 0x01)          */
#define TIM_DIER_UIE_Pos                         (0UL)                        /*!< UIE (Bit 0)                          */
#define TIM_DIER_UIE_Msk                         (0x00000001UL)               /*!< UIE (Bitfield-Mask: 0x01)            */
#define TIM_DIER_BIE_Pos                         (7UL)                        /*!< BIE (Bit 7)                          */
#define TIM_DIER_BIE_Msk                         (0x00000080UL)               /*!< BIE (Bitfield-Mask: 0x01)            */
#define TIM_DIER_COMIE_Pos                       (5UL)                        /*!< COMIE (Bit 5)                        */
#define TIM_DIER_COMIE_Msk                       (0x00000020UL)               /*!< COMIE (Bitfield-Mask: 0x01)          */
/* =============================  SR  ======================================= */
#define TIM_SR_CC5IF_Pos                         (16UL)                       /*!< CC5IF (Bit 16)                       */
#define TIM_SR_CC5IF_Msk                         (0x00010000UL)               /*!< CC5IF (Bitfield-Mask: 0x01)          */
#define TIM_SR_CC5OF_Pos                         (13UL)                       /*!< CC5OF (Bit 13)                       */
#define TIM_SR_CC5OF_Msk                         (0x00002000UL)               /*!< CC5OF (Bitfield-Mask: 0x01)          */
#define TIM_SR_CC4OF_Pos                         (12UL)                       /*!< CC4OF (Bit 12)                       */
#define TIM_SR_CC4OF_Msk                         (0x00001000UL)               /*!< CC4OF (Bitfield-Mask: 0x01)          */
#define TIM_SR_CC3OF_Pos                         (11UL)                       /*!< CC3OF (Bit 11)                       */
#define TIM_SR_CC3OF_Msk                         (0x00000800UL)               /*!< CC3OF (Bitfield-Mask: 0x01)          */
#define TIM_SR_CC2OF_Pos                         (10UL)                       /*!< CC2OF (Bit 10)                       */
#define TIM_SR_CC2OF_Msk                         (0x00000400UL)               /*!< CC2OF (Bitfield-Mask: 0x01)          */
#define TIM_SR_CC1OF_Pos                         (9UL)                        /*!< CC1OF (Bit 9)                        */
#define TIM_SR_CC1OF_Msk                         (0x00000200UL)               /*!< CC1OF (Bitfield-Mask: 0x01)          */
#define TIM_SR_BIF_Pos                           (7UL)                        /*!< BIF (Bit 7)                          */
#define TIM_SR_BIF_Msk                           (0x00000080UL)               /*!< BIF (Bitfield-Mask: 0x01)            */
#define TIM_SR_TIF_Pos                           (6UL)                        /*!< TIF (Bit 6)                          */
#define TIM_SR_TIF_Msk                           (0x00000040UL)               /*!< TIF (Bitfield-Mask: 0x01)            */
#define TIM_SR_COMIF_Pos                         (5UL)                        /*!< COMIF (Bit 5)                        */
#define TIM_SR_COMIF_Msk                         (0x00000020UL)               /*!< COMIF (Bitfield-Mask: 0x01)          */
#define TIM_SR_CC4IF_Pos                         (4UL)                        /*!< CC4IF (Bit 4)                        */
#define TIM_SR_CC4IF_Msk                         (0x00000010UL)               /*!< CC4IF (Bitfield-Mask: 0x01)          */
#define TIM_SR_CC3IF_Pos                         (3UL)                        /*!< CC3IF (Bit 3)                        */
#define TIM_SR_CC3IF_Msk                         (0x00000008UL)               /*!< CC3IF (Bitfield-Mask: 0x01)          */
#define TIM_SR_CC2IF_Pos                         (2UL)                        /*!< CC2IF (Bit 2)                        */
#define TIM_SR_CC2IF_Msk                         (0x00000004UL)               /*!< CC2IF (Bitfield-Mask: 0x01)          */
#define TIM_SR_CC1IF_Pos                         (1UL)                        /*!< CC1IF (Bit 1)                        */
#define TIM_SR_CC1IF_Msk                         (0x00000002UL)               /*!< CC1IF (Bitfield-Mask: 0x01)          */
#define TIM_SR_UIF_Pos                           (0UL)                        /*!< UIF (Bit 0)                          */
#define TIM_SR_UIF_Msk                           (0x00000001UL)               /*!< UIF (Bitfield-Mask: 0x01)            */
/* =============================  EGR  ====================================== */
#define TIM_EGR_CC5G_Pos                         (8UL)                        /*!< CC5G (Bit 9)                         */
#define TIM_EGR_CC5G_Msk                         (0x00000200UL)               /*!< CC5G (Bitfield-Mask: 0x01)           */
#define TIM_EGR_B2G_Pos                          (8UL)                        /*!< B2G (Bit 8)                          */
#define TIM_EGR_B2G_Msk                          (0x00000100UL)               /*!< B2G (Bitfield-Mask: 0x01)            */
#define TIM_EGR_BG_Pos                           (7UL)                        /*!< BG (Bit 7)                           */
#define TIM_EGR_BG_Msk                           (0x00000080UL)               /*!< BG (Bitfield-Mask: 0x01)             */
#define TIM_EGR_TG_Pos                           (6UL)                        /*!< TG (Bit 6)                           */
#define TIM_EGR_TG_Msk                           (0x00000040UL)               /*!< TG (Bitfield-Mask: 0x01)             */
#define TIM_EGR_COMG_Pos                         (5UL)                        /*!< COMG (Bit 5)                         */
#define TIM_EGR_COMG_Msk                         (0x00000020UL)               /*!< COMG (Bitfield-Mask: 0x01)           */
#define TIM_EGR_CC4G_Pos                         (4UL)                        /*!< CC4G (Bit 4)                         */
#define TIM_EGR_CC4G_Msk                         (0x00000010UL)               /*!< CC4G (Bitfield-Mask: 0x01)           */
#define TIM_EGR_CC3G_Pos                         (3UL)                        /*!< CC3G (Bit 3)                         */
#define TIM_EGR_CC3G_Msk                         (0x00000008UL)               /*!< CC3G (Bitfield-Mask: 0x01)           */
#define TIM_EGR_CC2G_Pos                         (2UL)                        /*!< CC2G (Bit 2)                         */
#define TIM_EGR_CC2G_Msk                         (0x00000004UL)               /*!< CC2G (Bitfield-Mask: 0x01)           */
#define TIM_EGR_CC1G_Pos                         (1UL)                        /*!< CC1G (Bit 1)                         */
#define TIM_EGR_CC1G_Msk                         (0x00000002UL)               /*!< CC1G (Bitfield-Mask: 0x01)           */
#define TIM_EGR_UG_Pos                           (0UL)                        /*!< UG (Bit 0)                           */
#define TIM_EGR_UG_Msk                           (0x00000001UL)               /*!< UG (Bitfield-Mask: 0x01)             */
/* =========================  CCMR1_Output  ================================= */
#define TIM_CCMR1_Output_OC2TE_Pos               (31UL)                       /*!< OC2TE (Bit 31)                       */
#define TIM_CCMR1_Output_OC2TE_Msk               (0x80000000UL)               /*!< OC2TE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR1_Output_OC2TUE_Pos              (30UL)                       /*!< OC2TUE (Bit 30)                      */
#define TIM_CCMR1_Output_OC2TUE_Msk              (0x40000000UL)               /*!< OC2TUE (Bitfield-Mask: 0x01)         */
#define TIM_CCMR1_Output_OC1TE_Pos               (23UL)                       /*!< OC1TE (Bit 23)                       */
#define TIM_CCMR1_Output_OC1TE_Msk               (0x00800000UL)               /*!< OC1TE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR1_Output_OC1TUE_Pos              (22UL)                       /*!< OC1TUE (Bit 22)                      */
#define TIM_CCMR1_Output_OC1TUE_Msk              (0x00400000UL)               /*!< OC1TUE (Bitfield-Mask: 0x01)         */
#define TIM_CCMR1_Output_OC2CE_Pos               (15UL)                       /*!< OC2CE (Bit 15)                       */
#define TIM_CCMR1_Output_OC2CE_Msk               (0x00008000UL)               /*!< OC2CE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR1_Output_OC2M_Pos                (12UL)                       /*!< OC2M (Bit 12)                        */
#define TIM_CCMR1_Output_OC2M_Msk                (0x00007000UL)               /*!< OC2M (Bitfield-Mask: 0x07)           */
#define TIM_CCMR1_Output_OC2PE_Pos               (11UL)                       /*!< OC2PE (Bit 11)                       */
#define TIM_CCMR1_Output_OC2PE_Msk               (0x00000800UL)               /*!< OC2PE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR1_Output_OC2FE_Pos               (10UL)                       /*!< OC2FE (Bit 10)                       */
#define TIM_CCMR1_Output_OC2FE_Msk               (0x00000400UL)               /*!< OC2FE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR1_Output_CC2S_Pos                (8UL)                        /*!< CC2S (Bit 8)                         */
#define TIM_CCMR1_Output_CC2S_Msk                (0x00000300UL)               /*!< CC2S (Bitfield-Mask: 0x03)           */
#define TIM_CCMR1_Output_OC1CE_Pos               (7UL)                        /*!< OC1CE (Bit 7)                        */
#define TIM_CCMR1_Output_OC1CE_Msk               (0x00000080UL)               /*!< OC1CE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR1_Output_OC1M_Pos                (4UL)                        /*!< OC1M (Bit 4)                         */
#define TIM_CCMR1_Output_OC1M_Msk                (0x00000070UL)               /*!< OC1M (Bitfield-Mask: 0x07)           */
#define TIM_CCMR1_Output_OC1PE_Pos               (3UL)                        /*!< OC1PE (Bit 3)                        */
#define TIM_CCMR1_Output_OC1PE_Msk               (0x00000008UL)               /*!< OC1PE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR1_Output_OC1FE_Pos               (2UL)                        /*!< OC1FE (Bit 2)                        */
#define TIM_CCMR1_Output_OC1FE_Msk               (0x00000004UL)               /*!< OC1FE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR1_Output_CC1S_Pos                (0UL)                        /*!< CC1S (Bit 0)                         */
#define TIM_CCMR1_Output_CC1S_Msk                (0x00000003UL)               /*!< CC1S (Bitfield-Mask: 0x03)           */
/* ========================  CCMR1_Input  =================================== */
#define TIM_CCMR1_Input_IC2F_Pos                 (12UL)                       /*!< IC2F (Bit 12)                        */
#define TIM_CCMR1_Input_IC2F_Msk                 (0x0000f000UL)               /*!< IC2F (Bitfield-Mask: 0x0f)           */
#define TIM_CCMR1_Input_IC2PSC_Pos               (10UL)                       /*!< IC2PCS (Bit 10)                      */
#define TIM_CCMR1_Input_IC2PSC_Msk               (0x00000C00UL)               /*!< IC2PCS (Bitfield-Mask: 0x03)         */
#define TIM_CCMR1_Input_CC2S_Pos                 (8UL)                        /*!< CC2S (Bit 8)                         */
#define TIM_CCMR1_Input_CC2S_Msk                 (0x00000300UL)               /*!< CC2S (Bitfield-Mask: 0x03)           */
#define TIM_CCMR1_Input_IC1F_Pos                 (4UL)                        /*!< IC1F (Bit 4)                         */
#define TIM_CCMR1_Input_IC1F_Msk                 (0x000000F0UL)               /*!< IC1F (Bitfield-Mask: 0x0f)           */
#define TIM_CCMR1_Input_IC1PSC_Pos               (2UL)                        /*!< IC1PSC (Bit 2)                       */
#define TIM_CCMR1_Input_IC1PSC_Msk               (0x0000000CUL)               /*!< IC1PSC (Bitfield-Mask: 0x03)         */
#define TIM_CCMR1_Input_CC1S_Pos                 (0UL)                        /*!< CC1S (Bit 0)                         */
#define TIM_CCMR1_Input_CC1S_Msk                 (0x00000003UL)               /*!< CC1S (Bitfield-Mask: 0x03)           */
/* ========================  CCMR2_Output  ================================== */
#define TIM_CCMR2_Output_OC4TE_Pos               (31UL)                       /*!< OC4TE (Bit 31)                       */
#define TIM_CCMR2_Output_OC4TE_Msk               (0x80000000UL)               /*!< OC4TE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR2_Output_OC4TUE_Pos              (30UL)                       /*!< OC4TUE (Bit 30)                      */
#define TIM_CCMR2_Output_OC4TUE_Msk              (0x40000000UL)               /*!< OC4TUE (Bitfield-Mask: 0x01)         */
#define TIM_CCMR2_Output_OC3TE_Pos               (23UL)                       /*!< OC3TE (Bit 23)                       */
#define TIM_CCMR2_Output_OC3TE_Msk               (0x00800000UL)               /*!< OC3TE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR2_Output_OC3TUE_Pos              (22UL)                       /*!< OC3TUE (Bit 22)                      */
#define TIM_CCMR2_Output_OC3TUE_Msk              (0x00400000UL)               /*!< OC3TUE (Bitfield-Mask: 0x01)         */
#define TIM_CCMR2_Output_OC4CE_Pos               (15UL)                       /*!< OC4CE (Bit 15)                       */
#define TIM_CCMR2_Output_OC4CE_Msk               (0x00008000UL)               /*!< OC4CE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR2_Output_OC4M_Pos                (12UL)                       /*!< OC4M (Bit 12)                        */
#define TIM_CCMR2_Output_OC4M_Msk                (0x00007000UL)               /*!< OC4M (Bitfield-Mask: 0x07)           */
#define TIM_CCMR2_Output_OC4PE_Pos               (11UL)                       /*!< OC4PE (Bit 11)                       */
#define TIM_CCMR2_Output_OC4PE_Msk               (0x00000800UL)               /*!< OC4PE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR2_Output_OC4FE_Pos               (10UL)                       /*!< OC4FE (Bit 10)                       */
#define TIM_CCMR2_Output_OC4FE_Msk               (0x00000400UL)               /*!< OC4FE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR2_Output_CC4S_Pos                (8UL)                        /*!< CC4S (Bit 8)                         */
#define TIM_CCMR2_Output_CC4S_Msk                (0x00000300UL)               /*!< CC4S (Bitfield-Mask: 0x03)           */
#define TIM_CCMR2_Output_OC3CE_Pos               (7UL)                        /*!< OC3CE (Bit 7)                        */
#define TIM_CCMR2_Output_OC3CE_Msk               (0x00000080UL)               /*!< OC3CE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR2_Output_OC3M_Pos                (4UL)                        /*!< OC3M (Bit 4)                         */
#define TIM_CCMR2_Output_OC3M_Msk                (0x00000070UL)               /*!< OC3M (Bitfield-Mask: 0x07)           */
#define TIM_CCMR2_Output_OC3PE_Pos               (3UL)                        /*!< OC3PE (Bit 3)                        */
#define TIM_CCMR2_Output_OC3PE_Msk               (0x00000008UL)               /*!< OC3PE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR2_Output_OC3FE_Pos               (2UL)                        /*!< OC3FE (Bit 2)                        */
#define TIM_CCMR2_Output_OC3FE_Msk               (0x00000004UL)               /*!< OC3FE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR2_Output_CC3S_Pos                (0UL)                        /*!< CC3S (Bit 0)                         */
#define TIM_CCMR2_Output_CC3S_Msk                (0x00000003UL)               /*!< CC3S (Bitfield-Mask: 0x03)           */
/* ==========================  CCMR2_Input  ================================= */
#define TIM_CCMR2_Input_IC4F_Pos                 (12UL)                       /*!< IC4F (Bit 12)                        */
#define TIM_CCMR2_Input_IC4F_Msk                 (0x0000F000UL)               /*!< IC4F (Bitfield-Mask: 0x0f)           */
#define TIM_CCMR2_Input_IC4PSC_Pos               (10UL)                       /*!< IC4PSC (Bit 10)                      */
#define TIM_CCMR2_Input_IC4PSC_Msk               (0x00000C00UL)               /*!< IC4PSC (Bitfield-Mask: 0x03)         */
#define TIM_CCMR2_Input_CC4S_Pos                 (8UL)                        /*!< CC4S (Bit 8)                         */
#define TIM_CCMR2_Input_CC4S_Msk                 (0x00000300UL)               /*!< CC4S (Bitfield-Mask: 0x03)           */
#define TIM_CCMR2_Input_IC3F_Pos                 (4UL)                        /*!< IC3F (Bit 4)                         */
#define TIM_CCMR2_Input_IC3F_Msk                 (0x000000F0UL)               /*!< IC3F (Bitfield-Mask: 0x0f)           */
#define TIM_CCMR2_Input_IC3PSC_Pos               (2UL)                        /*!< IC3PSC (Bit 2)                       */
#define TIM_CCMR2_Input_IC3PSC_Msk               (0x0000000CUL)               /*!< IC3PSC (Bitfield-Mask: 0x03)         */
#define TIM_CCMR2_Input_CC3S_Pos                 (0UL)                        /*!< CC3S (Bit 0)                         */
#define TIM_CCMR2_Input_CC3S_Msk                 (0x00000003UL)               /*!< CC3S (Bitfield-Mask: 0x03)           */
/* ==========================  CCMR3_Output  ================================ */
#define TIM_CCMR3_Output_OC5TE_Pos               (23UL)                       /*!< OC5TE (Bit 23)                       */
#define TIM_CCMR3_Output_OC5TE_Msk               (0x00800000UL)               /*!< OC5TE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR3_Output_OC5TUE_Pos              (22UL)                       /*!< OC5TUE (Bit 22)                      */
#define TIM_CCMR3_Output_OC5TUE_Msk              (0x00400000UL)               /*!< OC5TUE (Bitfield-Mask: 0x01)         */
#define TIM_CCMR3_Output_OC5CE_Pos               (7UL)                        /*!< OC5CE (Bit 7)                        */
#define TIM_CCMR3_Output_OC5CE_Msk               (0x00000080UL)               /*!< OC5CE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR3_Output_OC5M_Pos                (4UL)                        /*!< OC5M (Bit 4)                         */
#define TIM_CCMR3_Output_OC5M_Msk                (0x00000070UL)               /*!< OC5M (Bitfield-Mask: 0x07)           */
#define TIM_CCMR3_Output_OC5PE_Pos               (3UL)                        /*!< OC5PE (Bit 3)                        */
#define TIM_CCMR3_Output_OC5PE_Msk               (0x00000008UL)               /*!< OC5PE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR3_Output_OC5FE_Pos               (2UL)                        /*!< OC5FE (Bit 2)                        */
#define TIM_CCMR3_Output_OC5FE_Msk               (0x00000004UL)               /*!< OC5FE (Bitfield-Mask: 0x01)          */
#define TIM_CCMR3_Output_CC5S_Pos                (0UL)                        /*!< CC5S (Bit 0)                         */
#define TIM_CCMR3_Output_CC5S_Msk                (0x00000003UL)               /*!< CC5S (Bitfield-Mask: 0x03)           */
/* ==========================  CCMR3_Input  ================================= */
#define TIM_CCMR3_Input_IC5F_Pos                 (4UL)                        /*!< IC5F (Bit 4)                         */
#define TIM_CCMR3_Input_IC5F_Msk                 (0x000000F0UL)               /*!< IC5F (Bitfield-Mask: 0x0f)           */
#define TIM_CCMR3_Input_IC5PSC_Pos               (2UL)                        /*!< IC5PSC (Bit 2)                       */
#define TIM_CCMR3_Input_IC5PSC_Msk               (0x0000000CUL)               /*!< IC5PSC (Bitfield-Mask: 0x03)         */
#define TIM_CCMR3_Input_CC5S_Pos                 (0UL)                        /*!< CC5S (Bit 0)                         */
#define TIM_CCMR3_Input_CC5S_Msk                 (0x00000003UL)               /*!< CC5S (Bitfield-Mask: 0x03)           */
/* =============================  CCER  ===================================== */
#define TIM_CCER_CC5NP_Pos                       (19UL)                       /*!< CC5NP (Bit 19)                       */
#define TIM_CCER_CC5NP_Msk                       (0x00080000UL)               /*!< CC5NP (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC5NE_Pos                       (18UL)                       /*!< CC5NE (Bit 18)                       */
#define TIM_CCER_CC5NE_Msk                       (0x00040000UL)               /*!< CC5NE (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC5P_Pos                        (17UL)                       /*!< CC5NP (Bit 17)                       */
#define TIM_CCER_CC5P_Msk                        (0x00020000UL)               /*!< CC5NP (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC5E_Pos                        (16UL)                       /*!< CC5NE (Bit 16)                       */
#define TIM_CCER_CC5E_Msk                        (0x00010000UL)               /*!< CC5NE (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC4NP_Pos                       (15UL)                       /*!< CC4NP (Bit 15)                       */
#define TIM_CCER_CC4NP_Msk                       (0x00008000UL)               /*!< CC4NP (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC4NE_Pos                       (14UL)                       /*!< CC4NE (Bit 14)                       */
#define TIM_CCER_CC4NE_Msk                       (0x00004000UL)               /*!< CC4NE (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC4P_Pos                        (13UL)                       /*!< CC4P (Bit 13)                        */
#define TIM_CCER_CC4P_Msk                        (0x00002000UL)               /*!< CC4P (Bitfield-Mask: 0x01)           */
#define TIM_CCER_CC4E_Pos                        (12UL)                       /*!< CC4E (Bit 12)                        */
#define TIM_CCER_CC4E_Msk                        (0x00001000UL)               /*!< CC4E (Bitfield-Mask: 0x01)           */
#define TIM_CCER_CC3NP_Pos                       (11UL)                       /*!< CC3NP (Bit 11)                       */
#define TIM_CCER_CC3NP_Msk                       (0x00000800UL)               /*!< CC3NP (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC3NE_Pos                       (10UL)                       /*!< CC3NE (Bit 10)                       */
#define TIM_CCER_CC3NE_Msk                       (0x00000400UL)               /*!< CC3NE (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC3P_Pos                        (9UL)                        /*!< CC3P (Bit 9)                         */
#define TIM_CCER_CC3P_Msk                        (0x00000200UL)               /*!< CC3P (Bitfield-Mask: 0x01)           */
#define TIM_CCER_CC3E_Pos                        (8UL)                        /*!< CC3E (Bit 8)                         */
#define TIM_CCER_CC3E_Msk                        (0x00000100UL)               /*!< CC3E (Bitfield-Mask: 0x01)           */
#define TIM_CCER_CC2NP_Pos                       (7UL)                        /*!< CC2NP (Bit 7)                        */
#define TIM_CCER_CC2NP_Msk                       (0x00000080UL)               /*!< CC2NP (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC2NE_Pos                       (6UL)                        /*!< CC2NE (Bit 6)                        */
#define TIM_CCER_CC2NE_Msk                       (0x00000040UL)               /*!< CC2NE (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC2P_Pos                        (5UL)                        /*!< CC2P (Bit 5)                         */
#define TIM_CCER_CC2P_Msk                        (0x00000020UL)               /*!< CC2P (Bitfield-Mask: 0x01)           */
#define TIM_CCER_CC2E_Pos                        (4UL)                        /*!< CC2E (Bit 4)                         */
#define TIM_CCER_CC2E_Msk                        (0x00000010UL)               /*!< CC2E (Bitfield-Mask: 0x01)           */
#define TIM_CCER_CC1NP_Pos                       (3UL)                        /*!< CC1NP (Bit 3)                        */
#define TIM_CCER_CC1NP_Msk                       (0x00000008UL)               /*!< CC1NP (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC1NE_Pos                       (2UL)                        /*!< CC1NE (Bit 2)                        */
#define TIM_CCER_CC1NE_Msk                       (0x00000004UL)               /*!< CC1NE (Bitfield-Mask: 0x01)          */
#define TIM_CCER_CC1P_Pos                        (1UL)                        /*!< CC1P (Bit 1)                         */
#define TIM_CCER_CC1P_Msk                        (0x00000002UL)               /*!< CC1P (Bitfield-Mask: 0x01)           */
#define TIM_CCER_CC1E_Pos                        (0UL)                        /*!< CC1E (Bit 0)                         */
#define TIM_CCER_CC1E_Msk                        (0x00000001UL)               /*!< CC1E (Bitfield-Mask: 0x01)           */
/* =============================  CNT  ====================================== */
#define TIM_CNT_CNT_Pos                          (0UL)                        /*!< CNT (Bit 0)                          */
#define TIM_CNT_CNT_Msk                          (0x0000FFFFUL)               /*!< CNT (Bitfield-Mask: 0xffff)          */
/* =============================  PSC  ====================================== */
#define TIM_PSC_PSC_Pos                          (0UL)                        /*!< PSC (Bit 0)                          */
#define TIM_PSC_PSC_Msk                          (0x0000FFFFUL)               /*!< PSC (Bitfield-Mask: 0xffff)          */
/* =============================  ARR  ====================================== */
#define TIM_ARR_ARR_Pos                          (0UL)                        /*!< ARR (Bit 0)                          */
#define TIM_ARR_ARR_Msk                          (0x0000FFFFUL)               /*!< ARR (Bitfield-Mask: 0xffff)          */
/* =============================  RCR  ====================================== */
#define TIM_RCR_REP_Pos                          (0UL)                        /*!< REP (Bit 0)                          */
#define TIM_RCR_REP_Msk                          (0x000000FFUL)               /*!< REP (Bitfield-Mask: 0xff)            */
/* =============================  CCR1  ===================================== */
#define TIM_CCR1_CCR1_Pos                        (0UL)                        /*!< CCR1 (Bit 0)                         */
#define TIM_CCR1_CCR1_Msk                        (0x0000FFFFUL)               /*!< CCR1 (Bitfield-Mask: 0xffff)         */
/* =============================  CCR2  ===================================== */
#define TIM_CCR2_CCR2_Pos                        (0UL)                        /*!< CCR2 (Bit 0)                         */
#define TIM_CCR2_CCR2_Msk                        (0x0000FFFFUL)               /*!< CCR2 (Bitfield-Mask: 0xffff)         */
/* =============================  CCR3  ===================================== */
#define TIM_CCR3_CCR3_Pos                        (0UL)                        /*!< CCR3 (Bit 0)                         */
#define TIM_CCR3_CCR3_Msk                        (0x0000FFFFUL)               /*!< CCR3 (Bitfield-Mask: 0xffff)         */
/* =============================  CCR4  ===================================== */
#define TIM_CCR4_CCR4_Pos                        (0UL)                        /*!< CCR4 (Bit 0)                         */
#define TIM_CCR4_CCR4_Msk                        (0x0000FFFFUL)               /*!< CCR4 (Bitfield-Mask: 0xffff)         */
/* =============================  CCR5  ===================================== */
#define TIM_CCR5_CCR5_Pos                        (0UL)                        /*!< CCR5 (Bit 0)                         */
#define TIM_CCR5_CCR5_Msk                        (0x0000FFFFUL)               /*!< CCR5 (Bitfield-Mask: 0xffff)         */
/* =============================  DCR  ====================================== */
#define TIM_DCR_DBL_Pos                          (8UL)                        /*!< DBL (Bit 8)                          */
#define TIM_DCR_DBL_Msk                          (0x00001F00UL)               /*!< DBL (Bitfield-Mask: 0x1f)            */
#define TIM_DCR_DSEL_Pos                         (5UL)                        /*!< DBL (Bit 5)                          */
#define TIM_DCR_DSEL_Msk                         (0x000000E0UL)               /*!< DBL (Bitfield-Mask: 0x07)            */
#define TIM_DCR_DBA_Pos                          (0UL)                        /*!< DBA (Bit 0)                          */
#define TIM_DCR_DBA_Msk                          (0x0000001FUL)               /*!< DBA (Bitfield-Mask: 0x1f)            */
/* =============================  DMAR  ===================================== */
#define TIM_DMAR_DMAB_Pos                        (0UL)                        /*!< DMAB (Bit 0)                         */
#define TIM_DMAR_DMAB_Msk                        (0xFFFFFFFFUL)               /*!< DMAB (Bitfield-Mask: 0xffff)         */
/* =============================  BDTR  ===================================== */
#define TIM_BDTR_BKTRIG_Pos                      (27UL)                       /*!< BKTRIG (Bit 27)                      */
#define TIM_BDTR_BKTRIG_Msk                      (0x08000000UL)               /*!< BKTRIG (Bitfield-Mask: 0x01)         */
#define TIM_BDTR_AOE2_Pos                        (26UL)                       /*!< AOE2 (Bit 26)                        */
#define TIM_BDTR_AOE2_Msk                        (0x04000000UL)               /*!< AOE2 (Bitfield-Mask: 0x01)           */
#define TIM_BDTR_BK2P_Pos                        (25UL)                       /*!< BK2P (Bit 25)                        */
#define TIM_BDTR_BK2P_Msk                        (0x02000000UL)               /*!< BK2P (Bitfield-Mask: 0x01)           */
#define TIM_BDTR_BK2E_Pos                        (24UL)                       /*!< BK2E (Bit 24)                        */
#define TIM_BDTR_BK2E_Msk                        (0x01000000UL)               /*!< BK2E (Bitfield-Mask: 0x01)           */
#define TIM_BDTR_MOE_Pos                         (15UL)                       /*!< MOE (Bit 15)                         */
#define TIM_BDTR_MOE_Msk                         (0x00008000UL)               /*!< MOE (Bitfield-Mask: 0x01)            */
#define TIM_BDTR_AOE_Pos                         (14UL)                       /*!< AOE (Bit 14)                         */
#define TIM_BDTR_AOE_Msk                         (0x00004000UL)               /*!< AOE (Bitfield-Mask: 0x01)            */
#define TIM_BDTR_BKP_Pos                         (13UL)                       /*!< BKP (Bit 13)                         */
#define TIM_BDTR_BKP_Msk                         (0x00002000UL)               /*!< BKP (Bitfield-Mask: 0x01)            */
#define TIM_BDTR_BKE_Pos                         (12UL)                       /*!< BKE (Bit 12)                         */
#define TIM_BDTR_BKE_Msk                         (0x00001000UL)               /*!< BKE (Bitfield-Mask: 0x01)            */
#define TIM_BDTR_OSSR_Pos                        (11UL)                       /*!< OSSR (Bit 11)                        */
#define TIM_BDTR_OSSR_Msk                        (0x00000800UL)               /*!< OSSR (Bitfield-Mask: 0x01)           */
#define TIM_BDTR_OSSI_Pos                        (10UL)                       /*!< OSSI (Bit 10)                        */
#define TIM_BDTR_OSSI_Msk                        (0x00000400UL)               /*!< OSSI (Bitfield-Mask: 0x01)           */
#define TIM_BDTR_LOCK_Pos                        (8UL)                        /*!< LOCK (Bit 8)                         */
#define TIM_BDTR_LOCK_Msk                        (0x00000300UL)               /*!< LOCK (Bitfield-Mask: 0x03)           */
#define TIM_BDTR_DTG_Pos                         (0UL)                        /*!< DTG (Bit 0)                          */
#define TIM_BDTR_DTG_Msk                         (0x000000FFUL)               /*!< DTG (Bitfield-Mask: 0xFF)            */
/* ============================  DLAMT  ===================================== */
#define TIM_DLAMT_DLS_Pos                        (17UL)                       /*!< DLS (Bit 17)                         */
#define TIM_DLAMT_DLS_Msk                        (0x00020000UL)               /*!< DLS (Bitfield-Mask: 0x01)            */
#define TIM_DLAMT_AMTDL_Pos                      (16UL)                       /*!< AMTDL (Bit 16)                       */
#define TIM_DLAMT_AMTDL_Msk                      (0x00010000UL)               /*!< AMTDL (Bitfield-Mask: 0x01)          */
#define TIM_DLAMT_ATMG_Pos                       (8UL)                        /*!< ATMG (Bit 8)                         */
#define TIM_DLAMT_ATMG_Msk                       (0x0000FF00UL)               /*!< ATMG (Bitfield-Mask: 0xFF)           */
#define TIM_DLAMT_DLG_Pos                        (0UL)                        /*!< DLG (Bit 0)                          */
#define TIM_DLAMT_DLG_Msk                        (0x000000FFUL)               /*!< DLG (Bitfield-Mask: 0xFF)            */
/* ============================  CCTR1  ===================================== */
#define TIM_CCTR1_Pos                            (0UL)                        /*!< CCTR1 (Bit 0)                        */
#define TIM_CCTR1_Msk                            (0x0000FFFFUL)               /*!< CCTR1 (Bitfield-Mask: 0xFFFF)        */
/* ============================  CCTR2  ===================================== */
#define TIM_CCTR2_Pos                            (0UL)                        /*!< CCTR2 (Bit 0)                        */
#define TIM_CCTR2_Msk                            (0x0000FFFFUL)               /*!< CCTR2 (Bitfield-Mask: 0xFFFF)        */
/* ============================  CCTR3  ===================================== */
#define TIM_CCTR3_Pos                            (0UL)                        /*!< CCTR3 (Bit 0)                        */
#define TIM_CCTR3_Msk                            (0x0000FFFFUL)               /*!< CCTR3 (Bitfield-Mask: 0xFFFF)        */
/* ============================  CCTR4  ===================================== */
#define TIM_CCTR4_Pos                            (0UL)                        /*!< CCTR4 (Bit 0)                        */
#define TIM_CCTR4_Msk                            (0x0000FFFFUL)               /*!< CCTR4 (Bitfield-Mask: 0xFFFF)        */
/* ============================  CCTR5  ===================================== */
#define TIM_CCTR5_Pos                            (0UL)                        /*!< CCTR5 (Bit 0)                        */
#define TIM_CCTR5_Msk                            (0x0000FFFFUL)               /*!< CCTR5 (Bitfield-Mask: 0xFFFF)        */
/* ============================  CCOR1  ===================================== */
#define TIM_CCOR1_Pos                            (0UL)                        /*!< CCOR1 (Bit 0)                        */
#define TIM_CCOR1_Msk                            (0x0000FFFFUL)               /*!< CCOR1 (Bitfield-Mask: 0xFFFF)        */
/* ============================  CCOR2  ===================================== */
#define TIM_CCOR2_Pos                            (0UL)                        /*!< CCOR2 (Bit 0)                        */
#define TIM_CCOR2_Msk                            (0x0000FFFFUL)               /*!< CCOR2 (Bitfield-Mask: 0xFFFF)        */
/* ============================  CCOR3  ===================================== */
#define TIM_CCOR3_Pos                            (0UL)                        /*!< CCOR3 (Bit 0)                        */
#define TIM_CCOR3_Msk                            (0x0000FFFFUL)               /*!< CCOR3 (Bitfield-Mask: 0xFFFF)        */
/* ============================  CCOR4  ===================================== */
#define TIM_CCOR4_Pos                            (0UL)                        /*!< CCOR4 (Bit 0)                        */
#define TIM_CCOR4_Msk                            (0x0000FFFFUL)               /*!< CCOR4 (Bitfield-Mask: 0xFFFF)        */
/* ============================  CCOR5  ===================================== */
#define TIM_CCOR5_Pos                            (0UL)                        /*!< CCOR5 (Bit 0)                        */
#define TIM_CCOR5_Msk                            (0x0000FFFFUL)               /*!< CCOR5 (Bitfield-Mask: 0xFFFF)        */
/* =============================  OCR  ====================================== */
#define TIM_OCR_OC5_MODE_Pos                     (16UL)                       /*!< OC5_MODE (Bit 16)                    */
#define TIM_OCR_OC5_MODE_Msk                     (0x000F0000UL)               /*!< OC5_MODE (Bitfield-Mask: 0xf)        */
#define TIM_OCR_OC4_MODE_Pos                     (12UL)                       /*!< OC4_MODE (Bit 12)                    */
#define TIM_OCR_OC4_MODE_Msk                     (0x0000F000UL)               /*!< OC4_MODE (Bitfield-Mask: 0xf)        */
#define TIM_OCR_OC3_MODE_Pos                     (8UL)                        /*!< OC3_MODE (Bit 8)                     */
#define TIM_OCR_OC3_MODE_Msk                     (0x00000F00UL)               /*!< OC3_MODE (Bitfield-Mask: 0xf)        */
#define TIM_OCR_OC2_MODE_Pos                     (4UL)                        /*!< OC2_MODE (Bit 4)                     */
#define TIM_OCR_OC2_MODE_Msk                     (0x000000F0UL)               /*!< OC2_MODE (Bitfield-Mask: 0xf)        */
#define TIM_OCR_OC1_MODE_Pos                     (0UL)                        /*!< OC1_MODE (Bit 0)                     */
#define TIM_OCR_OC1_MODE_Msk                     (0x0000000fUL)               /*!< OC1_MODE (Bitfield-Mask: 0xf)        */
/* =============================  OMR  ====================================== */
#define TIM_OMR_PWM8_SEL_Pos                     (28UL)                       /*!< PWM8_SEL (Bit 28)                    */
#define TIM_OMR_PWM8_SEL_Msk                     (0xF0000000UL)               /*!< PWM8_SEL (Bitfield-Mask: 0xf)        */
#define TIM_OMR_PWM7_SEL_Pos                     (24UL)                       /*!< PWM7_SEL (Bit 24)                    */
#define TIM_OMR_PWM7_SEL_Msk                     (0x0F000000UL)               /*!< PWM7_SEL (Bitfield-Mask: 0xf)        */
#define TIM_OMR_PWM6_SEL_Pos                     (20UL)                       /*!< PWM6_SEL (Bit 20)                    */
#define TIM_OMR_PWM6_SEL_Msk                     (0x00F00000UL)               /*!< PWM6_SEL (Bitfield-Mask: 0xf)        */
#define TIM_OMR_PWM5_SEL_Pos                     (16UL)                       /*!< PWM5_SEL (Bit 16)                    */
#define TIM_OMR_PWM5_SEL_Msk                     (0x000F0000UL)               /*!< PWM5_SEL (Bitfield-Mask: 0xf)        */
#define TIM_OMR_PWM4_SEL_Pos                     (12UL)                       /*!< PWM4_SEL (Bit 12)                    */
#define TIM_OMR_PWM4_SEL_Msk                     (0x0000F000UL)               /*!< PWM4_SEL (Bitfield-Mask: 0xf)        */
#define TIM_OMR_PWM3_SEL_Pos                     (8UL)                        /*!< PWM3_SEL (Bit 8)                     */
#define TIM_OMR_PWM3_SEL_Msk                     (0x00000F00UL)               /*!< PWM3_SEL (Bitfield-Mask: 0xf)        */
#define TIM_OMR_PWM2_SEL_Pos                     (4UL)                        /*!< PWM2_SEL (Bit 4)                     */
#define TIM_OMR_PWM2_SEL_Msk                     (0x000000F0UL)               /*!< PWM2_SEL (Bitfield-Mask: 0xf)        */
#define TIM_OMR_PWM1_SEL_Pos                     (0UL)                        /*!< PWM1_SEL (Bit 0)                     */
#define TIM_OMR_PWM1_SEL_Msk                     (0x0000000FUL)               /*!< PWM1_SEL (Bitfield-Mask: 0xf)        */

/* ========================================================================== */
/* ================                  END                      =============== */
/* ========================================================================== */


void TIM_DeInit(TIM_TypeDef *TIMx);
void TIM_TimeBaseInit(TIM_TypeDef *TIMx, TIM_TimeBaseInitTypeDef *TIM_TimeBaseInitStruct);
void TIM_TimeBaseStructInit(TIM_TimeBaseInitTypeDef *TIM_TimeBaseInitStruct);
void TIM_PrescalerConfig(TIM_TypeDef *TIMx, uint32_t Prescaler, uint32_t TIM_PSCReloadMode);
void TIM_CounterModeConfig(TIM_TypeDef *TIMx, uint32_t TIM_CounterMode);
void TIM_SetCounter(TIM_TypeDef *TIMx, uint32_t Counter);
void TIM_SetAutoreload(TIM_TypeDef *TIMx, uint32_t Autoreload);
uint32_t TIM_GetCounter(TIM_TypeDef *TIMx);
uint32_t TIM_GetPrescaler(TIM_TypeDef *TIMx);
void TIM_UpdateDisableConfig(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_UpdateRequestConfig(TIM_TypeDef *TIMx, uint32_t TIM_UpdateSource);
void TIM_ARRPreloadConfig(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_SelectOnePulseMode(TIM_TypeDef *TIMx, uint32_t TIM_OPMode);
void TIM_SetClockDivision(TIM_TypeDef *TIMx, uint32_t TIM_ClockDivision);
void TIM_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);

/* Advanced-control timers (TIM1) specific features*******************/
void TIM_BDTRConfig(TIM_TypeDef *TIMx, TIM_BDTRInitTypeDef *TIM_BDTRInitStruct);
void TIM_BDTRStructInit(TIM_BDTRInitTypeDef *TIM_BDTRInitStruct);
void TIM_CtrlPWMOutputs(TIM_TypeDef *TIMx, FunctionalState NewState);

/* Output Compare management **************************************************/
void TIM_OC1Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct);
void TIM_OC2Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct);
void TIM_OC3Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct);
void TIM_OC4Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct);
void TIM_OC5Init(TIM_TypeDef *TIMx, TIM_OCInitTypeDef *TIM_OCInitStruct);
void TIM_OCStructInit(TIM_OCInitTypeDef *TIM_OCInitStruct);
void TIM_SelectOCxM(TIM_TypeDef *TIMx, uint32_t TIM_Channel, uint32_t TIM_OCMode);
void TIM_SetCompare1(TIM_TypeDef *TIMx, uint32_t Compare1);
void TIM_SetCompare2(TIM_TypeDef *TIMx, uint32_t Compare2);
void TIM_SetCompare3(TIM_TypeDef *TIMx, uint32_t Compare3);
void TIM_SetCompare4(TIM_TypeDef *TIMx, uint32_t Compare4);
void TIM_SetCompare5(TIM_TypeDef *TIMx, uint32_t Compare5);
void TIM_ForcedOC1Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction);
void TIM_ForcedOC2Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction);
void TIM_ForcedOC3Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction);
void TIM_ForcedOC4Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction);
void TIM_ForcedOC5Config(TIM_TypeDef *TIMx, uint32_t TIM_ForcedAction);
void TIM_CCPreloadControl(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OC1PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload);
void TIM_OC2PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload);
void TIM_OC3PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload);
void TIM_OC4PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload);
void TIM_OC5PreloadConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPreload);
void TIM_OC1FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast);
void TIM_OC2FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast);
void TIM_OC3FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast);
void TIM_OC4FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast);
void TIM_OC5FastConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCFast);
void TIM_ClearOC1Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear);
void TIM_ClearOC2Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear);
void TIM_ClearOC3Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear);
void TIM_ClearOC4Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear);
void TIM_ClearOC5Ref(TIM_TypeDef *TIMx, uint32_t TIM_OCClear);
void TIM_OC1PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity);
void TIM_OC1NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity);
void TIM_OC2PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity);
void TIM_OC2NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity);
void TIM_OC3PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity);
void TIM_OC3NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity);
void TIM_OC4PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity);
void TIM_OC4NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity);
void TIM_OC5PolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCPolarity);
void TIM_OC5NPolarityConfig(TIM_TypeDef *TIMx, uint32_t TIM_OCNPolarity);
void TIM_SelectOCREFClear(TIM_TypeDef *TIMx, uint32_t TIM_OCReferenceClear);
void TIM_CCxCmd(TIM_TypeDef *TIMx, uint32_t TIM_Channel, uint32_t TIM_CCx);
void TIM_CCxNCmd(TIM_TypeDef *TIMx, uint32_t TIM_Channel, uint32_t TIM_CCxN);
void TIM_SelectCOM(TIM_TypeDef *TIMx, FunctionalState NewState);

/* Input Capture management ***************************************************/
void TIM_ICInit(TIM_TypeDef *TIMx, TIM_ICInitTypeDef *TIM_ICInitStruct);
void TIM_ICStructInit(TIM_ICInitTypeDef *TIM_ICInitStruct);
void TIM_PWMIConfig(TIM_TypeDef *TIMx, TIM_ICInitTypeDef *TIM_ICInitStruct);
uint32_t TIM_GetCapture1(TIM_TypeDef *TIMx);
uint32_t TIM_GetCapture2(TIM_TypeDef *TIMx);
uint32_t TIM_GetCapture3(TIM_TypeDef *TIMx);
uint32_t TIM_GetCapture4(TIM_TypeDef *TIMx);
uint32_t TIM_GetCapture5(TIM_TypeDef *TIMx);
void TIM_SetIC1Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC);
void TIM_SetIC2Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC);
void TIM_SetIC3Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC);
void TIM_SetIC4Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC);
void TIM_SetIC5Prescaler(TIM_TypeDef *TIMx, uint32_t TIM_ICPSC);

/* Interrupts, DMA and flags management ***************************************/
void TIM_ITConfig(TIM_TypeDef *TIMx, uint32_t TIM_IT, FunctionalState NewState);
void TIM_GenerateEvent(TIM_TypeDef *TIMx, uint32_t TIM_EventSource);
FlagStatus TIM_GetFlagStatus(TIM_TypeDef *TIMx, uint32_t TIM_FLAG);
void TIM_ClearFlag(TIM_TypeDef *TIMx, uint32_t TIM_FLAG);
ITStatus TIM_GetITStatus(TIM_TypeDef *TIMx, uint32_t TIM_IT);
void TIM_ClearITPendingBit(TIM_TypeDef *TIMx, uint32_t TIM_IT);
void TIM_DMAConfig(TIM_TypeDef *TIMx, uint32_t TIM_DMABase, uint32_t TIM_DMABurstLength);
void TIM_DMACmd(TIM_TypeDef *TIMx, uint32_t TIM_DMASource, FunctionalState NewState);
void TIM_WriteDMAR(TIM_TypeDef *TIMx, uint32_t TIM_DMAB);
uint32_t TIM_ReadDMAR(TIM_TypeDef *TIMx);
void TIM_SelectCCDMA(TIM_TypeDef *TIMx, FunctionalState NewState);

/* Clocks management **********************************************************/
void TIM_InternalClockConfig(TIM_TypeDef *TIMx);
void TIM_ITRxExternalClockConfig(TIM_TypeDef *TIMx, uint32_t TIM_InputTriggerSource);
void TIM_TIxExternalClockConfig(TIM_TypeDef *TIMx, uint32_t TIM_TIxExternalCLKSource,
                                uint32_t TIM_ICPolarity, uint32_t ICFilter);
void TIM_ETRClockMode1Config(TIM_TypeDef *TIMx, uint32_t TIM_ExtTRGPrescaler, uint32_t TIM_ExtTRGPolarity,
                             uint32_t ExtTRGFilter);
void TIM_ETRClockMode2Config(TIM_TypeDef *TIMx, uint32_t TIM_ExtTRGPrescaler,
                             uint32_t TIM_ExtTRGPolarity, uint32_t ExtTRGFilter);

/* Synchronization management *************************************************/
void TIM_SelectInputTrigger(TIM_TypeDef *TIMx, uint32_t TIM_InputTriggerSource);
void TIM_SelectOutputTrigger(TIM_TypeDef *TIMx, uint32_t TIM_TRGOSource);
void TIM_SelectSlaveMode(TIM_TypeDef *TIMx, uint32_t TIM_SlaveMode);
void TIM_SelectMasterSlaveMode(TIM_TypeDef *TIMx, uint32_t TIM_MasterSlaveMode);
void TIM_ETRConfig(TIM_TypeDef *TIMx, uint32_t TIM_ExtTRGPrescaler, uint32_t TIM_ExtTRGPolarity,
                   uint32_t ExtTRGFilter);

/* Specific interface management **********************************************/
void TIM_EncoderInterfaceConfig(TIM_TypeDef *TIMx, uint32_t TIM_EncoderMode,
                                uint32_t TIM_IC1Polarity, uint32_t TIM_IC2Polarity);
void TIM_SelectHallSensor(TIM_TypeDef *TIMx, FunctionalState NewState);

/* Specific remapping management **********************************************/

void TIM_DTS_Cfg(TIM_TypeDef *TIMx);
void TIM_TI1E_Set(TIM_TypeDef *TIMx);
void TIM_TI2E_Set(TIM_TypeDef *TIMx);
void TIM_TI3E_Set(TIM_TypeDef *TIMx);
void TIM_TI4E_Set(TIM_TypeDef *TIMx);
void TIM_TI5E_Set(TIM_TypeDef *TIMx);
void TIM_ETRE_Enable(TIM_TypeDef *TIMx);
/*SYSCTRL_EDU_CFGx confg****************************************************/
void TIM1_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource);
void TIM2_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource);
void TIM15_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource);
void TIM16_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource);
void TIM17_IntputSourceCfg(uint32_t TIx, uint32_t TIM_IntputSource);

void TIM1_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState);
void TIM1_OCREF_CLR_SourceCfg(uint32_t TIM_OCREF_CLR_Source, uint32_t TIM_Level, FunctionalState NewState);
void TIM15_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState);
void TIM17_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState);
void TIM16_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState);
void TIM2_BreakSourceCfg(uint32_t TIM_BreakSource, uint32_t TIM_Level, FunctionalState NewState);
void TIM2_OCREF_CLR_SourceCfg(uint32_t TIM_OCREF_CLR_Source, uint32_t TIM_Level, FunctionalState NewState);

void TIM1_PWMxSel_Config (uint32_t PWM_Num, uint32_t OCx_Sel);
void TIM1_CompareOffsetMode (uint32_t TIM_Channel, uint32_t OCx_Mode);

void TIM1_SetCompareOffset1(uint32_t CompareOffset);
void TIM1_SetCompareOffset2(uint32_t CompareOffset);
void TIM1_SetCompareOffset3(uint32_t CompareOffset);
void TIM1_SetCompareOffset4(uint32_t CompareOffset);
void TIM1_SetCompareOffset5(uint32_t CompareOffset);
uint32_t TIM_GetCaptureOffset1(TIM_TypeDef *TIMx);
uint32_t TIM_GetCaptureOffset2(TIM_TypeDef *TIMx);
uint32_t TIM_GetCaptureOffset3(TIM_TypeDef *TIMx);
uint32_t TIM_GetCaptureOffset4(TIM_TypeDef *TIMx);
uint32_t TIM_GetCaptureOffset5(TIM_TypeDef *TIMx);
void TIM_SetCompareTuning1(TIM_TypeDef *TIMx, uint32_t Tuning);
void TIM_SetCompareTuning2(TIM_TypeDef *TIMx, uint32_t Tuning);
void TIM_SetCompareTuning3(TIM_TypeDef *TIMx, uint32_t Tuning);
void TIM_SetCompareTuning4(TIM_TypeDef *TIMx, uint32_t Tuning);
void TIM_SetCompareTuning5(TIM_TypeDef *TIMx, uint32_t Tuning);
uint32_t TIM_GetCompareTuning1(TIM_TypeDef *TIMx);
uint32_t TIM_GetCompareTuning2(TIM_TypeDef *TIMx);
uint32_t TIM_GetCompareTuning3(TIM_TypeDef *TIMx);
uint32_t TIM_GetCompareTuning4(TIM_TypeDef *TIMx);
uint32_t TIM_GetCompareTuning5(TIM_TypeDef *TIMx);

void TIM_BreakTrigOci_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);

void TIM_OCTuning1_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OCTuning2_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OCTuning3_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OCTuning4_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OCTuning5_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OCTuning1Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OCTuning2Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OCTuning3Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OCTuning4Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_OCTuning5Update_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_TrigDelayClear_Cmd(TIM_TypeDef *TIMx, FunctionalState NewState);
void TIM_AntiMisoperateTrig_Config(TIM_TypeDef *TIMx, uint32_t AntiMisoperateStart, uint8_t AntiMisoperateTime);
void TIM_TrigDelay_Config(TIM_TypeDef *TIMx, uint8_t TrigDelayTime);
#ifdef __cplusplus
}
#endif

#endif // __lCM32F06X_TIM_H__
