/**
  ******************************************************************************
  * @file    lcm32f06x_rcc.h 
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   Header for rcc.c module
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __LCM32F06X_RCC_H__
#define __LCM32F06X_RCC_H__

/* Includes ------------------------------------------------------------------*/
#include "lcm32f06x.h"

#ifdef __cplusplus
  extern "C" {
#endif

#define DELAY_10US_COUNT          (uint32_t)(0x1000000 - SYSTEM_CORE_CLOCK/100000)

typedef struct
{
  uint32_t HSI_Frequency;
  uint32_t LSI_Frequency;
  uint32_t LSE_Frequency;
  uint32_t SYSCLK_IN_Frequency; // system clock frequency before prescale
  uint32_t SYSCLK_Frequency;    // system clock frqeuency
  uint32_t HCLK_Frequency;      // AHB clock frequency
  uint32_t PCLK0_Frequency;     // APB0 clock frequency
  uint32_t PLL_Frequency;       // PLL clock frequency;
  uint32_t USART0_Frequency;     // APB1 bridge.
  uint32_t USART1_Frequency;     // APB1 bridge.
  uint32_t I2C0_Frequency;      // APB1 bridge.

  uint32_t SPI0_Frequency; // APB1 bridge.

} RCC_ClocksTypeDef;

/* @defgroup RCC_LSE */
#define RCC_LSE_OFF                            ((uint32_t)0x0000) /*@EN:turn OFF the LSE oscillator, LSERDY flag goes low after 6 LSE oscillator clock cycles @CN:OSCL ر*/
#define RCC_LSE_ON                             ((uint32_t)0x0001) /*@EN:turn ON the LSE oscillator clock @CN: OSCL  */
#define RCC_LSE_Bypass                         ((uint32_t)0x0002) /*@EN:LSE oscillator bypassed with external clock @CN: OSCL·  */
#define IS_RCC_LSE(RCC_LSE)                    ((RCC_LSE == RCC_LSE_OFF) || (RCC_LSE == RCC_LSE_ON) || (RCC_LSE == RCC_LSE_Bypass))
/* @end */

/* @defgroup RCC_AHBPeriph */
#define RCC_AHBPeriph_RAM1                     ((uint32_t)0x00040000) /*@EN: AHB bus RAM1 clock @CN: AHB RAM1 ʱ */
#define RCC_AHBPeriph_RAM0                     ((uint32_t)0x00020000) /*@EN: AHB bus RAM0 clock @CN: AHB RAM0 ʱ */
#define RCC_AHBPeriph_EFLS                     ((uint32_t)0x00010000) /*@EN: AHB bus FLASH clock @CN: AHB FLASH ʱ */
#define RCC_AHBPeriph_GPIOA                    ((uint32_t)0x00000001) /*@EN: AHB bus GPIOA clock @CN: AHB GPIOA ʱ */
#define RCC_AHBPeriph_GPIOB                    ((uint32_t)0x00000002) /*@EN: AHB bus GPIOB clock @CN: AHB GPIOB ʱ */
#define RCC_AHBPeriph_CRC                      ((uint32_t)0x00000040) /*@EN: AHB bus CRC clock @CN: AHB CRC ʱ */
#define RCC_AHBPeriph_DMA                      ((uint32_t)0x00000100) /*@EN: AHB bus DMA clock @CN: AHB DMA ʱ */
#define RCC_AHBPeriph_DIV                      ((uint32_t)0x00000080) /*@EN: AHB bus DIV clock @CN: AHB DIV ʱ */

#define IS_RCC_AHB_RST_PERIPH(RCC_AHBPeriph)   ((RCC_AHBPeriph == RCC_AHBPeriph_RAM1)  || \
                                                (RCC_AHBPeriph == RCC_AHBPeriph_RAM0)  || \
                                                (RCC_AHBPeriph == RCC_AHBPeriph_EFLS)  || \
                                                (RCC_AHBPeriph == RCC_AHBPeriph_GPIOA) || \
                                                (RCC_AHBPeriph == RCC_AHBPeriph_GPIOB) || \
                                                (RCC_AHBPeriph == RCC_AHBPeriph_CRC)   || \
                                                (RCC_AHBPeriph == RCC_AHBPeriph_DMA)   || \
                                                (RCC_AHBPeriph == RCC_AHBPeriph_DIV))
#define IS_RCC_AHB_PERIPH(RCC_AHBPeriph)       ((RCC_AHBPeriph == RCC_AHBPeriph_RAM1)  || \
                                                 (RCC_AHBPeriph == RCC_AHBPeriph_RAM0)  || \
                                                 (RCC_AHBPeriph == RCC_AHBPeriph_EFLS)  || \
                                                 (RCC_AHBPeriph == RCC_AHBPeriph_GPIOA) || \
                                                 (RCC_AHBPeriph == RCC_AHBPeriph_GPIOB) || \
                                                 (RCC_AHBPeriph == RCC_AHBPeriph_CRC)   || \
                                                 (RCC_AHBPeriph == RCC_AHBPeriph_DMA)   || \
                                                 (RCC_AHBPeriph == RCC_AHBPeriph_DIV))
/* @end */

/* @defgroup RCC_APB0Periph */
#define RCC_APB0Periph_TIM2                    ((uint32_t)0x80000000) /*@EN: APB0 bus TIM2 clock @CN: APB0 TIM2 ʱ */
#define RCC_APB0Periph_TIM17                   ((uint32_t)0x40000000) /*@EN: APB0 bus TIM17 clock @CN: APB0 TIM17 ʱ */
#define RCC_APB0Periph_TIM16                   ((uint32_t)0x20000000) /*@EN: APB0 bus TIM16 clock @CN: APB0 TIM16 ʱ */
#define RCC_APB0Periph_TIM15                   ((uint32_t)0x10000000) /*@EN: APB0 bus TIM15 clock @CN: APB0 TIM15 ʱ */
#define RCC_APB0Periph_TIM1                    ((uint32_t)0x00080000) /*@EN: APB0 bus TIM1 clock @CN: APB0 TIM1 ʱ */
#define RCC_APB0Periph_USART1                  ((uint32_t)0x00004000) /*@EN: APB0 bus USART1 clock @CN: APB0 USART1 ʱ */
#define RCC_APB0Periph_USART0                  ((uint32_t)0x00002000) /*@EN: APB0 bus USART0 clock @CN: APB0 USART0 ʱ */
#define RCC_APB0Periph_SPI0                    ((uint32_t)0x00001000) /*@EN: APB0 bus SPI0 clock @CN: APB0 SPI0 ʱ */
#define RCC_APB0Periph_I2C0                    ((uint32_t)0x00000400) /*@EN: APB0 bus I2C0 clock @CN: APB0 I2C0 ʱ */
#define RCC_APB0Periph_ADC                     ((uint32_t)0x00000200) /*@EN: APB0 bus ADC clock @CN: APB0 ADC ʱ */
#define RCC_APB0Periph_ANACTRL                 ((uint32_t)0x00000020) /*@EN: APB0 bus ANACTRL clock @CN: APB0 ANACTRL ʱ */
#define RCC_APB0Periph_EXTI                    ((uint32_t)0x00000040) /*@EN: APB0 bus EXTI clock @CN: APB0 EXTI ʱ */
#define RCC_APB0Periph_CORDIC                  ((uint32_t)0x00000080) /*@EN: APB0 bus CORDIC clock @CN: APB0 CORDIC ʱ */

#define IS_RCC_APB0_PERIPH(RCC_APB0Periph)     ((RCC_APB0Periph == RCC_APB0Periph_TIM2) || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_TIM17)  || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_TIM16)  || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_TIM15)  || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_TIM1)   || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_LCD)    || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_USART1) || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_SPI0)   || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_I2C0)   || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_ADC)    || \
                                                 (RCC_APB0Periph == RCC_APB0Periph_ANACTRL)|| \
                                                 (RCC_APB0Periph == RCC_APB0Periph_EXTI))
/* @end */

/* @defgroup RCC_MCOSource */
#define RCC_MCOSource_NoClock                  ((uint8_t)0x0000) /*@EN: Do not output clock (default) @CN: ʱӣĬϣ */
#define RCC_MCOSource_PLL                      ((uint8_t)0x0001) /*@EN: PLL_CLK @CN: PLLʱ */
#define RCC_MCOSource_PLL_CLK_IN               ((uint8_t)0x0002) /*@EN: SYS_CLK_IN @CN:  SYS CLK INʱ */
#define RCC_MCOSource_SYSCLK                   ((uint8_t)0x0003) /*@EN: SYS_CLK @CN: SYS CLKʱ */
#define RCC_MCOSource_LSI                      ((uint8_t)0x0004) /*@EN: RCL_CLK @CN: RCLʱ */
#define RCC_MCOSource_LSE                      ((uint8_t)0x0005) /*@EN: OSCL_CLK @CN: OSCLʱ */
#define RCC_MCOSource_HSI                      ((uint8_t)0x0006) /*@EN: RCH_CLK @CN: RCHʱ */
#define IS_RCC_MCO_SOURCE(RCC_MCOSource)       ((RCC_MCOSource == RCC_MCOSource_NoClock) ||    \
                                                 (RCC_MCOSource == RCC_MCOSource_PLL) ||        \
                                                 (RCC_MCOSource == RCC_MCOSource_PLL_CLK_IN) || \
                                                 (RCC_MCOSource == RCC_MCOSource_SYSCLK) ||     \
                                                 (RCC_MCOSource == RCC_MCOSource_LSI) ||        \
                                                 (RCC_MCOSource == RCC_MCOSource_LSE) ||        \
                                                 (RCC_MCOSource == RCC_MCOSource_HSI))
/* @end */

/* @defgroup RCC_MCOPrescaler */
#define RCC_MCOPrescaler_1                     ((uint8_t)0x0000) /*@EN: 1 division frequency (default) @CN: 1Ƶ Ĭϣ */
#define RCC_MCOPrescaler_2                     ((uint8_t)0x0001) /*@EN: 2 division frequency @CN: 2Ƶ */
#define RCC_MCOPrescaler_4                     ((uint8_t)0x0002) /*@EN: 4 division frequency @CN: 4Ƶ */
#define RCC_MCOPrescaler_8                     ((uint8_t)0x0003) /*@EN: 8 division frequency @CN: 8Ƶ */
#define RCC_MCOPrescaler_16                    ((uint8_t)0x0004) /*@EN: 16 division frequency @CN: 16Ƶ */
#define RCC_MCOPrescaler_32                    ((uint8_t)0x0005) /*@EN: 32 division frequency @CN: 32Ƶ */
#define RCC_MCOPrescaler_64                    ((uint8_t)0x0006) /*@EN: 64 division frequency @CN: 64Ƶ */
#define RCC_MCOPrescaler_128                   ((uint8_t)0x0007) /*@EN: 128 division frequency @CN: 128Ƶ */
#define IS_RCC_MCO_PRESCALER(RCC_MCOPrescaler) ((RCC_MCOPrescaler == RCC_MCOPrescaler_1) ||  \
                                                (RCC_MCOPrescaler == RCC_MCOPrescaler_2) ||  \
                                                (RCC_MCOPrescaler == RCC_MCOPrescaler_4) ||  \
                                                (RCC_MCOPrescaler == RCC_MCOPrescaler_8) ||  \
                                                (RCC_MCOPrescaler == RCC_MCOPrescaler_16) || \
                                                (RCC_MCOPrescaler == RCC_MCOPrescaler_32) || \
                                                (RCC_MCOPrescaler == RCC_MCOPrescaler_64) || \
                                                (RCC_MCOPrescaler == RCC_MCOPrescaler_128))
/* @end */

/* @defgroup RCC_SYSCLKSource */
#define RCC_SYSCLKSource_HSI                   ((uint32_t)0x00000000) /*@EN: HSI selected as system clock source @CN: ѡHSIΪϵͳʱԴ */
#define RCC_SYSCLKSource_PLLCLK                ((uint32_t)0x00200000) /*@EN: PLL selected as system clock source @CN: ѡPLLΪϵͳʱԴ */
#define RCC_SYSCLKSource_LSI                   ((uint32_t)0x00300000) /*@EN: LSI selected as system clock source @CN: ѡLSIΪϵͳʱԴ */
#define IS_RCC_SYSCLK_SOURCE(SOURCE)           (((SOURCE) == RCC_SYSCLKSource_HSI)    || \
                                                ((SOURCE) == RCC_SYSCLKSource_PLLCLK) || \
                                                ((SOURCE) == RCC_SYSCLKSource_LSI))
/* @end */

/* @defgroup RCC_SYSCLK_Div */
#define RCC_SYSCLK_Div1         ((uint32_t)0x00000000) /*@EN: clock division 1   @CN: ʱӷƵ1 */
#define RCC_SYSCLK_Div2         ((uint32_t)0x00000040) /*@EN: clock division 2   @CN: ʱӷƵ2 */
#define RCC_SYSCLK_Div3         ((uint32_t)0x00000080) /*@EN: clock division 3   @CN: ʱӷƵ3 */
#define RCC_SYSCLK_Div4         ((uint32_t)0x00000041) /*@EN: clock division 4   @CN: ʱӷƵ4 */
#define RCC_SYSCLK_Div5         ((uint32_t)0x000000C0) /*@EN: clock division 5   @CN: ʱӷƵ5 */
#define RCC_SYSCLK_Div6         ((uint32_t)0x00000042) /*@EN: clock division 6   @CN: ʱӷƵ6 */
#define RCC_SYSCLK_Div8         ((uint32_t)0x00000043) /*@EN: clock division 8   @CN: ʱӷƵ8 */
#define RCC_SYSCLK_Div10        ((uint32_t)0x00000044) /*@EN: clock division 10  @CN: ʱӷƵ10 */
#define RCC_SYSCLK_Div12        ((uint32_t)0x00000045) /*@EN: clock division 12  @CN: ʱӷƵ12 */
#define RCC_SYSCLK_Div14        ((uint32_t)0x00000046) /*@EN: clock division 14  @CN: ʱӷƵ14 */
#define RCC_SYSCLK_Div16        ((uint32_t)0x00000047) /*@EN: clock division 16  @CN: ʱӷƵ16 */
#define RCC_SYSCLK_Div18        ((uint32_t)0x00000048) /*@EN: clock division 18  @CN: ʱӷƵ18 */
#define RCC_SYSCLK_Div20        ((uint32_t)0x00000049) /*@EN: clock division 20  @CN: ʱӷƵ20 */
#define RCC_SYSCLK_Div22        ((uint32_t)0x0000004A) /*@EN: clock division 22  @CN: ʱӷƵ22 */
#define RCC_SYSCLK_Div24        ((uint32_t)0x0000004B) /*@EN: clock division 24  @CN: ʱӷƵ24 */
#define RCC_SYSCLK_Div26        ((uint32_t)0x0000004C) /*@EN: clock division 26  @CN: ʱӷƵ26 */
#define RCC_SYSCLK_Div28        ((uint32_t)0x0000004D) /*@EN: clock division 28  @CN: ʱӷƵ28 */
#define RCC_SYSCLK_Div30        ((uint32_t)0x0000004E) /*@EN: clock division 30  @CN: ʱӷƵ30 */
#define RCC_SYSCLK_Div32        ((uint32_t)0x0000004F) /*@EN: clock division 32  @CN: ʱӷƵ32 */
#define RCC_SYSCLK_Div34        ((uint32_t)0x00000050) /*@EN: clock division 34  @CN: ʱӷƵ34 */
#define RCC_SYSCLK_Div36        ((uint32_t)0x00000051) /*@EN: clock division 36  @CN: ʱӷƵ36 */
#define RCC_SYSCLK_Div38        ((uint32_t)0x00000052) /*@EN: clock division 38  @CN: ʱӷƵ38 */
#define RCC_SYSCLK_Div40        ((uint32_t)0x00000053) /*@EN: clock division 40  @CN: ʱӷƵ40 */
#define RCC_SYSCLK_Div42        ((uint32_t)0x00000054) /*@EN: clock division 42  @CN: ʱӷƵ42 */
#define RCC_SYSCLK_Div44        ((uint32_t)0x00000055) /*@EN: clock division 44  @CN: ʱӷƵ44 */
#define RCC_SYSCLK_Div46        ((uint32_t)0x00000056) /*@EN: clock division 46  @CN: ʱӷƵ46 */
#define RCC_SYSCLK_Div48        ((uint32_t)0x00000057) /*@EN: clock division 48  @CN: ʱӷƵ48 */
#define RCC_SYSCLK_Div50        ((uint32_t)0x00000058) /*@EN: clock division 50  @CN: ʱӷƵ50 */
#define RCC_SYSCLK_Div52        ((uint32_t)0x00000059) /*@EN: clock division 52  @CN: ʱӷƵ52 */
#define RCC_SYSCLK_Div54        ((uint32_t)0x0000005A) /*@EN: clock division 54  @CN: ʱӷƵ54 */
#define RCC_SYSCLK_Div56        ((uint32_t)0x0000005B) /*@EN: clock division 56  @CN: ʱӷƵ56 */
#define RCC_SYSCLK_Div58        ((uint32_t)0x0000005C) /*@EN: clock division 58  @CN: ʱӷƵ58 */
#define RCC_SYSCLK_Div60        ((uint32_t)0x0000005D) /*@EN: clock division 60  @CN: ʱӷƵ60 */
#define RCC_SYSCLK_Div62        ((uint32_t)0x0000005E) /*@EN: clock division 62  @CN: ʱӷƵ62 */
#define RCC_SYSCLK_Div64        ((uint32_t)0x0000005F) /*@EN: clock division 64  @CN: ʱӷƵ64 */
#define RCC_SYSCLK_Div66        ((uint32_t)0x00000060) /*@EN: clock division 66  @CN: ʱӷƵ66 */
#define RCC_SYSCLK_Div68        ((uint32_t)0x00000061) /*@EN: clock division 68  @CN: ʱӷƵ68 */
#define RCC_SYSCLK_Div70        ((uint32_t)0x00000062) /*@EN: clock division 70  @CN: ʱӷƵ70 */
#define RCC_SYSCLK_Div72        ((uint32_t)0x00000063) /*@EN: clock division 72  @CN: ʱӷƵ72 */
#define RCC_SYSCLK_Div74        ((uint32_t)0x00000064) /*@EN: clock division 74  @CN: ʱӷƵ74 */
#define RCC_SYSCLK_Div76        ((uint32_t)0x00000065) /*@EN: clock division 76  @CN: ʱӷƵ76 */
#define RCC_SYSCLK_Div78        ((uint32_t)0x00000066) /*@EN: clock division 78  @CN: ʱӷƵ78 */
#define RCC_SYSCLK_Div80        ((uint32_t)0x00000067) /*@EN: clock division 80  @CN: ʱӷƵ80 */
#define RCC_SYSCLK_Div82        ((uint32_t)0x00000068) /*@EN: clock division 82  @CN: ʱӷƵ82 */
#define RCC_SYSCLK_Div84        ((uint32_t)0x00000069) /*@EN: clock division 84  @CN: ʱӷƵ84 */
#define RCC_SYSCLK_Div86        ((uint32_t)0x0000006A) /*@EN: clock division 86  @CN: ʱӷƵ86 */
#define RCC_SYSCLK_Div88        ((uint32_t)0x0000006B) /*@EN: clock division 88  @CN: ʱӷƵ88 */
#define RCC_SYSCLK_Div90        ((uint32_t)0x0000006C) /*@EN: clock division 90  @CN: ʱӷƵ90 */
#define RCC_SYSCLK_Div92        ((uint32_t)0x0000006D) /*@EN: clock division 92  @CN: ʱӷƵ92 */
#define RCC_SYSCLK_Div94        ((uint32_t)0x0000006E) /*@EN: clock division 94  @CN: ʱӷƵ94 */
#define RCC_SYSCLK_Div96        ((uint32_t)0x0000006F) /*@EN: clock division 96  @CN: ʱӷƵ96 */
#define RCC_SYSCLK_Div98        ((uint32_t)0x00000070) /*@EN: clock division 98  @CN: ʱӷƵ98 */
#define RCC_SYSCLK_Div100       ((uint32_t)0x00000071) /*@EN: clock division 100 @CN: ʱӷƵ100 */
#define RCC_SYSCLK_Div102       ((uint32_t)0x00000072) /*@EN: clock division 102 @CN: ʱӷƵ102 */
#define RCC_SYSCLK_Div104       ((uint32_t)0x00000073) /*@EN: clock division 104 @CN: ʱӷƵ104 */
#define RCC_SYSCLK_Div106       ((uint32_t)0x00000074) /*@EN: clock division 106 @CN: ʱӷƵ106 */
#define RCC_SYSCLK_Div108       ((uint32_t)0x00000075) /*@EN: clock division 108 @CN: ʱӷƵ108 */
#define RCC_SYSCLK_Div110       ((uint32_t)0x00000076) /*@EN: clock division 110 @CN: ʱӷƵ110 */
#define RCC_SYSCLK_Div112       ((uint32_t)0x00000077) /*@EN: clock division 112 @CN: ʱӷƵ112 */
#define RCC_SYSCLK_Div114       ((uint32_t)0x00000078) /*@EN: clock division 114 @CN: ʱӷƵ114 */
#define RCC_SYSCLK_Div116       ((uint32_t)0x00000079) /*@EN: clock division 116 @CN: ʱӷƵ116 */
#define RCC_SYSCLK_Div118       ((uint32_t)0x0000007A) /*@EN: clock division 118 @CN: ʱӷƵ118 */
#define RCC_SYSCLK_Div120       ((uint32_t)0x0000007B) /*@EN: clock division 120 @CN: ʱӷƵ120 */
#define RCC_SYSCLK_Div122       ((uint32_t)0x0000007C) /*@EN: clock division 122 @CN: ʱӷƵ122 */
#define RCC_SYSCLK_Div124       ((uint32_t)0x0000007D) /*@EN: clock division 124 @CN: ʱӷƵ124 */
#define RCC_SYSCLK_Div126       ((uint32_t)0x0000007E) /*@EN: clock division 126 @CN: ʱӷƵ126 */
#define RCC_SYSCLK_Div128       ((uint32_t)0x0000007F) /*@EN: clock division 128 @CN: ʱӷƵ128 */
/* @end */

/* @defgroup RCC_HCLK_Div1 */
#define RCC_HCLK_Div1                   ((uint32_t)0x00000000) /*@EN: HCLK clock division 1  @CN: HCLK ʱӷƵ 1 */
#define RCC_HCLK_Div2                   ((uint32_t)0x00000001) /*@EN: HCLK clock division 2  @CN: HCLK ʱӷƵ 2  */
#define RCC_HCLK_Div3                   ((uint32_t)0x00000002) /*@EN: HCLK clock division 3  @CN: HCLK ʱӷƵ 3  */
#define RCC_HCLK_Div4                   ((uint32_t)0x00000003) /*@EN: HCLK clock division 4  @CN: HCLK ʱӷƵ 4  */
#define RCC_HCLK_Div5                   ((uint32_t)0x00000004) /*@EN: HCLK clock division 5  @CN: HCLK ʱӷƵ 5  */
#define RCC_HCLK_Div6                   ((uint32_t)0x00000005) /*@EN: HCLK clock division 6  @CN: HCLK ʱӷƵ 6  */
#define RCC_HCLK_Div7                   ((uint32_t)0x00000006) /*@EN: HCLK clock division 7  @CN: HCLK ʱӷƵ 7  */
#define RCC_HCLK_Div8                   ((uint32_t)0x00000007) /*@EN: HCLK clock division 8  @CN: HCLK ʱӷƵ 8  */
#define RCC_HCLK_Div9                   ((uint32_t)0x00000008) /*@EN: HCLK clock division 9  @CN: HCLK ʱӷƵ 9  */
#define RCC_HCLK_Div10                  ((uint32_t)0x00000009) /*@EN: HCLK clock division 10 @CN: HCLK ʱӷƵ 10 */
#define RCC_HCLK_Div11                  ((uint32_t)0x0000000A) /*@EN: HCLK clock division 11 @CN: HCLK ʱӷƵ 11 */
#define RCC_HCLK_Div12                  ((uint32_t)0x0000000B) /*@EN: HCLK clock division 12 @CN: HCLK ʱӷƵ 12 */
#define RCC_HCLK_Div13                  ((uint32_t)0x0000000C) /*@EN: HCLK clock division 13 @CN: HCLK ʱӷƵ 13 */
#define RCC_HCLK_Div14                  ((uint32_t)0x0000000D) /*@EN: HCLK clock division 14 @CN: HCLK ʱӷƵ 14 */
#define RCC_HCLK_Div15                  ((uint32_t)0x0000000E) /*@EN: HCLK clock division 15 @CN: HCLK ʱӷƵ 15 */
#define RCC_HCLK_Div16                  ((uint32_t)0x0000000F) /*@EN: HCLK clock division 16 @CN: HCLK ʱӷƵ 16 */
#define IS_RCC_PCLK(PCLK)               ((PCLK == RCC_HCLK_Div1) || \
                                         (PCLK == RCC_HCLK_Div2) || \
                                         (PCLK == RCC_HCLK_Div3) || \
                                         (PCLK == RCC_HCLK_Div4) || \
                                         (PCLK == RCC_HCLK_Div5) || \
                                         (PCLK == RCC_HCLK_Div6) || \
                                         (PCLK == RCC_HCLK_Div7) || \
                                         (PCLK == RCC_HCLK_Div8) || \
                                         (PCLK == RCC_HCLK_Div9) || \
                                         (PCLK == RCC_HCLK_Div10) || \
                                         (PCLK == RCC_HCLK_Div11) || \
                                         (PCLK == RCC_HCLK_Div12) || \
                                         (PCLK == RCC_HCLK_Div13) || \
                                         (PCLK == RCC_HCLK_Div14) || \
                                         (PCLK == RCC_HCLK_Div15)|| \
                                         (PCLK == RCC_HCLK_Div16))
/* @end */

/* @defgroup RCC_TIM17CLK */
#define RCC_TIM17CLK_SYS_CLK            ((uint32_t)0x00000000) /*@EN: Clock source selection system clock @CN: ʱԴѡϵͳʱ */
#define RCC_TIM17CLK_APB0               ((uint32_t)0x00000001) /*@EN: Clock source selection APB0 @CN: ʱԴѡAPB0 */
#define IS_RCC_TIM17CLK_SOURCE(SOURCE)  ((SOURCE == RCC_TIM17CLK_SYS_CLK) || \
                                         (SOURCE == RCC_TIM17CLK_APB0))
/* @end */

/* @defgroup RCC_TIM16CLK */
#define RCC_TIM16CLK_SYS_CLK            ((uint32_t)0x00000000) /*@EN: Clock source selection system clock @CN: ʱԴѡϵͳʱ */
#define RCC_TIM16CLK_APB0               ((uint32_t)0x00000001) /*@EN: Clock source selection APB0 @CN: ʱԴѡAPB0 */
#define IS_RCC_TIM16CLK_SOURCE(SOURCE)  ((SOURCE == RCC_TIM16CLK_SYS_CLK) || \
                                         (SOURCE == RCC_TIM16CLK_APB0))
/* @end */

/* @defgroup RCC_TIM15CLK */
#define RCC_TIM15CLK_SYS_CLK            ((uint32_t)0x00000000) /*@EN: Clock source selection system clock @CN: ʱԴѡϵͳʱ */
#define RCC_TIM15CLK_APB0               ((uint32_t)0x00000001) /*@EN: Clock source selection APB0 @CN: ʱԴѡAPB0 */
#define IS_RCC_TIM15CLK_SOURCE(SOURCE)  ((SOURCE == RCC_TIM15CLK_SYS_CLK) || \
                                         (SOURCE == RCC_TIM15CLK_APB0))
/* @end */

/* @defgroup RCC_TIM2CLK */
#define RCC_TIM2CLK_SYS_CLK             ((uint32_t)0x00000000) /*@EN: Clock source selection system clock @CN: ʱԴѡϵͳʱ */
#define RCC_TIM2CLK_APB0                ((uint32_t)0x00000001) /*@EN: Clock source selection APB0 @CN: ʱԴѡAPB0 */
#define IS_RCC_TIM2CLK_SOURCE(SOURCE)   ((SOURCE == RCC_TIM2CLK_SYS_CLK) || \
                                         (SOURCE == RCC_TIM2CLK_APB0))
/* @end */

/* @defgroup RCC_TIM1CLK */
#define RCC_TIM1CLK_SYS_CLK             ((uint32_t)0x00000000) /*@EN: Clock source selection system clock @CN: ʱԴѡϵͳʱ */
#define RCC_TIM1CLK_APB0                ((uint32_t)0x00000001) /*@EN: Clock source selection APB0 @CN: ʱԴѡAPB0 */
#define IS_RCC_TIM1CLK_SOURCE(SOURCE)   ((SOURCE == RCC_TIM1CLK_SYS_CLK) || \
                                         (SOURCE == RCC_TIM1CLK_APB0))
/* @end */

/* @defgroup RCC_SPI0CLK */
#define RCC_SPI0CLK_APB0                ((uint32_t)0x00000000) /*@EN: Clock source selection APB0 @CN: ʱԴѡAPB0 */
#define RCC_SPI0CLK_SYS_CLK_IN          ((uint32_t)0x00000001) /*@EN: Clock source selection SYSCLK_IN @CN: ʱԴѡSYSCLK_IN*/
#define RCC_SPI0CLK_RCH                 ((uint32_t)0x00000002) /*@EN: Clock source selection RCH  @CN: ʱԴѡRCH */
#define RCC_SPI0CLK_OSCL                ((uint32_t)0x00000003) /*@EN: Clock source selection OSCL @CN: ʱԴѡOSCL */
#define IS_RCC_SPI0CLK_SOURCE(SOURCE)   ((SOURCE == RCC_SPI0CLK_APB0)       || \
                                         (SOURCE == RCC_SPI0CLK_SYS_CLK_IN) || \
                                         (SOURCE == RCC_SPI0CLK_RCH)        || \
                                         (SOURCE == RCC_SPI0CLK_OSCL))
/* @end */

/* @defgroup RCC_SPI0CLK_DIV */
#define RCC_SPI0CLK_DIV_1               ((uint32_t)0x00000000) /*@EN: No frequency division @CN: Ƶ */
#define RCC_SPI0CLK_DIV_2               ((uint32_t)0x00000001) /*@EN: 2 division frequency @CN: 2Ƶ */
#define IS_RCC_SPI0CLK_DIV(DIV)         ((DIV == RCC_SPI0CLK_DIV_1)  || \
                                         (DIV == RCC_SPI0CLK_DIV_2))
/* @end */

/* @defgroup RCC_I2C0CLK */
#define RCC_I2C0CLK_APB0                ((uint32_t)0x00000000) /*@EN: Clock source selection APB0 @CN: ʱԴѡAPB0 */
#define RCC_I2C0CLK_SYS_CLK_IN          ((uint32_t)0x00000001) /*@EN: Clock source selection SYSCLK_IN @CN: ʱԴѡSYSCLK_IN*/
#define RCC_I2C0CLK_RCH                 ((uint32_t)0x00000002) /*@EN: Clock source selection RCH  @CN: ʱԴѡRCH */
#define RCC_I2C0CLK_OSCL                ((uint32_t)0x00000003) /*@EN: Clock source selection OSCL @CN: ʱԴѡOSCL */
#define IS_RCC_I2C0CLK_SOURCE(SOURCE)   ((SOURCE == RCC_I2C0CLK_APB0)       || \
                                         (SOURCE == RCC_I2C0CLK_SYS_CLK_IN) || \
                                         (SOURCE == RCC_I2C0CLK_RCH)        || \
                                         (SOURCE == RCC_I2C0CLK_OSCL))
/* @end */

/* @defgroup RCC_I2C0CLK_DIV */
#define RCC_I2C0CLK_DIV_1               ((uint32_t)0x00000000) /*@EN: No frequency division @CN: Ƶ */
#define RCC_I2C0CLK_DIV_2               ((uint32_t)0x00000001) /*@EN: 2 division frequency @CN: 2Ƶ */
#define IS_RCC_I2C0CLK_DIV(DIV)         ((DIV == RCC_I2C0CLK_DIV_1)  || \
                                         (DIV == RCC_I2C0CLK_DIV_2))
/* @end */

/* @defgroup RCC_USART0CLK */
#define RCC_USART0CLK_APB0              ((uint32_t)0x00000000) /*@EN: Clock source selection APB0 @CN: ʱԴѡAPB0 */
#define RCC_USART0CLK_SYS_CLK_IN        ((uint32_t)0x00000001) /*@EN: Clock source selection SYSCLK_IN @CN: ʱԴѡSYSCLK_IN*/
#define RCC_USART0CLK_RCH               ((uint32_t)0x00000002) /*@EN: Clock source selection RCH  @CN: ʱԴѡRCH */
#define RCC_USART0CLK_OSCL              ((uint32_t)0x00000003) /*@EN: Clock source selection OSCL @CN: ʱԴѡOSCL */
#define IS_RCC_USART0CLK_SOURCE(SOURCE) ((SOURCE == RCC_USART0CLK_APB0)       || \
                                         (SOURCE == RCC_USART0CLK_SYS_CLK_IN) || \
                                         (SOURCE == RCC_USART0CLK_RCH)        || \
                                         (SOURCE == RCC_USART0CLK_OSCL))
/* @end */

/* @defgroup RCC_USART0CLK_DIV */
#define RCC_USART0CLK_DIV_1             ((uint32_t)0x00000000) /*@EN: No frequency division @CN: Ƶ */
#define RCC_USART0CLK_DIV_2             ((uint32_t)0x00000001) /*@EN: 2 division frequency @CN: 2Ƶ */
#define IS_RCC_USART0CLK_DIV(DIV)       ((DIV == RCC_USART0CLK_DIV_1)  || \
                                         (DIV == RCC_USART0CLK_DIV_2))
/* @end */

/* @defgroup RCC_USART1CLK */
#define RCC_USART1CLK_APB0              ((uint32_t)0x00000000) /*@EN: Clock source selection APB0 @CN: ʱԴѡAPB0 */
#define RCC_USART1CLK_SYS_CLK_IN        ((uint32_t)0x00000001) /*@EN: Clock source selection SYSCLK_IN @CN: ʱԴѡSYSCLK_IN*/
#define RCC_USART1CLK_RCH               ((uint32_t)0x00000002) /*@EN: Clock source selection RCH  @CN: ʱԴѡRCH */
#define RCC_USART1CLK_OSCL              ((uint32_t)0x00000003) /*@EN: Clock source selection OSCL @CN: ʱԴѡOSCL */
#define IS_RCC_USART1CLK_SOURCE(SOURCE) ((SOURCE == RCC_USART1CLK_APB0)       || \
                                         (SOURCE == RCC_USART1CLK_SYS_CLK_IN) || \
                                         (SOURCE == RCC_USART1CLK_RCH)        || \
                                         (SOURCE == RCC_USART1CLK_OSCL))
/* @end */

/* @defgroup RCC_USART1CLK_DIV */
#define RCC_USART1CLK_DIV_1             ((uint32_t)0x00000000) /*@EN: No frequency division @CN: Ƶ */
#define RCC_USART1CLK_DIV_2             ((uint32_t)0x00000001) /*@EN: 2 division frequency @CN: 2Ƶ */
#define IS_RCC_USART1CLK_DIV(DIV)       ((DIV == RCC_USART1CLK_DIV_1)  || \
                                         (DIV == RCC_USART1CLK_DIV_2))
/* @end */

/* @defgroup RCC_WTCLKSource */
#define RCC_WTCLKSource_NONE            ((uint32_t)0x00000000) /*@EN: Frequency division clock of RCH @CN: RCHķƵʱ */
#define RCC_WTCLKSource_LSE             ((uint32_t)0x00000001) /*@EN: External low vibration OSCL @CN: ⲿOSCL */
#define RCC_WTCLKSource_LSI             ((uint32_t)0x00000002) /*@EN: Internal low vibration RCL @CN: ڲRCL */
#define IS_RCC_WTCLK_SOURCE(SOURCE)     ((SOURCE == RCC_WTCLKSource_NONE) || \
                                         (SOURCE == RCC_WTCLKSource_LSE)  || \
                                         (SOURCE == RCC_WTCLKSource_LSI)  ||)
/* @end */

/* @defgroup RCC_IT */
#define RCC_IT_CLXMUXERR                ((uint8_t)0x40) /*@EN: Clock enable error @CN: ʱʹܴ */
#define RCC_IT_CLKENERR                 ((uint8_t)0x20) /*@EN: System clock source switching error @CN: ϵͳʱԴл */
#define RCC_IT_PLL                      ((uint8_t)0x10) /*@EN: PLL state change @CN: PLL״̬仯 */
#define RCC_IT_LSE                      ((uint8_t)0x04) /*@EN: OSCL state change @CN: OSCL״̬仯 */
#define RCC_IT_HSI                      ((uint8_t)0x02) /*@EN: RCH state change @CN: RCH״̬仯 */
#define RCC_IT_LSI                      ((uint8_t)0x01) /*@EN: RCL state change @CN: RCL״̬仯 */
#define IS_RCC_IT(RCC_IT)               ((RCC_IT == RCC_IT_CLXMUXERR) || \
                                         (RCC_IT == RCC_IT_CLKENERR)  || \
                                         (RCC_IT == RCC_IT_PLL)       || \
                                         (RCC_IT == RCC_IT_LSE)       || \
                                         (RCC_IT == RCC_IT_HSI)       || \
                                         (RCC_IT == RCC_IT_LSI))
#define IS_RCC_CLEAR_IT(RCC_IT)         ((RCC_IT == RCC_IT_CLXMUXERR) || \
                                         (RCC_IT == RCC_IT_CLKENERR)  || \
                                         (RCC_IT == RCC_IT_PLL)       || \
                                         (RCC_IT == RCC_IT_LSE)       || \
                                         (RCC_IT == RCC_IT_HSI)       || \
                                         (RCC_IT == RCC_IT_LSI))
/* @end */

/* @defgroup RCC_FLAG */
#define RCC_FLAG_LVD                    ((uint8_t)0x18) /*@EN: LVD low-voltage monitoring @CN: LVDѹ */
#define RCC_FLAG_CLKMUX_LOCK            ((uint8_t)0x17) /*@EN: System clock source switching @CN: ϵͳʱԴл */
#define RCC_FLAG_SYS_CLK_LOCK           ((uint8_t)0x16) /*@EN: System Clock stable@CN: ϵͳʱȶ */
#define RCC_FLAG_PLL_LOCK               ((uint8_t)0x14) /*@EN: PLL stable @CN: PLLȶ */
#define RCC_FLAG_HSIRDY                 ((uint8_t)0x12) /*@EN: OSCL stable @CN: OSCL ȶ */
#define RCC_FLAG_LSERDY                 ((uint8_t)0x11) /*@EN: RCH stable @CN: RCH ȶ */
#define RCC_FLAG_LSIRDY                 ((uint8_t)0x10) /*@EN: RCL stable @CN: RCL ȶ */
#define IS_RCC_FLAG(RCC_FLAG)           ((RCC_FLAG == RCC_FLAG_LVD)          || \
                                         (RCC_FLAG == RCC_FLAG_CLKMUX_LOCK)  || \
                                         (RCC_FLAG == RCC_FLAG_SYS_CLK_LOCK) || \
                                         (RCC_FLAG == RCC_FLAG_PLL_LOCK) || \
                                         (RCC_FLAG == RCC_FLAG_HSERDY) || \
                                         (RCC_FLAG == RCC_FLAG_HSIRDY) || \
                                         (RCC_FLAG == RCC_FLAG_LSERDY) || \
                                         (RCC_FLAG == RCC_FLAG_LSIRDY))
/* @end */

/* @defgroup FjRange */
#define RCC_FJRANGE_0                   ((uint32_t)0x00000000) /*@EN: The minimum range of jitter frequency variation @CN: Ƶ仯ΧС */
#define RCC_FJRANGE_1                   ((uint32_t)0x00100000) /*@EN: Middle 1 of the range of jitter frequency variation @CN: Ƶ仯Χм1 */
#define RCC_FJRANGE_2                   ((uint32_t)0x00200000) /*@EN: Middle 2 of the range of jitter frequency variation @CN: Ƶ仯Χм2 */
#define RCC_FJRANGE_3                   ((uint32_t)0x00300000) /*@EN: The maximum range of jitter frequency variation @CN: Ƶ仯Χ */
#define IS_RCC_FJRANGE(RCC_FJRANGE)     ((RCC_FJRANGE == RCC_FJRANGE_0) || \
                                         (RCC_FJRANGE == RCC_FJRANGE_1) || \
                                         (RCC_FJRANGE == RCC_FJRANGE_2) || \
                                         (RCC_FJRANGE == RCC_FJRANGE_3))
/* @end */

/* @defgroup FjInterval */
#define RCC_FJINTERVAL_64               ((uint32_t)0x00000000) /*@EN: Change once every 64 RCH cycles @CN: 64  RCH ڱ仯һ */
#define RCC_FJINTERVAL_128              ((uint32_t)0x00020000) /*@EN: Change once every 128 RCH cycles @CN: 128  RCH ڱ仯һ */
#define RCC_FJINTERVAL_256              ((uint32_t)0x00040000) /*@EN: Change once every 256 RCH cycles @CN: 256  RCH ڱ仯һ */
#define RCC_FJINTERVAL_512              ((uint32_t)0x00060000) /*@EN: Change once every 512 RCH cycles @CN: 512  RCH ڱ仯һ */
#define RCC_FJINTERVAL_1024             ((uint32_t)0x00080000) /*@EN: Change once every 1024 RCH cycles @CN: 1024  RCH ڱ仯һ */
#define RCC_FJINTERVAL_2048             ((uint32_t)0x000A0000) /*@EN: Change once every 2048 RCH cycles @CN: 2048  RCH ڱ仯һ */
#define RCC_FJINTERVAL_4096             ((uint32_t)0x000C0000) /*@EN: Change once every 4096 RCH cycles @CN: 4096  RCH ڱ仯һ */
#define RCC_FJINTERVAL_8192             ((uint32_t)0x000E0000) /*@EN: Change once every 8192 RCH cycles @CN: 8192  RCH ڱ仯һ */
#define IS_RCC_FJINTERVAL(RCC_FJINTERVAL)  ((RCC_FJINTERVAL == RCC_FJINTERVAL_64) || \
                                            (RCC_FJINTERVAL == RCC_FJINTERVAL_128) || \
                                            (RCC_FJINTERVAL == RCC_FJINTERVAL_256) || \
                                            (RCC_FJINTERVAL == RCC_FJINTERVAL_512) || \
                                            (RCC_FJINTERVAL == RCC_FJINTERVAL_1024) || \
                                            (RCC_FJINTERVAL == RCC_FJINTERVAL_2048) || \
                                            (RCC_FJINTERVAL == RCC_FJINTERVAL_4096) || \
                                            (RCC_FJINTERVAL == RCC_FJINTERVAL_8192))
/* @end */

/* system */
void HSI_SetSysClockTo8(void);
void HSI_SetSysClockTo16(void);
void HSI_SetSysClockTo24(void);
void HSI_SetSysClockTo28(void);
void HSI_SetSysClockTo32(void);
void HSI_SetSysClockTo40(void);
void HSI_SetSysClockTo48(void);
void HSI_SetSysClockTo56(void);
void HSI_SetSysClockTo64(void);
void HSI_SetSysClockTo72(void);
void HSI_SetSysClockTo84(void);
void HSI_SetSysClockTo88(void);
void HSI_SetSysClockTo92(void);
void HSI_SetSysClockTo96(void);
void HSI_SetSysClockTo108(void);

void delay10us(uint32_t u32Cnt);
void RCC_DeInit(void);
void RCC_MCOConfig(uint8_t RCC_MCOSource, uint32_t RCC_MCOPrescaler);
void FLASH_LATENCY(uint8_t clkrank);
void SystemInit(void);
void Sysclk_ResetRch(void);
void SetSysClock(void);
void RCC_HCLKConfig(uint32_t RCC_SYSCLK);
void RCC_BackupResetCmd(FunctionalState NewState);

void RCC_GetClockFreq(RCC_ClocksTypeDef *RCC_Clocks);

/*HSI*/
void RCC_AdjustHSICalibrationValue(uint32_t HSICalibrationValue);
void RCC_HSICmd(FunctionalState NewState);

/*LSI*/
void RCC_LSICmd(FunctionalState NewState);

/*LSE*/
void RCC_LSEConfig(uint32_t RCC_LSE);

/*sys*/
void RCC_SYSCLKConfig(uint32_t RCC_SYSCLKSource);
uint32_t RCC_GetSYSCLKSource(void);

/*AHB*/
void RCC_AHBPeriphClockCmd(uint32_t RCC_AHBPeriph, FunctionalState NewState);
void RCC_AHBPeriphResetCmd(uint32_t RCC_AHBPeriph, FunctionalState NewState);

/*APB0*/
void RCC_PCLK0Config(uint32_t RCC_HCLK);
void RCC_APB0PeriphResetCmd(uint32_t RCC_APB0Periph, FunctionalState NewState);
void RCC_APB0PeriphClockCmd(uint32_t RCC_APB0Periph, FunctionalState NewState);
void RCC_TIM17CLKConfig(uint32_t RCC_TIM17CLK_Source);
void RCC_TIM16CLKConfig(uint32_t RCC_TIM16CLK_Source);
void RCC_TIM15CLKConfig(uint32_t RCC_TIM15CLK_Source);
void RCC_TIM2CLKConfig(uint32_t RCC_TIM2CLK_Source);
void RCC_TIM1CLKConfig(uint32_t RCC_TIM1CLK_Source);
void RCC_SPI0CLKConfig(uint32_t RCC_SPI0CLK_Source, uint32_t RCC_SPI0CLK_Div);
void RCC_I2C0CLKConfig(uint32_t RCC_I2C0CLK_Source, uint32_t RCC_I2C0CLK_Div);
void RCC_USART1CLKConfig(uint32_t RCC_USART1CLK_Source, uint32_t RCC_USART1CLK_Div);
void RCC_USART0CLKConfig(uint32_t RCC_USART0CLK_Source, uint32_t RCC_USART0CLK_Div);
void RCC_WTCLKConfig(uint32_t RCC_WTCLKSource);
void RCC_WTCLKCmd(FunctionalState NewState);

void OSCL_GPIO_INIT(void);

/*status*/
FlagStatus RCC_GetFlagStatus(uint8_t RCC_FLAG);
ITStatus RCC_GetITStatus(uint8_t RCC_IT);

/*interrupt*/
void RCC_ITConfig(uint8_t RCC_IT, FunctionalState NewState);//
void RCC_ClearITPendingBit(uint8_t RCC_IT);

void RCC_FrequrencyJitter_Config(uint32_t FjRange, uint32_t FjInterval, FunctionalState NewState);
#ifdef __cplusplus
}
#endif

#endif // __LCM32F06X_RCC_H__
