/**
  ******************************************************************************
  * @file    lcm32f06x_i2c.h 
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   Header for i2c.c module
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __LCM32F06X_I2C_H__
#define __LCM32F06X_I2C_H__

/* Includes ------------------------------------------------------------------*/
#include "lcm32f06x.h"
#include "lcm32f06x_rcc.h"
#include <stdio.h>

#ifdef __cplusplus
  extern "C" {
#endif

typedef struct
{
  uint32_t I2C_mode;                 /*!< EN: Specifies the operating master and slave mode for the selected I2C.
                                          CN: ָѡ I2C ģʽ
                                          @Register: IC_CON @Bits:[6][0] RESTEN MST_MODE
                                          @default: I2C_MasterOnlyMode        */
  
  uint32_t I2C_Speed;                /*!< EN: Specify I2C rate mode.
                                          CN: ָ I2C ģʽ
                                          @Register: IC_CON @Bits:[2:1] SPEED
                                          @default: I2C_SPEED_FAST_MODE       */

  uint32_t I2C_SlaveAddressedLength; /*!< EN: Address length when specifying I2C slave mode.
                                          CN: ָ I2C ӻģʽʱַȡ
                                          @Register: IC_CON @Bits:[3] 10ADDR_SLV
                                          @default: I2C_SlaveAddressedLength_10bit */

  uint32_t I2C_StopDetSlave;         /*!< EN: Generate STOP_DET interrupt condition in slave mode.
                                          CN: ӻģʽ² STOP_DET ж
                                          @Register: IC_CON @Bits:[7] SDIAD
                                          @default: I2C_StopDet_SlaveActive   */

  uint32_t I2C_TxEmptyIntr;          /*!< EN: Control the generation method of TX-EMPTY interrupt.
                                          CN:  TX_EMPTY жϵĲʽ
                                          @Register: IC_CON @Bits:[8] TXEC
                                          @default: I2C_TxEmptyIntr_DISABLE  */

  uint32_t I2C_RxFifoFullHold;       /*!< EN: Decide whether I2C occupies the bus when the RX FIFO reaches RX-RX-BUFFER-DEPTH.
                                          CN:  I2C  RX FIFO ﵽ RX_RX_BUFFER_DEPTH ʱǷռߡ
                                          @Register: IC_CON @Bits:[9] RFFHC
                                          @default: I2C_RxFifoFullHold_DISABLE */

  uint32_t I2C_StopDetMaster;        /*!< EN: STOP_DET interrupt in host mode.
                                          CN: ģʽµ STOP_DET жϡ
                                          @Register: IC_CON @Bits:[10] SDIMA
                                          @default: I2C_StopDet_MasterAlways  */

  uint32_t I2C_DualAddr;             /*!< EN: Enable or disable slave dual address.
                                          CN: ûôӻ˫ַ
                                          @Register: IC_CON @Bits:[11] DUAL_ADDR
                                          @default: I2C_TxDual_Addr_DISABLE   */
} I2C_InitTypeDef;

typedef enum
{
  STANDARD_MODE = 0,
  FAST_MODE,
  HIGH_SPEED
} SpeedSelectType;

/* @defgroup I2C_mode */
#define I2C_MasterOnlyMode               ((uint32_t)0x00000041) /* @EN: I2C master mode @CN: I2C ģʽ */
#define I2C_SlaveOnlyMode                ((uint32_t)0x00000000) /* @EN: I2C slave mode  @CN: I2C ӻģʽ */
/* end */

/* @defgroup I2C_Speed */
#define I2C_SPEED_STANDARD_MODE          ((uint32_t)0x00000002) /* @EN: I2C standard mode @CN: I2C ׼ģʽ */
#define I2C_SPEED_FAST_MODE              ((uint32_t)0x00000004) /* @EN: I2C Quick Mode    @CN: I2C ģʽ */
/* end */

/* @defgroup I2C_SlaveAddressedLength */
#define I2C_SlaveAddressedLength_10bit   ((uint32_t)0x00000008) /* @EN: I2C slave mode 10-bit address @CN: I2C ӻģʽ 10 λַ */
#define I2C_SlaveAddressedLength_7bit    ((uint32_t)0x00000000) /* @EN: I2C slave mode 7-bit address  @CN: I2C ӻģʽ 7 λַ  */
/* end */

/* @defgroup I2C_StopDetSlave */
#define I2C_StopDet_SlaveActive          ((uint32_t)0x00000080) /* @EN: Address matching generates interrupt @CN: Ѱַƥж */
#define I2C_StopDet_SlaveAlways          ((uint32_t)0x00000000) /* @EN: Interrupt generated regardless of addressing matching @CN: Ѱַƥ䶼ж */
/* end */

/* @defgroup I2C_TxEmptyIntr */
#define I2C_TxEmptyIntr_ENABLE           ((uint32_t)0x00000100) /* @EN: Enable TxEmpty  @CN:  TxEmpty */
#define I2C_TxEmptyIntr_DISABLE          ((uint32_t)0x00000000) /* @EN: Disable TxEmpty @CN:  TxEmpty */
/* end */

/* @defgroup I2C_RxFifoFullHold */
#define I2C_RxFifoFullHold_ENABLE        ((uint32_t)0x00000200) /* @EN: control bus   @CN:    */
#define I2C_RxFifoFullHold_DISABLE       ((uint32_t)0x00000000) /* @EN: uncontrol bus @CN:  */
/* end */

/* @defgroup I2C_StopDetMaster */
#define I2C_StopDet_MasterActive         ((uint32_t)0x00000400) /* @EN: Generate STOP_DET interrupt only when the host is valid  @CN: Чʱ STOP_DET ж */
#define I2C_StopDet_MasterAlways         ((uint32_t)0x00000000) /* @EN: STOP_DET interrupt is independent of whether the host is valid or not  @CN: STOP_DET жǷЧ޹ */
/* end */

/* @defgroup I2C_DualAddr */
#define I2C_TxDual_Addr_ENABLE           ((uint32_t)0x00000800) /* @EN: Enable dual address  @CN: ˫ַ */
#define I2C_TxDual_Addr_DISABLE          ((uint32_t)0x00000000) /* @EN: Disable dual address @CN: ˫ַ */
/* end */

/* @defgroup TxFifoValue */
#define I2C_TX_FIFO_IT_0                 ((uint32_t)0x00000000) /* @EN: TX-EMPTY interrupt trigger threshold is 0  @CN: TX_EMPTY жϴֵΪ 0 */
#define I2C_TX_FIFO_IT_1                 ((uint32_t)0x00000001) /* @EN: TX-EMPTY interrupt trigger threshold is 1  @CN: TX_EMPTY жϴֵΪ 1 */
#define I2C_TX_FIFO_IT_2                 ((uint32_t)0x00000002) /* @EN: TX-EMPTY interrupt trigger threshold is 2  @CN: TX_EMPTY жϴֵΪ 2 */
#define I2C_TX_FIFO_IT_3                 ((uint32_t)0x00000003) /* @EN: TX-EMPTY interrupt trigger threshold is 3  @CN: TX_EMPTY жϴֵΪ 3 */
#define I2C_TX_FIFO_IT_4                 ((uint32_t)0x00000004) /* @EN: TX-EMPTY interrupt trigger threshold is 4  @CN: TX_EMPTY жϴֵΪ 4 */
#define I2C_TX_FIFO_IT_5                 ((uint32_t)0x00000005) /* @EN: TX-EMPTY interrupt trigger threshold is 5  @CN: TX_EMPTY жϴֵΪ 5 */
#define I2C_TX_FIFO_IT_6                 ((uint32_t)0x00000006) /* @EN: TX-EMPTY interrupt trigger threshold is 6  @CN: TX_EMPTY жϴֵΪ 6 */
#define I2C_TX_FIFO_IT_7                 ((uint32_t)0x00000007) /* @EN: TX-EMPTY interrupt trigger threshold is 7  @CN: TX_EMPTY жϴֵΪ 7 */
#define IS_I2C_TX_FIFO_IT(TxFifoValue)   ((TxFifoValue == I2C_TX_FIFO_IT_0) || \
                                          (TxFifoValue == I2C_TX_FIFO_IT_1) || \
                                          (TxFifoValue == I2C_TX_FIFO_IT_2) || \
                                          (TxFifoValue == I2C_TX_FIFO_IT_3) || \
                                          (TxFifoValue == I2C_TX_FIFO_IT_4) || \
                                          (TxFifoValue == I2C_TX_FIFO_IT_5) || \
                                          (TxFifoValue == I2C_TX_FIFO_IT_6) || \
                                          (TxFifoValue == I2C_TX_FIFO_IT_7))
/* end */

/* @defgroup RxFifoValue */
#define I2C_RX_FIFO_IT_0                 ((uint32_t)0x00000000) /* @EN: RX_FULL interrupt trigger threshold is 0  @CN: RX_FULL жϴֵΪ 0 */
#define I2C_RX_FIFO_IT_1                 ((uint32_t)0x00000001) /* @EN: RX_FULL interrupt trigger threshold is 1  @CN: RX_FULL жϴֵΪ 1 */
#define I2C_RX_FIFO_IT_2                 ((uint32_t)0x00000002) /* @EN: RX_FULL interrupt trigger threshold is 2  @CN: RX_FULL жϴֵΪ 2 */
#define I2C_RX_FIFO_IT_3                 ((uint32_t)0x00000003) /* @EN: RX_FULL interrupt trigger threshold is 3  @CN: RX_FULL жϴֵΪ 3 */
#define I2C_RX_FIFO_IT_4                 ((uint32_t)0x00000004) /* @EN: RX_FULL interrupt trigger threshold is 4  @CN: RX_FULL жϴֵΪ 4 */
#define I2C_RX_FIFO_IT_5                 ((uint32_t)0x00000005) /* @EN: RX_FULL interrupt trigger threshold is 5  @CN: RX_FULL жϴֵΪ 5 */
#define I2C_RX_FIFO_IT_6                 ((uint32_t)0x00000006) /* @EN: RX_FULL interrupt trigger threshold is 6  @CN: RX_FULL жϴֵΪ 6 */
#define I2C_RX_FIFO_IT_7                 ((uint32_t)0x00000007) /* @EN: RX_FULL interrupt trigger threshold is 7  @CN: RX_FULL жϴֵΪ 7 */
#define IS_I2C_RX_FIFO_IT(RxFifoValue)   ((RxFifoValue == I2C_RX_FIFO_IT_0) || \
                                          (RxFifoValue == I2C_RX_FIFO_IT_1) || \
                                          (RxFifoValue == I2C_RX_FIFO_IT_2) || \
                                          (RxFifoValue == I2C_RX_FIFO_IT_3) || \
                                          (RxFifoValue == I2C_RX_FIFO_IT_4) || \
                                          (RxFifoValue == I2C_RX_FIFO_IT_5) || \
                                          (RxFifoValue == I2C_RX_FIFO_IT_6) || \
                                          (RxFifoValue == I2C_RX_FIFO_IT_7))
/* end */

/* @defgroup I2C_DMAReq */
#define I2C_DMAReq_Rx                    ((uint32_t)0x00000001) /* @EN: Enable DMA receive  @CN:  DMA ʹ */
#define I2C_DMAReq_Tx                    ((uint32_t)0x00000002) /* @EN: Enable DMA transmit @CN:  DMA ʹ */
#define IS_I2C_DMAReq(I2C_DMAReq)        ((I2C_DMAReq == I2C_DMAReq_Rx) || \
                                          (I2C_DMAReq == I2C_DMAReq_Tx))
/* end */

/* @defgroup TxDMAValue */
#define I2C_TX_DMA_IT_0                  ((uint32_t)0x00000000) /* @EN: transmit data level is 0  @CN: ˮƽΪ 0 */
#define I2C_TX_DMA_IT_1                  ((uint32_t)0x00000001) /* @EN: transmit data level is 1  @CN: ˮƽΪ 1 */
#define I2C_TX_DMA_IT_2                  ((uint32_t)0x00000002) /* @EN: transmit data level is 2  @CN: ˮƽΪ 2 */
#define I2C_TX_DMA_IT_3                  ((uint32_t)0x00000003) /* @EN: transmit data level is 3  @CN: ˮƽΪ 3 */
#define I2C_TX_DMA_IT_4                  ((uint32_t)0x00000004) /* @EN: transmit data level is 4  @CN: ˮƽΪ 4 */
#define I2C_TX_DMA_IT_5                  ((uint32_t)0x00000005) /* @EN: transmit data level is 5  @CN: ˮƽΪ 5 */
#define I2C_TX_DMA_IT_6                  ((uint32_t)0x00000006) /* @EN: transmit data level is 6  @CN: ˮƽΪ 6 */
#define I2C_TX_DMA_IT_7                  ((uint32_t)0x00000007) /* @EN: transmit data level is 7  @CN: ˮƽΪ 7 */
#define IS_I2C_TX_DMA_IT(TxDMAValue)     ((TxDMAValue == I2C_TX_DMA_IT_0) || \
                                          (TxDMAValue == I2C_TX_DMA_IT_1) || \
                                          (TxDMAValue == I2C_TX_DMA_IT_2) || \
                                          (TxDMAValue == I2C_TX_DMA_IT_3) || \
                                          (TxDMAValue == I2C_TX_DMA_IT_4) || \
                                          (TxDMAValue == I2C_TX_DMA_IT_5) || \
                                          (TxDMAValue == I2C_TX_DMA_IT_6) || \
                                          (TxDMAValue == I2C_TX_DMA_IT_7))
/* end */

/* @defgroup RxDMAValue */
#define I2C_RX_DMA_IT_0                  ((uint32_t)0x00000000) /* @EN: Received data level is 0  @CN: ˮƽΪ 0 */
#define I2C_RX_DMA_IT_1                  ((uint32_t)0x00000001) /* @EN: Received data level is 1  @CN: ˮƽΪ 1 */
#define I2C_RX_DMA_IT_2                  ((uint32_t)0x00000002) /* @EN: Received data level is 2  @CN: ˮƽΪ 2 */
#define I2C_RX_DMA_IT_3                  ((uint32_t)0x00000003) /* @EN: Received data level is 3  @CN: ˮƽΪ 3 */
#define I2C_RX_DMA_IT_4                  ((uint32_t)0x00000004) /* @EN: Received data level is 4  @CN: ˮƽΪ 4 */
#define I2C_RX_DMA_IT_5                  ((uint32_t)0x00000005) /* @EN: Received data level is 5  @CN: ˮƽΪ 5 */
#define I2C_RX_DMA_IT_6                  ((uint32_t)0x00000006) /* @EN: Received data level is 6  @CN: ˮƽΪ 6 */
#define I2C_RX_DMA_IT_7                  ((uint32_t)0x00000007) /* @EN: Received data level is 7  @CN: ˮƽΪ 7 */
#define IS_I2C_RX_DMA_IT(RxDMAValue)     ((RxDMAValue == I2C_RX_DMA_IT_0) || \
                                          (RxDMAValue == I2C_RX_DMA_IT_1) || \
                                          (RxDMAValue == I2C_RX_DMA_IT_2) || \
                                          (RxDMAValue == I2C_RX_DMA_IT_3) || \
                                          (RxDMAValue == I2C_RX_DMA_IT_4) || \
                                          (RxDMAValue == I2C_RX_DMA_IT_5) || \
                                          (RxDMAValue == I2C_RX_DMA_IT_6) || \
                                          (RxDMAValue == I2C_RX_DMA_IT_7))
/* end */

/* @defgroup I2C_Register */
#define I2C_Register_CON                 ((uint8_t)0x00) /* @EN: IIC Register CON            @CN: IIC Ĵ CON            */
#define I2C_Register_TAR                 ((uint8_t)0x04) /* @EN: IIC Register TAR            @CN: IIC Ĵ CON            */
#define I2C_Register_SAR                 ((uint8_t)0x08) /* @EN: IIC Register SAR            @CN: IIC Ĵ CON            */
#define I2C_Register_HSMADDR             ((uint8_t)0x0C) /* @EN: IIC Register HSMADDR        @CN: IIC Ĵ HSMADDR        */
#define I2C_Register_DATACMD             ((uint8_t)0x10) /* @EN: IIC Register DATACMD        @CN: IIC Ĵ DATACMD        */
#define I2C_Register_SSHCNT              ((uint8_t)0x14) /* @EN: IIC Register SSHCNT         @CN: IIC Ĵ SSHCNT         */
#define I2C_Register_SSLCNT              ((uint8_t)0x18) /* @EN: IIC Register SSLCNT         @CN: IIC Ĵ SSLCNT         */
#define I2C_Register_FSHCNT              ((uint8_t)0x1C) /* @EN: IIC Register FSHCNT         @CN: IIC Ĵ FSHCNT         */
#define I2C_Register_FSLCNT              ((uint8_t)0x20) /* @EN: IIC Register FSLCNT         @CN: IIC Ĵ FSLCNT         */
#define I2C_Register_HSHCNT              ((uint8_t)0x24) /* @EN: IIC Register HSHCNT         @CN: IIC Ĵ HSHCNT         */
#define I2C_Register_HSLCNT              ((uint8_t)0x28) /* @EN: IIC Register HSLCNT         @CN: IIC Ĵ HSLCNT         */
#define I2C_Register_INTRSTAT            ((uint8_t)0x2C) /* @EN: IIC Register INTRSTAT       @CN: IIC Ĵ INTRSTAT       */
#define I2C_Register_INTRMASK            ((uint8_t)0x30) /* @EN: IIC Register INTRMASK       @CN: IIC Ĵ INTRMASK       */
#define I2C_Register_RAWINTRSTAT         ((uint8_t)0x34) /* @EN: IIC Register RAWINTRSTAT    @CN: IIC Ĵ RAWINTRSTAT    */
#define I2C_Register_RXTL                ((uint8_t)0x38) /* @EN: IIC Register RXTL           @CN: IIC Ĵ RXTL           */
#define I2C_Register_TXTL                ((uint8_t)0x3C) /* @EN: IIC Register TXTL           @CN: IIC Ĵ TXTL           */
#define I2C_Register_CLRINTR             ((uint8_t)0x40) /* @EN: IIC Register CLRINTR        @CN: IIC Ĵ CLRINTR        */
#define I2C_Register_CLRRXUNDER          ((uint8_t)0x44) /* @EN: IIC Register CLRRXUNDER     @CN: IIC Ĵ CLRRXUNDER     */
#define I2C_Register_CLRRXOVER           ((uint8_t)0x48) /* @EN: IIC Register CLRRXOVER      @CN: IIC Ĵ CLRRXOVER      */
#define I2C_Register_CLRTXOVER           ((uint8_t)0x4C) /* @EN: IIC Register CLRTXOVER      @CN: IIC Ĵ CLRTXOVER      */
#define I2C_Register_CLRRDREQ            ((uint8_t)0x50) /* @EN: IIC Register CLRRDREQ       @CN: IIC Ĵ CLRRDREQ       */
#define I2C_Register_CLRTXABRT           ((uint8_t)0x54) /* @EN: IIC Register CLRTXABRT      @CN: IIC Ĵ CLRTXABRT      */
#define I2C_Register_CLRRXDONE           ((uint8_t)0x58) /* @EN: IIC Register CLRRXDONE      @CN: IIC Ĵ CLRRXDONE      */
#define I2C_Register_CLRACTIVITY         ((uint8_t)0x5C) /* @EN: IIC Register CLRACTIVITY    @CN: IIC Ĵ CLRACTIVITY    */
#define I2C_Register_CLRSTOP_DET         ((uint8_t)0x60) /* @EN: IIC Register CLRSTOP_DET    @CN: IIC Ĵ CLRSTOP_DET    */
#define I2C_Register_CLRSTARTDET         ((uint8_t)0x64) /* @EN: IIC Register CLRSTARTDET    @CN: IIC Ĵ CLRSTARTDET    */
#define I2C_Register_CLRGENCALL          ((uint8_t)0x68) /* @EN: IIC Register CLRGENCALL     @CN: IIC Ĵ CLRGENCALL     */
#define I2C_Register_ENABLE              ((uint8_t)0x6C) /* @EN: IIC Register ENABLE         @CN: IIC Ĵ ENABLE         */
#define I2C_Register_STATUS              ((uint8_t)0x70) /* @EN: IIC Register STATUS         @CN: IIC Ĵ STATUS         */
#define I2C_Register_TXFLR               ((uint8_t)0x74) /* @EN: IIC Register TXFLR          @CN: IIC Ĵ TXFLR          */
#define I2C_Register_RXFLR               ((uint8_t)0x78) /* @EN: IIC Register RXFLR          @CN: IIC Ĵ RXFLR          */
#define I2C_Register_SDAHOLD             ((uint8_t)0x7C) /* @EN: IIC Register SDAHOLD        @CN: IIC Ĵ SDAHOLD        */
#define I2C_Register_TXABRTSOURCE        ((uint8_t)0x80) /* @EN: IIC Register TXABRTSOURCE   @CN: IIC Ĵ TXABRTSOURCE   */
#define I2C_Register_SLVDATANACK         ((uint8_t)0x84) /* @EN: IIC Register SLVDATANACK    @CN: IIC Ĵ SLVDATANACK    */
#define I2C_Register_DMACR               ((uint8_t)0x88) /* @EN: IIC Register DMACR          @CN: IIC Ĵ DMACR          */
#define I2C_Register_DMATDLR             ((uint8_t)0x8C) /* @EN: IIC Register DMATDLR        @CN: IIC Ĵ DMATDLR        */
#define I2C_Register_DMARDLR             ((uint8_t)0x90) /* @EN: IIC Register DMARDLR        @CN: IIC Ĵ DMARDLR        */
#define I2C_Register_SDASETUP            ((uint8_t)0x94) /* @EN: IIC Register SDASETUP       @CN: IIC Ĵ SDASETUP       */
#define I2C_Register_ACKGENERALCALL      ((uint8_t)0x98) /* @EN: IIC Register ACKGENERALCALL @CN: IIC Ĵ ACKGENERALCALL */
#define I2C_Register_ENABLESTATUS        ((uint8_t)0x9C) /* @EN: IIC Register ENABLESTATUS   @CN: IIC Ĵ ENABLESTATUS   */
#define I2C_Register_FSSPKLEN            ((uint8_t)0xA0) /* @EN: IIC Register FSSPKLEN       @CN: IIC Ĵ FSSPKLEN       */
#define I2C_Register_HSSPKLEN            ((uint8_t)0xA4) /* @EN: IIC Register HSSPKLEN       @CN: IIC Ĵ HSSPKLEN       */
#define I2C_Register_CLRRESTARTDET       ((uint8_t)0xA8) /* @EN: IIC Register CLRRESTARTDET  @CN: IIC Ĵ CLRRESTARTDET  */
#define I2C_Register_COMPPARAM1          ((uint8_t)0xF4) /* @EN: IIC Register COMPPARAM1     @CN: IIC Ĵ COMPPARAM1     */
#define I2C_Register_COMPVERSION         ((uint8_t)0xF8) /* @EN: IIC Register COMPVERSION    @CN: IIC Ĵ COMPVERSION    */
#define I2C_Register_COMPTYPE            ((uint8_t)0xFC) /* @EN: IIC Register COMPTYPE       @CN: IIC Ĵ COMPTYPE       */
#define IS_I2C_REGISTER(REGISTER)        (((REGISTER) == I2C_Register_CON)            || \
                                          ((REGISTER) == I2C_Register_TAR)            || \
                                          ((REGISTER) == I2C_Register_SAR)            || \
                                          ((REGISTER) == I2C_Register_HSMADDR)        || \
                                          ((REGISTER) == I2C_Register_DATACMD)        || \
                                          ((REGISTER) == I2C_Register_SSHCNT)         || \
                                          ((REGISTER) == I2C_Register_SSLCNT)         || \
                                          ((REGISTER) == I2C_Register_FSHCNT)         || \
                                          ((REGISTER) == I2C_Register_FSLCNT)         || \
                                          ((REGISTER) == I2C_Register_HSHCNT)         || \
                                          ((REGISTER) == I2C_Register_HSLCNT)         || \
                                          ((REGISTER) == I2C_Register_INTRSTAT)       || \
                                          ((REGISTER) == I2C_Register_INTRMASK)       || \
                                          ((REGISTER) == I2C_Register_RAWINTRSTAT)    || \
                                          ((REGISTER) == I2C_Register_RXTL)           || \
                                          ((REGISTER) == I2C_Register_TXTL)           || \
                                          ((REGISTER) == I2C_Register_CLRINTR)        || \
                                          ((REGISTER) == I2C_Register_CLRRXUNDER)     || \
                                          ((REGISTER) == I2C_Register_CLRRXOVER)      || \
                                          ((REGISTER) == I2C_Register_CLRTXOVER)      || \
                                          ((REGISTER) == I2C_Register_CLRRDREQ)       || \
                                          ((REGISTER) == I2C_Register_CLRTXABRT)      || \
                                          ((REGISTER) == I2C_Register_CLRRXDONE)      || \
                                          ((REGISTER) == I2C_Register_CLRACTIVITY)    || \
                                          ((REGISTER) == I2C_Register_CLRSTOP_DET)    || \
                                          ((REGISTER) == I2C_Register_CLRSTARTDET)    || \
                                          ((REGISTER) == I2C_Register_CLRGENCALL)     || \
                                          ((REGISTER) == I2C_Register_ENABLE)         || \
                                          ((REGISTER) == I2C_Register_STATUS)         || \
                                          ((REGISTER) == I2C_Register_TXFLR)          || \
                                          ((REGISTER) == I2C_Register_RXFLR)          || \
                                          ((REGISTER) == I2C_Register_SDAHOLD)        || \
                                          ((REGISTER) == I2C_Register_TXABRTSOURCE)   || \
                                          ((REGISTER) == I2C_Register_SLVDATANACK)    || \
                                          ((REGISTER) == I2C_Register_DMACR)          || \
                                          ((REGISTER) == I2C_Register_DMATDLR)        || \
                                          ((REGISTER) == I2C_Register_DMARDLR)        || \
                                          ((REGISTER) == I2C_Register_SDASETUP)       || \
                                          ((REGISTER) == I2C_Register_ACKGENERALCALL) || \
                                          ((REGISTER) == I2C_Register_ENABLESTATUS)   || \
                                          ((REGISTER) == I2C_Register_FSSPKLEN)       || \
                                          ((REGISTER) == I2C_Register_HSSPKLEN)       || \
                                          ((REGISTER) == I2C_Register_CLRRESTARTDET)  || \
                                          ((REGISTER) == I2C_Register_COMPPARAM1)     || \
                                          ((REGISTER) == I2C_Register_COMPVERSION)    || \
                                          ((REGISTER) == I2C_Register_COMPTYPE))
/* end */

/* @defgroup I2C_IT */

#define I2C_IT_MST_ADDR1_MATCH           ((uint32_t)0x00040000) /* @EN: @CN: I2Cڴӻģʽ£ʹ˫ַܺIC_SAR1ƥһѰַʱ*/
#define I2C_IT_MST_ADDR0_MATCH           ((uint32_t)0x00020000) /* @EN: @CN: I2Cڴӻģʽ£ַIC_SAR0ƥһѰַʱ */
#define I2C_IT_PEC_ERROR                 ((uint32_t)0x00010000) /* @EN: @CN: I2CSMBUSģʽ¿PEC⹦ܺPECУ */
#define I2C_IT_TIME_OUT                  ((uint32_t)0x00008000) /* @EN: @CN: I2CSMBUSģʽ·ʱʱλ */
#define I2C_IT_SMBUS_ALERT               ((uint32_t)0x00004000) /* @EN: @CN: ĳ豸SMBALERTϲ͵͵ƽź */
#define I2C_IT_MST_ON_HOLD               ((uint32_t)0x00002000) /* @EN: @CN: ָʾǷռߣTxFIFOΪ */
#define I2C_IT_RESTART_DET               ((uint32_t)0x00001000) /* @EN: @CN: ӻģʽѱѰַ£⵽һRESTART */
#define I2C_IT_GEN_CALL                  ((uint32_t)0x00000800) /* @EN: @CN: I2CյʶGeneralCallַ */
#define I2C_IT_START_DET                 ((uint32_t)0x00000400) /* @EN: @CN: I2CյSTARTRESTART*/
#define I2C_IT_STOP_DET                  ((uint32_t)0x00000200) /* @EN: @CN: I2C յSTOP */
#define I2C_IT_ACTIVITY                  ((uint32_t)0x00000100) /* @EN: @CN: ʾI2CģǷв֮ǰһֱά */
#define I2C_IT_RX_DONE                   ((uint32_t)0x00000080) /* @EN: @CN: I2CΪӻʱѴ */
#define I2C_IT_TX_ABRT                   ((uint32_t)0x00000040) /* @EN: @CN: I2CΪ޷FIFOڵʱλ1ڴӻģʽ¶ܷ */
#define I2C_IT_RD_REQ                    ((uint32_t)0x00000020) /* @EN: @CN: I2CΪӻһͼӸôӻȡʱ */
#define I2C_IT_TX_EMPTY                  ((uint32_t)0x00000010) /* @EN: @CN: bufferIC_TX_TL趨ֵڲλĴеĵַݷ */
#define I2C_IT_TX_OVER                   ((uint32_t)0x00000008) /* @EN: @CN: I2CڷģʽTXFIFOѾ£IC_DATA_CMDд */
#define I2C_IT_RX_FULL                   ((uint32_t)0x00000004) /* @EN: @CN: I2CĽbufferIC_RX_TL趨ֵʱ */
#define I2C_IT_RX_OVER                   ((uint32_t)0x00000002) /* @EN: @CN: I2CĽbufferRXҴⲿI2Cյֽ */
#define I2C_IT_RX_UNDER                  ((uint32_t)0x00000001) /* @EN: @CN: bufferΪʱCPUͨIC_DATA_CMDĴbuffe */
#define I2C_IT_ALL                       ((uint32_t)0x00003FFF) /* @EN: @CN: ж */
#define IS_I2C_GET_IT(I2C_IT)            (((I2C_IT) == I2C_IT_MST_ADDR1_MATCH) || \
                                          ((I2C_IT) == I2C_IT_MST_ADDR0_MATCH) || \
                                          ((I2C_IT) == I2C_IT_PEC_ERROR)   || \
                                          ((I2C_IT) == I2C_IT_TIME_OUT)    || \
                                          ((I2C_IT) == I2C_IT_SMBUS_ALERT) || \
                                          ((I2C_IT) == I2C_IT_MST_ON_HOLD) || \
                                          ((I2C_IT) == I2C_IT_RESTART_DET) || \
                                          ((I2C_IT) == I2C_IT_GEN_CALL)    || \
                                          ((I2C_IT) == I2C_IT_START_DET)   || \
                                          ((I2C_IT) == I2C_IT_STOP_DET)    || \
                                          ((I2C_IT) == I2C_IT_ACTIVITY)    || \
                                          ((I2C_IT) == I2C_IT_RX_DONE)     || \
                                          ((I2C_IT) == I2C_IT_TX_ABRT)     || \
                                          ((I2C_IT) == I2C_IT_RD_REQ)      || \
                                          ((I2C_IT) == I2C_IT_TX_EMPTY)    || \
                                          ((I2C_IT) == I2C_IT_TX_OVER)     || \
                                          ((I2C_IT) == I2C_IT_RX_FULL)     || \
                                          ((I2C_IT) == I2C_IT_RX_OVER)     || \
                                          ((I2C_IT) == I2C_IT_RX_UNDER)    || \
                                          ((I2C_IT) == I2C_IT_ALL))
/* end */

/* @defgroup I2C_FLAG */
#define I2C_FLAG_MST_ADDR1_MATCH         I2C_INTRSTAT_R_ADDR1_MATCH_Msk  /* @EN: @CN: I2Cڴӻģʽ£ʹ˫ַܺIC_SAR1ƥһѰַʱ*/
#define I2C_FLAG_MST_ADDR0_MATCH         I2C_INTRSTAT_R_ADDR0_MATCH_Msk  /* @EN: @CN: I2Cڴӻģʽ£ַIC_SAR0ƥһѰַʱ */
#define I2C_FLAG_PEC_ERROR               I2C_INTRMASK_M_PEC_ERROR_Msk    /* @EN: @CN: I2CSMBUSģʽ¿PEC⹦ܺPECУ */
#define I2C_FLAG_TIME_OUT                I2C_INTRMASK_M_TIMEOUT_Msk      /* @EN: @CN: I2CSMBUSģʽ·ʱʱλ */
#define I2C_FLAG_SMBUS_ALERT             I2C_INTRMASK_M_SMBUS_ALERT_Msk  /* @EN: @CN: ĳ豸SMBALERTϲ͵͵ƽź */
#define I2C_FLAG_MST_ON_HOLD             I2C_INTRMASK_M_MST_ON_HOLD_Msk  /* @EN: @CN: ָʾǷռߣTxFIFOΪ */
#define I2C_FLAG_RESTART_DET             I2C_INTRMASK_M_RESTART_DET_Msk  /* @EN: @CN: ӻģʽѱѰַ£⵽һRESTART */
#define I2C_FLAG_GEN_CALL                I2C_INTRMASK_M_GEN_CALL_Msk     /* @EN: @CN: I2CյʶGeneralCallַ */
#define I2C_FLAG_START_DET               I2C_INTRMASK_M_START_DET_Msk    /* @EN: @CN: I2CյSTARTRESTART*/
#define I2C_FLAG_STOP_DET                I2C_INTRMASK_M_STOP_DET_Msk     /* @EN: @CN: I2C յSTOP */
#define I2C_FLAG_ACTIVITY                I2C_INTRMASK_M_ACTIVITY_Msk     /* @EN: @CN: ʾI2CģǷв֮ǰһֱά */
#define I2C_FLAG_RX_DONE                 I2C_INTRMASK_M_RX_DONE_Msk      /* @EN: @CN: I2CΪӻʱѴ */
#define I2C_FLAG_TX_ABRT                 I2C_INTRMASK_M_TX_ABRT_Msk      /* @EN: @CN: I2CΪ޷FIFOڵʱλ1ڴӻģʽ¶ܷ */
#define I2C_FLAG_RD_REQ                  I2C_INTRMASK_M_RD_REQ_Msk       /* @EN: @CN: I2CΪӻһͼӸôӻȡʱ */
#define I2C_FLAG_TX_EMPTY                I2C_INTRMASK_M_TX_EMPTY_Msk     /* @EN: @CN: bufferIC_TX_TL趨ֵڲλĴеĵַݷ */
#define I2C_FLAG_TX_OVER                 I2C_INTRMASK_M_RX_OVER_Msk      /* @EN: @CN: I2CڷģʽTXFIFOѾ£IC_DATA_CMDд */
#define I2C_FLAG_RX_FULL                 I2C_INTRMASK_M_RX_FULL_Msk      /* @EN: @CN: I2CĽbufferIC_RX_TL趨ֵʱ */
#define I2C_FLAG_RX_OVER                 I2C_INTRMASK_M_RX_OVER_Msk      /* @EN: @CN: I2CĽbufferRXҴⲿI2Cյֽ */
#define I2C_FLAG_RX_UNDER                I2C_INTRMASK_M_RX_UNDER_Msk     /* @EN: @CN: bufferΪʱCPUͨIC_DATA_CMDĴbuffe */

#define IS_I2C_GET_FLAG(I2C_FLAG)        (((I2C_FLAG) == I2C_FLAG_MST_ADDR1_MATCH) || \
                                          ((I2C_FLAG) == I2C_FLAG_MST_ADDR0_MATCH) || \
                                          ((I2C_FLAG) == I2C_FLAG_PEC_ERROR)   || \
                                          ((I2C_FLAG) == I2C_FLAG_TIME_OUT)    || \
                                          ((I2C_FLAG) == I2C_FLAG_SMBUS_ALERT) || \
                                          ((I2C_FLAG) == I2C_FLAG_MST_ON_HOLD) || \
                                          ((I2C_FLAG) == I2C_FLAG_RESTART_DET) || \
                                          ((I2C_FLAG) == I2C_FLAG_GEN_CALL)    || \
                                          ((I2C_FLAG) == I2C_FLAG_START_DET)   || \
                                          ((I2C_FLAG) == I2C_FLAG_STOP_DET)    || \
                                          ((I2C_FLAG) == I2C_FLAG_ACTIVITY)    || \
                                          ((I2C_FLAG) == I2C_FLAG_RX_DONE)     || \
                                          ((I2C_FLAG) == I2C_FLAG_TX_ABRT)     || \
                                          ((I2C_FLAG) == I2C_FLAG_RD_REQ)      || \
                                          ((I2C_FLAG) == I2C_FLAG_TX_EMPTY)    || \
                                          ((I2C_FLAG) == I2C_FLAG_TX_OVER)     || \
                                          ((I2C_FLAG) == I2C_FLAG_RX_FULL)     || \
                                          ((I2C_FLAG) == I2C_FLAG_RX_OVER)     || \
                                          ((I2C_FLAG) == I2C_FLAG_RX_UNDER))
/* end */

/* @defgroup I2C_STATUS */
#define I2C_STATUS_SLV_HOLD_RXFIFO_FULL  I2C_STATUS_SLV_HOLD_RXFIFO_FULL_Msk /* @EN: The current RxFIFO is full and additional bytes have been received. The I2C bus is controlled by the slave @CN: ǰRxFIFOյֽڣI2C߱ӻ */
#define I2C_STATUS_SLV_HOLD_TXFIFO_EMPTY I2C_STATUS_SLV_HOLD_TXFIFO_Msk      /* @EN: The current TxFIFO is empty and a read request has been received. The I2C bus is controlled by the slave @CN: ǰTxFIFOյI2C߱ӻ */
#define I2C_STATUS_MST_HOLD_RXFIFO_FULL  I2C_STATUS_MST_HOLD_RXFIFO_Msk      /* @EN: The current RxFIFO is full and additional bytes have been received, and the bus is controlled by the host @CN: ǰRxFIFOյֽڣ߱ */
#define I2C_STATUS_MST_HOLD_TXFIFO_EMPTY I2C_STATUS_MST_HOLD_TXFIFO_Msk      /* @EN: The current TxFIFO is empty and there is no stop bit for the current transmission command, causing the bus to be taken over by the host @CN: ǰTxFIFOҵǰûֹͣλ߱ӹ */
#define I2C_STATUS_SLV_ACTIVITY          I2C_STATUS_SLV_ACTIVITY_Msk         /* @EN: The slave state is not in IDLE state @CN: ӻ״̬IDLE״̬ */
#define I2C_STATUS_MST_ACTIVITY          I2C_STATUS_MST_ACTIVITY_Msk         /* @EN: The host status is not in IDLE state @CN: ״̬IDLE״̬ */
#define I2C_STATUS_RFF                   I2C_STATUS_RFF_Msk                  /* @EN: When receiving FIFO full @CN: FIFOȫʱ */
#define I2C_STATUS_RFNF                  I2C_STATUS_RFNF_Msk                 /* @EN: Receive FIFO non empty @CN: FIFOǿ */
#define I2C_STATUS_TFE                   I2C_STATUS_TFE_Msk                  /* @EN: Send FIFO empty @CN: FIFO */
#define I2C_STATUS_TFNF                  I2C_STATUS_TFNF_Msk                 /* @EN: Send FIFO not full @CN: FIFO */
#define I2C_STATUS_ACTIVITY              I2C_STATUS_ACTIVITY_Msk             /* @EN: I2C active state @CN: I2CԾ״̬ */

#define IS_I2C_GET_STATUS(STATUS)        (((STATUS) == I2C_STATUS_SLV_HOLD_RXFIFO_FULL)  || \
                                          ((STATUS) == I2C_STATUS_SLV_HOLD_TXFIFO_EMPTY) || \
                                          ((STATUS) == I2C_STATUS_MST_HOLD_RXFIFO_FULL)  || \
                                          ((STATUS) == I2C_STATUS_MST_HOLD_TXFIFO_EMPTY) || \
                                          ((STATUS) == I2C_STATUS_SLV_ACTIVITY)          || \
                                          ((STATUS) == I2C_STATUS_MST_ACTIVITY)          || \
                                          ((STATUS) == I2C_STATUS_RFF)                   || \
                                          ((STATUS) == I2C_STATUS_RFNF)                  || \
                                          ((STATUS) == I2C_STATUS_TFE)                   || \
                                          ((STATUS) == I2C_STATUS_RFNE)                  || \
                                          ((STATUS) == I2C_STATUS_TFNF)                  || \
                                          ((STATUS) == I2C_STATUS_ACTIVITY))
/* end */

/* @defgroup I2Cx */
#define IS_I2C_ALL_PERIPH(PERIPH)        (PERIPH) == I2C0)
/* end */

/* ========================================================================== */
/* ================             I2C  REGISTER BIT               ============= */
/* ========================================================================== */

/* =================================  CON  ================================== */
#define I2C_CON_TIMEOUT_EN_Pos                    (15UL)                       /*!< TIMEOUT_EN (Bit 15)                            */
#define I2C_CON_TIMEOUT_EN_Msk                    (0x00008000UL)               /*!< TIMEOUT_EN (Bitfield-Mask: 0x01)               */
#define I2C_CON_TEXT_EN_Pos                       (14UL)                       /*!< TEXT_EN (Bit 14)                               */
#define I2C_CON_TEXT_EN_Msk                       (0x00004000UL)               /*!< TEXT_EN (Bitfield-Mask: 0x01)                  */
#define I2C_CON_PEC_EN_Pos                        (13UL)                       /*!< PEC_EN (Bit 13)                                */
#define I2C_CON_PEC_EN_Msk                        (0x00002000UL)               /*!< PEC_EN (Bitfield-Mask: 0x01)                   */
#define I2C_CON_SMB_MODE_Pos                      (12UL)                       /*!< SMB_MODE (Bit 12)                              */
#define I2C_CON_SMB_MODE_Msk                      (0x00001000UL)               /*!< SMB_MODE (Bitfield-Mask: 0x01)                 */
#define I2C_CON_DUAL_ADDR_Pos                     (11UL)                       /*!< DUAL_ADDR (Bit 11)                             */
#define I2C_CON_DUAL_ADDR_Msk                     (0x00000800UL)               /*!< DUAL_ADDR (Bitfield-Mask: 0x01)                */
#define I2C_CON_SDIMA_Pos                         (10UL)                       /*!< SDIMA (Bit 10)                                 */
#define I2C_CON_SDIMA_Msk                         (0x00000400UL)               /*!< SDIMA (Bitfield-Mask: 0x01)                    */
#define I2C_CON_RFFHC_Pos                         (9UL)                        /*!< RFFHC (Bit 9)                                  */
#define I2C_CON_RFFHC_Msk                         (0x00000200UL)               /*!< RFFHC (Bitfield-Mask: 0x01)                    */
#define I2C_CON_TXEC_Pos                          (8UL)                        /*!< TXEC (Bit 8)                                   */
#define I2C_CON_TXEC_Msk                          (0x00000100UL)               /*!< TXEC (Bitfield-Mask: 0x01)                     */
#define I2C_CON_SDIAD_Pos                         (7UL)                        /*!< SDIAD (Bit 7)                                  */
#define I2C_CON_SDIAD_Msk                         (0x00000080UL)               /*!< SDIAD (Bitfield-Mask: 0x01)                    */
#define I2C_CON_SLVDIS_Pos                        (6UL)                        /*!< SLVDIS (Bit 6)                                 */
#define I2C_CON_SLVDIS_Msk                        (0x00000040UL)               /*!< SLVDIS (Bitfield-Mask: 0x01)                   */
#define I2C_CON_RESTEN_Pos                        (5UL)                        /*!< RESTEN (Bit 5)                                 */
#define I2C_CON_RESTEN_Msk                        (0x00000020UL)               /*!< RESTEN (Bitfield-Mask: 0x01)                   */
#define I2C_CON_10ADDR_MAS_Pos                    (4UL)                        /*!< 10ADDR_MAS (Bit 4)                             */
#define I2C_CON_10ADDR_MAS_Msk                    (0x00000010UL)               /*!< 10ADDR_MAS (Bitfield-Mask: 0x01)               */
#define I2C_CON_10ADDR_SLV_Pos                    (3UL)                        /*!< 10ADDR_SLV (Bit 3)                             */
#define I2C_CON_10ADDR_SLV_Msk                    (0x00000008UL)               /*!< 10ADDR_SLV (Bitfield-Mask: 0x01)               */
#define I2C_CON_SPEED_Pos                         (1UL)                        /*!< SPEED (Bit 1)                                  */
#define I2C_CON_SPEED_Msk                         (0x00000006UL)               /*!< SPEED (Bitfield-Mask: 0x03)                    */
#define I2C_CON_MST_MODE_Pos                      (0UL)                        /*!< MST_MODE (Bit 0)                               */
#define I2C_CON_MST_MODE_Msk                      (0x00000001UL)               /*!< MST_MODE (Bitfield-Mask: 0x01)                 */
/* =================================  TAR  ================================== */
#define I2C_TAR_10ADDR_MAS_Pos                    (12UL)                       /*!< 10ADDR_MAS (Bit 12)                            */
#define I2C_TAR_10ADDR_MAS_Msk                    (0x00001000UL)               /*!< 10ADDR_MAS (Bitfield-Mask: 0x01)               */
#define I2C_TAR_SPECIAL_Pos                       (11UL)                       /*!< SPECIAL (Bit 11)                               */
#define I2C_TAR_SPECIAL_Msk                       (0x00000800UL)               /*!< SPECIAL (Bitfield-Mask: 0x01)                  */
#define I2C_TAR_GC_STR_Pos                        (10UL)                       /*!< GC_STR (Bit 11)                                */
#define I2C_TAR_GC_STR_Msk                        (0x00000400UL)               /*!< GC_STR (Bitfield-Mask: 0x01)                   */
#define I2C_TAR_IC_TAR_Pos                        (0UL)                        /*!< IC_TAR (Bit 11)                                */
#define I2C_TAR_IC_TAR_Msk                        (0x000003FFUL)               /*!< IC_TAR (Bitfield-Mask: 0x01)                   */
/* =================================  SAR  ================================== */
#define I2C_SAR_IC_SAR1_Pos                       (16UL)                       /*!< IC_SAR1 (Bit 0)                                */
#define I2C_SAR_IC_SAR1_Msk                       (0x000001FFUL)               /*!< IC_SAR1 (Bitfield-Mask: 0x1FF)                 */
#define I2C_SAR_IC_SAR0_Pos                       (0UL)                        /*!< IC_SAR0 (Bit 0)                                */
#define I2C_SAR_IC_SAR0_Msk                       (0x000001FFUL)               /*!< IC_SAR0 (Bitfield-Mask: 0x1FF)                 */
/* ===============================  DATACMD  ================================ */
#define I2C_DATACMD_FDB_Pos                       (11UL)                       /*!< FDB (Bit 11)                                   */
#define I2C_DATACMD_FDB_Msk                       (0x00000800UL)               /*!< FDB (Bitfield-Mask: 0x01)                      */
#define I2C_DATACMD_RESTART_Pos                   (10UL)                       /*!< RESTART (Bit 10)                               */
#define I2C_DATACMD_RESTART_Msk                   (0x00000400UL)               /*!< RESTART (Bitfield-Mask: 0x01)                  */
#define I2C_DATACMD_STOP_Pos                      (9UL)                        /*!< STOP (Bit 9)                                   */
#define I2C_DATACMD_STOP_Msk                      (0x00000200UL)               /*!< STOP (Bitfield-Mask: 0x01)                     */
#define I2C_DATACMD_CMD_Pos                       (8UL)                        /*!< CMD (Bit 8)                                    */
#define I2C_DATACMD_CMD_Msk                       (0x00000100UL)               /*!< CMD (Bitfield-Mask: 0x01)                      */
#define I2C_DATACMD_DAT_Pos                       (0UL)                        /*!< DAT (Bit 0)                                    */
#define I2C_DATACMD_DAT_Msk                       (0x000000ffUL)               /*!< DAT (Bitfield-Mask: 0xff)                      */
/* ==============================  SS_SCL_HCNT  ============================= */
#define I2C_SSHCNT_IC_SS_SCL_HCNT_Pos             (0UL)                        /*!< SS_SCL_HCNT (Bit 0)                            */
#define I2C_SSHCNT_IC_SS_SCL_HCNT_Msk             (0x0000FFFFUL)               /*!< SS_SCL_HCNT (Bitfield-Mask: 0xFFFF)            */
/* ==============================  SS_SCL_LCNT  ============================= */
#define I2C_SSLCNT_IC_SS_SCL_LCNT_Pos             (0UL)                        /*!< SS_SCL_LCNT (Bit 0)                            */
#define I2C_SSLCNT_IC_SS_SCL_LCNT_Msk             (0x0000FFFFUL)               /*!< SS_SCL_LCNT (Bitfield-Mask: 0xFFFF)            */
/* ==============================  FS_SCL_HCNT  ============================= */
#define I2C_FSHCNT_IC_FS_SCL_HCNT_Pos             (0UL)                        /*!< FS_SCL_HCNT (Bit 0)                            */
#define I2C_FSHCNT_IC_FS_SCL_HCNT_Msk             (0x0000FFFFUL)               /*!< FS_SCL_HCNT (Bitfield-Mask: 0xFFFF)            */
/* ==============================  FS_SCL_LCNT  ============================= */
#define I2C_FSLCNT_IC_FS_SCL_LCNT_Pos             (0UL)                        /*!< FS_SCL_LCNT (Bit 0)                            */
#define I2C_FSLCNT_IC_FS_SCL_LCNT_Msk             (0x0000FFFFUL)               /*!< FS_SCL_LCNT (Bitfield-Mask: 0xFFFF)            */
/* ==============================   INTRSTAT  =============================== */
#define I2C_INTRSTAT_R_ADDR1_MATCH_Pos            (18UL)                       /*!< R_ADDR1_MATCH (Bit 18)                         */
#define I2C_INTRSTAT_R_ADDR1_MATCH_Msk            (0x00040000UL)               /*!< R_ADDR1_MATCH (Bitfield-Mask: 0x01)            */
#define I2C_INTRSTAT_R_ADDR0_MATCH_Pos            (17UL)                       /*!< R_ADDR0_MATCH (Bit 17)                         */
#define I2C_INTRSTAT_R_ADDR0_MATCH_Msk            (0x00020000UL)               /*!< R_ADDR0_MATCH (Bitfield-Mask: 0x01)            */
#define I2C_INTRSTAT_R_PEC_ERROR_Pos              (16UL)                       /*!< R_PEC_ERROR (Bit 16)                           */
#define I2C_INTRSTAT_R_PEC_ERROR_Msk              (0x00010000UL)               /*!< R_PEC_ERROR (Bitfield-Mask: 0x01)              */
#define I2C_INTRSTAT_R_TIMEOUT_Pos                (15UL)                       /*!< R_TIMEOUT (Bit 15)                             */
#define I2C_INTRSTAT_R_TIMEOUT_Msk                (0x00008000UL)               /*!< R_TIMEOUT (Bitfield-Mask: 0x01)                */
#define I2C_INTRSTAT_R_SMBUS_ALERT_Pos            (14UL)                       /*!< R_SMBUS_ALERT (Bit 14)                         */
#define I2C_INTRSTAT_R_SMBUS_ALERT_Msk            (0x00004000UL)               /*!< R_SMBUS_ALERT (Bitfield-Mask: 0x01)            */
#define I2C_INTRSTAT_R_MST_ON_HOLD_Pos           (13UL)                       /*!< R_MST_ON_HOLD (Bit 13)                         */
#define I2C_INTRSTAT_R_MST_ON_HOLD_Msk           (0x2000UL)                   /*!< R_MST_ON_HOLD (Bitfield-Mask: 0x01)            */
#define I2C_INTRSTAT_R_RESTART_DET_Pos           (12UL)                       /*!< R_RESTART_DET (Bit 12)                         */
#define I2C_INTRSTAT_R_RESTART_DET_Msk           (0x1000UL)                   /*!< R_RESTART_DET (Bitfield-Mask: 0x01)            */
#define I2C_INTRSTAT_R_GEN_CALL_Pos              (11UL)                       /*!< R_GEN_CALL (Bit 11)                            */
#define I2C_INTRSTAT_R_GEN_CALL_Msk              (0x800UL)                    /*!< R_GEN_CALL (Bitfield-Mask: 0x01)               */
#define I2C_INTRSTAT_R_START_DET_Pos             (10UL)                       /*!< R_START_DET (Bit 10)                           */
#define I2C_INTRSTAT_R_START_DET_Msk             (0x400UL)                    /*!< R_START_DET (Bitfield-Mask: 0x01)              */
#define I2C_INTRSTAT_R_STOP_DET_Pos              (9UL)                        /*!< R_STOP_DET (Bit 9)                             */
#define I2C_INTRSTAT_R_STOP_DET_Msk              (0x200UL)                    /*!< R_STOP_DET (Bitfield-Mask: 0x01)               */
#define I2C_INTRSTAT_R_ACTIVITY_Pos              (8UL)                        /*!< R_ACTIVITY (Bit 8)                             */
#define I2C_INTRSTAT_R_ACTIVITY_Msk              (0x100UL)                    /*!< R_ACTIVITY (Bitfield-Mask: 0x01)               */
#define I2C_INTRSTAT_R_RX_DONE_Pos               (7UL)                        /*!< R_RX_DONE (Bit 7)                              */
#define I2C_INTRSTAT_R_RX_DONE_Msk               (0x80UL)                     /*!< R_RX_DONE (Bitfield-Mask: 0x01)                */
#define I2C_INTRSTAT_R_TX_ABRT_Pos               (6UL)                        /*!< R_TX_ABRT (Bit 6)                              */
#define I2C_INTRSTAT_R_TX_ABRT_Msk               (0x40UL)                     /*!< R_TX_ABRT (Bitfield-Mask: 0x01)                */
#define I2C_INTRSTAT_R_RD_REQ_Pos                (5UL)                        /*!< R_RD_REQ (Bit 5)                               */
#define I2C_INTRSTAT_R_RD_REQ_Msk                (0x20UL)                     /*!< R_RD_REQ (Bitfield-Mask: 0x01)                 */
#define I2C_INTRSTAT_R_TX_EMPTY_Pos              (4UL)                        /*!< R_TX_EMPTY (Bit 4)                             */
#define I2C_INTRSTAT_R_TX_EMPTY_Msk              (0x10UL)                     /*!< R_TX_EMPTY (Bitfield-Mask: 0x01)               */
#define I2C_INTRSTAT_R_TX_OVER_Pos               (3UL)                        /*!< R_TX_OVER (Bit 3)                              */
#define I2C_INTRSTAT_R_TX_OVER_Msk               (0x8UL)                      /*!< R_TX_OVER (Bitfield-Mask: 0x01)                */
#define I2C_INTRSTAT_R_RX_FULL_Pos               (2UL)                        /*!< R_RX_FULL (Bit 2)                              */
#define I2C_INTRSTAT_R_RX_FULL_Msk               (0x4UL)                      /*!< R_RX_FULL (Bitfield-Mask: 0x01)                */
#define I2C_INTRSTAT_R_RX_OVER_Pos               (1UL)                        /*!< R_RX_OVER (Bit 1)                              */
#define I2C_INTRSTAT_R_RX_OVER_Msk               (0x2UL)                      /*!< R_RX_OVER (Bitfield-Mask: 0x01)                */
#define I2C_INTRSTAT_R_RX_UNDER_Pos              (0UL)                        /*!< R_RX_UNDER (Bit 0)                             */
#define I2C_INTRSTAT_R_RX_UNDER_Msk              (0x1UL)                      /*!< R_RX_UNDER (Bitfield-Mask: 0x01)               */
/* ===============================  INTRMASK  =============================== */
#define I2C_INTRMASK_M_ADDR1_MATCH_Pos            (18UL)                       /*!< M_ADDR1_MATCH (Bit 18)                         */
#define I2C_INTRMASK_M_ADDR1_MATCH_Msk            (0x00040000UL)               /*!< M_ADDR1_MATCH (Bitfield-Mask: 0x01)            */
#define I2C_INTRMASK_M_ADDR0_MATCH_Pos            (17UL)                       /*!< M_ADDR0_MATCH (Bit 17)                         */
#define I2C_INTRMASK_M_ADDR0_MATCH_Msk            (0x00020000UL)               /*!< M_ADDR0_MATCH (Bitfield-Mask: 0x01)            */
#define I2C_INTRMASK_M_PEC_ERROR_Pos              (16UL)                       /*!< M_PEC_ERROR (Bit 16)                           */
#define I2C_INTRMASK_M_PEC_ERROR_Msk              (0x00010000UL)               /*!< M_PEC_ERROR (Bitfield-Mask: 0x01)              */
#define I2C_INTRMASK_M_TIMEOUT_Pos                (15UL)                       /*!< M_TIMEOUT (Bit 15)                             */
#define I2C_INTRMASK_M_TIMEOUT_Msk                (0x00008000UL)               /*!< M_TIMEOUT (Bitfield-Mask: 0x01)                */
#define I2C_INTRMASK_M_SMBUS_ALERT_Pos            (14UL)                       /*!< M_SMBUS_ALERT (Bit 14)                         */
#define I2C_INTRMASK_M_SMBUS_ALERT_Msk            (0x00004000UL)               /*!< M_SMBUS_ALERT (Bitfield-Mask: 0x01)            */
#define I2C_INTRMASK_M_MST_ON_HOLD_Pos            (13UL)                       /*!< M_MST_ON_HOLD (Bit 13)                         */
#define I2C_INTRMASK_M_MST_ON_HOLD_Msk            (0x2000UL)                   /*!< M_MST_ON_HOLD (Bitfield-Mask: 0x01)            */
#define I2C_INTRMASK_M_RESTART_DET_Pos            (12UL)                       /*!< M_RESTART_DET (Bit 12)                         */
#define I2C_INTRMASK_M_RESTART_DET_Msk            (0x1000UL)                   /*!< M_RESTART_DET (Bitfield-Mask: 0x01)            */
#define I2C_INTRMASK_M_GEN_CALL_Pos               (11UL)                       /*!< M_GEN_CALL (Bit 11)                            */
#define I2C_INTRMASK_M_GEN_CALL_Msk               (0x800UL)                    /*!< M_GEN_CALL (Bitfield-Mask: 0x01)               */
#define I2C_INTRMASK_M_START_DET_Pos              (10UL)                       /*!< M_START_DET (Bit 10)                           */
#define I2C_INTRMASK_M_START_DET_Msk              (0x400UL)                    /*!< M_START_DET (Bitfield-Mask: 0x01)              */
#define I2C_INTRMASK_M_STOP_DET_Pos               (9UL)                        /*!< M_STOP_DET (Bit 9)                             */
#define I2C_INTRMASK_M_STOP_DET_Msk               (0x200UL)                    /*!< M_STOP_DET (Bitfield-Mask: 0x01)               */
#define I2C_INTRMASK_M_ACTIVITY_Pos               (8UL)                        /*!< M_ACTIVITY (Bit 8)                             */
#define I2C_INTRMASK_M_ACTIVITY_Msk               (0x100UL)                    /*!< M_ACTIVITY (Bitfield-Mask: 0x01)               */
#define I2C_INTRMASK_M_RX_DONE_Pos                (7UL)                        /*!< M_RX_DONE (Bit 7)                              */
#define I2C_INTRMASK_M_RX_DONE_Msk                (0x80UL)                     /*!< M_RX_DONE (Bitfield-Mask: 0x01)                */
#define I2C_INTRMASK_M_TX_ABRT_Pos                (6UL)                        /*!< M_TX_ABRT (Bit 6)                              */
#define I2C_INTRMASK_M_TX_ABRT_Msk                (0x40UL)                     /*!< M_TX_ABRT (Bitfield-Mask: 0x01)                */
#define I2C_INTRMASK_M_RD_REQ_Pos                 (5UL)                        /*!< M_RD_REQ (Bit 5)                               */
#define I2C_INTRMASK_M_RD_REQ_Msk                 (0x20UL)                     /*!< M_RD_REQ (Bitfield-Mask: 0x01)                 */
#define I2C_INTRMASK_M_TX_EMPTY_Pos               (4UL)                        /*!< M_TX_EMPTY (Bit 4)                             */
#define I2C_INTRMASK_M_TX_EMPTY_Msk               (0x10UL)                     /*!< M_TX_EMPTY (Bitfield-Mask: 0x01)               */
#define I2C_INTRMASK_M_TX_OVER_Pos                (3UL)                        /*!< M_TX_OVER (Bit 3)                              */
#define I2C_INTRMASK_M_TX_OVER_Msk                (0x8UL)                      /*!< M_TX_OVER (Bitfield-Mask: 0x01)                */
#define I2C_INTRMASK_M_RX_FULL_Pos                (2UL)                        /*!< M_RX_FULL (Bit 2)                              */
#define I2C_INTRMASK_M_RX_FULL_Msk                (0x4UL)                      /*!< M_RX_FULL (Bitfield-Mask: 0x01)                */
#define I2C_INTRMASK_M_RX_OVER_Pos                (1UL)                        /*!< M_RX_OVER (Bit 1)                              */
#define I2C_INTRMASK_M_RX_OVER_Msk                (0x2UL)                      /*!< M_RX_OVER (Bitfield-Mask: 0x01)                */
#define I2C_INTRMASK_M_RX_UNDER_Pos               (0UL)                        /*!< M_RX_UNDER (Bit 0)                             */
#define I2C_INTRMASK_M_RX_UNDER_Msk               (0x1UL)                      /*!< M_RX_UNDER (Bitfield-Mask: 0x01)               */
/* ==============================  RAWINTRSTAT  ============================= */
#define I2C_RAWINTRSTAT_ADDR1_MATCH_Pos           (18UL)                       /*!< ADDR1_MATCH (Bit 18)                           */
#define I2C_RAWINTRSTAT_ADDR1_MATCH_Msk           (0x00040000UL)               /*!< ADDR1_MATCH (Bitfield-Mask: 0x01)              */
#define I2C_RAWINTRSTAT_ADDR0_MATCH_Pos           (17UL)                       /*!< ADDR0_MATCH (Bit 17)                           */
#define I2C_RAWINTRSTAT_ADDR0_MATCH_Msk           (0x00020000UL)               /*!< ADDR0_MATCH (Bitfield-Mask: 0x01)              */
#define I2C_RAWINTRSTAT_PEC_ERROR_Pos             (16UL)                       /*!< PEC_ERROR (Bit 16)                             */
#define I2C_RAWINTRSTAT_PEC_ERROR_Msk             (0x00010000UL)               /*!< PEC_ERROR (Bitfield-Mask: 0x01)                */
#define I2C_RAWINTRSTAT_TIMEOUT_Pos               (15UL)                       /*!< TIMEOUT (Bit 15)                               */
#define I2C_RAWINTRSTAT_TIMEOUT_Msk               (0x00008000UL)               /*!< TIMEOUT (Bitfield-Mask: 0x01)                  */
#define I2C_RAWINTRSTAT_SMBUS_ALERT_Pos           (14UL)                       /*!< SMBUS_ALERT (Bit 14)                           */
#define I2C_RAWINTRSTAT_SMBUS_ALERT_Msk           (0x00004000UL)               /*!< SMBUS_ALERT (Bitfield-Mask: 0x01)              */
#define I2C_RAWINTRSTAT_MST_ON_HOLD_Pos           (13UL)                       /*!< MST_ON_HOLD (Bit 13)                           */
#define I2C_RAWINTRSTAT_MST_ON_HOLD_Msk           (0x2000UL)                   /*!< MST_ON_HOLD (Bitfield-Mask: 0x01)              */
#define I2C_RAWINTRSTAT_RESTART_DET_Pos           (12UL)                       /*!< RESTART_DET (Bit 12)                           */
#define I2C_RAWINTRSTAT_RESTART_DET_Msk           (0x1000UL)                   /*!< RESTART_DET (Bitfield-Mask: 0x01)              */
#define I2C_RAWINTRSTAT_GEN_CALL_Pos              (11UL)                       /*!< GEN_CALL (Bit 11)                              */
#define I2C_RAWINTRSTAT_GEN_CALL_Msk              (0x800UL)                    /*!< GEN_CALL (Bitfield-Mask: 0x01)                 */
#define I2C_RAWINTRSTAT_START_DET_Pos             (10UL)                       /*!< START_DET (Bit 10)                             */
#define I2C_RAWINTRSTAT_START_DET_Msk             (0x400UL)                    /*!< START_DET (Bitfield-Mask: 0x01)                */
#define I2C_RAWINTRSTAT_STOP_DET_Pos              (9UL)                        /*!< STOP_DET (Bit 9)                               */
#define I2C_RAWINTRSTAT_STOP_DET_Msk              (0x200UL)                    /*!< STOP_DET (Bitfield-Mask: 0x01)                 */
#define I2C_RAWINTRSTAT_ACTIVITY_Pos              (8UL)                        /*!< ACTIVITY (Bit 8)                               */
#define I2C_RAWINTRSTAT_ACTIVITY_Msk              (0x100UL)                    /*!< ACTIVITY (Bitfield-Mask: 0x01)                 */
#define I2C_RAWINTRSTAT_RX_DONE_Pos               (7UL)                        /*!< RX_DONE (Bit 7)                                */
#define I2C_RAWINTRSTAT_RX_DONE_Msk               (0x80UL)                     /*!< RX_DONE (Bitfield-Mask: 0x01)                  */
#define I2C_RAWINTRSTAT_TX_ABRT_Pos               (6UL)                        /*!< TX_ABRT (Bit 6)                                */
#define I2C_RAWINTRSTAT_TX_ABRT_Msk               (0x40UL)                     /*!< TX_ABRT (Bitfield-Mask: 0x01)                  */
#define I2C_RAWINTRSTAT_RD_REQ_Pos                (5UL)                        /*!< RD_REQ (Bit 5)                                 */
#define I2C_RAWINTRSTAT_RD_REQ_Msk                (0x20UL)                     /*!< RD_REQ (Bitfield-Mask: 0x01)                   */
#define I2C_RAWINTRSTAT_TX_EMPTY_Pos              (4UL)                        /*!< TX_EMPTY (Bit 4)                               */
#define I2C_RAWINTRSTAT_TX_EMPTY_Msk              (0x10UL)                     /*!< TX_EMPTY (Bitfield-Mask: 0x01)                 */
#define I2C_RAWINTRSTAT_TX_OVER_Pos               (3UL)                        /*!< TX_OVER (Bit 3)                                */
#define I2C_RAWINTRSTAT_TX_OVER_Msk               (0x8UL)                      /*!< TX_OVER (Bitfield-Mask: 0x01)                  */
#define I2C_RAWINTRSTAT_RX_FULL_Pos               (2UL)                        /*!< RX_FULL (Bit 2)                                */
#define I2C_RAWINTRSTAT_RX_FULL_Msk               (0x4UL)                      /*!< RX_FULL (Bitfield-Mask: 0x01)                  */
#define I2C_RAWINTRSTAT_RX_OVER_Pos               (1UL)                        /*!< RX_OVER (Bit 1)                                */
#define I2C_RAWINTRSTAT_RX_OVER_Msk               (0x2UL)                      /*!< RX_OVER (Bitfield-Mask: 0x01)                  */
#define I2C_RAWINTRSTAT_RX_UNDER_Pos              (0UL)                        /*!< RX_UNDER (Bit 0)                               */
#define I2C_RAWINTRSTAT_RX_UNDER_Msk              (0x1UL)                      /*!< RX_UNDER (Bitfield-Mask: 0x01)                 */
/* ====================================  RXTL  ============================== */
#define I2C_RXTL_RX_TL_Pos                        (0UL)                        /*!< RX_TL (Bit 0)                                  */
#define I2C_RXTL_RX_TL_Msk                        (0x00000007UL)               /*!< RX_TL (Bitfield-Mask: 0x07)                    */
/* ====================================  TXTL  ============================== */
#define I2C_TXTL_TX_TL_Pos                        (0UL)                        /*!< TX_TL (Bit 0)                                  */
#define I2C_TXTL_TX_TL_Msk                        (0x00000007UL)               /*!< TX_TL (Bitfield-Mask: 0x07)                    */
/* ==================================  CLRINTR  ============================= */
#define I2C_CLRINTR_CLR_INTR_Pos                  (0UL)                        /*!< CLR_INTR (Bit 0)                               */
#define I2C_CLRINTR_CLR_INTR_Msk                  (0x00000001UL)               /*!< CLR_INTR (Bitfield-Mask: 0x01)                 */
/* =================================  CLRRXUNDER  =========================== */
#define I2C_CLRRXUNDER_CLR_RX_UNDER_Pos           (0UL)                        /*!< CLR_RX_UNDER (Bit 0)                           */
#define I2C_CLRRXUNDER_CLR_RX_UNDER_Msk           (0x00000001UL)               /*!< CLR_RX_UNDER (Bitfield-Mask: 0x01)             */
/* ================================== CLRRXOVER  ============================ */
#define I2C_CLRRXOVER_CLR_RX_OVER_Pos             (0UL)                        /*!< CLR_RX_OVER (Bit 0)                            */
#define I2C_CLRRXOVER_CLR_RX_OVER_Msk             (0x00000001UL)               /*!< CLR_RX_OVER (Bitfield-Mask: 0x01)              */
/* =================================  CLRTXOVER  ============================ */
#define I2C_CLRTXOVER_CLR_TX_OVER_Pos             (0UL)                        /*!< CLR_TX_OVER (Bit 0)                            */
#define I2C_CLRTXOVER_CLR_TX_OVER_Msk             (0x00000001UL)               /*!< CLR_TX_OVER (Bitfield-Mask: 0x01)              */
/* ==================================  CLRRDREQ  ============================ */
#define I2C_CLRRDREQ_CLR_RD_REQ_Pos               (0UL)                        /*!< CLR_RD_REQ (Bit 0)                             */
#define I2C_CLRRDREQ_CLR_RD_REQ_Msk               (0x00000001UL)               /*!< CLR_RD_REQ (Bitfield-Mask: 0x01)               */
/* =================================  CLRTXABRT  ============================ */
#define I2C_CLRTXABRT_CLR_TX_ABRT_Pos             (0UL)                        /*!< CLR_TX_ABRT (Bit 0)                            */
#define I2C_CLRTXABRT_CLR_TX_ABRT_Msk             (0x00000001UL)               /*!< CLR_TX_ABRT (Bitfield-Mask: 0x01)              */
/* =================================  CLRRXDONE  ============================ */
#define I2C_CLRRXDONE_CLR_RX_DONE_Pos             (0UL)                        /*!< CLR_RX_DONE (Bit 0)                            */
#define I2C_CLRRXDONE_CLR_RX_DONE_Msk             (0x00000001UL)               /*!< CLR_RX_DONE (Bitfield-Mask: 0x01)              */
/* ================================  CLRACTIVITY  =========================== */
#define I2C_CLRACTIVITY_CLR_ACTIVITY_Pos          (0UL)                        /*!< CLR_ACTIVITY (Bit 0)                           */
#define I2C_CLRACTIVITY_CLR_ACTIVITY_Msk          (0x00000001UL)               /*!< CLR_ACTIVITY (Bitfield-Mask: 0x01)             */
/* ================================  CLRSTOP_DET  =========================== */
#define I2C_CLRSTOP_DET_CLR_STOP_DET_Pos          (0UL)                        /*!< CLR_STOP_DET (Bit 0)                           */
#define I2C_CLRSTOP_DET_CLR_STOP_DET_Msk          (0x00000001UL)               /*!< CLR_STOP_DET (Bitfield-Mask: 0x01)             */
/* ================================  CLRSTARTDET  ============================*/
#define I2C_CLRSTARTDET_CLR_START_DET_Pos         (0UL)                        /*!< CLR_START_DET (Bit 0)                          */
#define I2C_CLRSTARTDET_CLR_START_DET_Msk         (0x00000001UL)               /*!< CLR_START_DET (Bitfield-Mask: 0x01)            */
/* =================================  CLRGENCALL  =========================== */
#define I2C_CLRGENCALL_CLR_GEN_CALL_Pos           (0UL)                        /*!< CLR_GEN_CALL (Bit 0)                           */
#define I2C_CLRGENCALL_CLR_GEN_CALL_Msk           (0x00000001UL)               /*!< CLR_GEN_CALL (Bitfield-Mask: 0x01)             */
/* ==================================  ENABLE  ============================== */
#define I2C_ENABLE_SMBUS_ALERT_Pos                (4UL)                        /*!< SMBUS_ALERT (Bit 4)                            */
#define I2C_ENABLE_SMBUS_ALERT_Msk                (0x00000010UL)               /*!< SMBUS_ALERT (Bitfield-Mask: 0x01)              */
#define I2C_ENABLE_ACK_MODE_Pos                   (3UL)                        /*!< ACK_MODE (Bit 3)                               */
#define I2C_ENABLE_ACK_MODE_Msk                   (0x00000008UL)               /*!< ACK_MODE (Bitfield-Mask: 0x01)                 */
#define I2C_ENABLE_TX_CMD_BLOCK_Pos               (2UL)                        /*!< TX_CMD_BLOCK (Bit 2)                           */
#define I2C_ENABLE_TX_CMD_BLOCK_Msk               (0x00000004UL)               /*!< TX_CMD_BLOCK (Bitfield-Mask: 0x01)             */
#define I2C_ENABLE_ABORT_Pos                      (1UL)                        /*!< ABORT (Bit 1)                                  */
#define I2C_ENABLE_ABORT_Msk                      (0x00000002UL)               /*!< ABORT (Bitfield-Mask: 0x01)                    */
#define I2C_ENABLE_ENABLE_Pos                     (0UL)                        /*!< ENABLE (Bit 0)                                 */
#define I2C_ENABLE_ENABLE_Msk                     (0x00000001UL)               /*!< ENABLE (Bitfield-Mask: 0x01)                   */
/* ===================================  STATUS  ============================= */
#define I2C_STATUS_SLV_HOLD_RX_FIFO_FULL_Pos      (10UL)                       /*!< SLV_HOLD_RX_FIFO_FULL (Bit 10)                 */
#define I2C_STATUS_SLV_HOLD_RX_FIFO_FULL_Msk      (0x00000400UL)               /*!< SLV_HOLD_RX_FIFO_FULL (Bitfield-Mask: 0x01)    */
#define I2C_STATUS_SLV_HOLD_TX_FIFO_EMPTY_Pos     (9UL)                        /*!< SLV_HOLD_TX_FIFO_EMPTY (Bit 9)                 */
#define I2C_STATUS_SLV_HOLD_TX_FIFO_EMPTY_Msk     (0x00000200UL)               /*!< SLV_HOLD_TX_FIFO_EMPTY (Bitfield-Mask: 0x01)   */
#define I2C_STATUS_MST_HOLD_RX_FIFO_FULL_Pos      (8UL)                        /*!< MST_HOLD_RX_FIFO_FULL (Bit 8)                  */
#define I2C_STATUS_MST_HOLD_RX_FIFO_FULL_Msk      (0x00000100UL)               /*!< MST_HOLD_RX_FIFO_FULL (Bitfield-Mask: 0x01)    */
#define I2C_STATUS_MST_HOLD_TX_FIFO_EMPTY_Pos     (7UL)                        /*!< MST_HOLD_TX_FIFO_EMPTY (Bit 7)                 */
#define I2C_STATUS_MST_HOLD_TX_FIFO_EMPTY_Msk     (0x00000080UL)               /*!< MST_HOLD_TX_FIFO_EMPTY (Bitfield-Mask: 0x01)   */
#define I2C_STATUS_SLV_ACTIVITY_Pos               (6UL)                        /*!< SLV_ACTIVITY (Bit 6)                           */
#define I2C_STATUS_SLV_ACTIVITY_Msk               (0x00000040UL)               /*!< SLV_ACTIVITY (Bitfield-Mask: 0x01)             */
#define I2C_STATUS_MST_ACTIVITY_Pos               (5UL)                        /*!< MST_ACTIVITY (Bit 5)                           */
#define I2C_STATUS_MST_ACTIVITY_Msk               (0x00000020UL)               /*!< MST_ACTIVITY (Bitfield-Mask: 0x01)             */
#define I2C_STATUS_RFF_Pos                        (4UL)                        /*!< RFF (Bit 4)                                    */
#define I2C_STATUS_RFF_Msk                        (0x00000010UL)               /*!< RFF (Bitfield-Mask: 0x01)                      */
#define I2C_STATUS_RFNF_Pos                       (3UL)                        /*!< RFNE (Bit 3)                                   */
#define I2C_STATUS_RFNF_Msk                       (0x00000008UL)               /*!< RFNE (Bitfield-Mask: 0x01)                     */
#define I2C_STATUS_TFE_Pos                        (2UL)                        /*!< TFE (Bit 2)                                    */
#define I2C_STATUS_TFE_Msk                        (0x00000004UL)               /*!< TFE (Bitfield-Mask: 0x01)                      */
#define I2C_STATUS_TFNF_Pos                       (1UL)                        /*!< TFNF (Bit 1)                                   */
#define I2C_STATUS_TFNF_Msk                       (0x00000002UL)               /*!< TFNF (Bitfield-Mask: 0x01)                     */
#define I2C_STATUS_ACTIVITY_Pos                   (0UL)                        /*!< ACTIVITY (Bit 0)                               */
#define I2C_STATUS_ACTIVITY_Msk                   (0x00000001UL)               /*!< ACTIVITY (Bitfield-Mask: 0x01)                 */
/* ===================================  TXFLR  ============================== */
#define I2C_TXFLR_TXFLR_Pos                       (0UL)                        /*!< TXFLR (Bit 0)                                  */
#define I2C_TXFLR_TXFLR_Msk                       (0x0000000FUL)               /*!< TXFLR (Bitfield-Mask: 0x0F)                    */
/* ===================================  RXFLR  ============================== */
#define I2C_RXFLR_RXFLR_Pos                       (0UL)                        /*!< RXFLR (Bit 0)                                  */
#define I2C_RXFLR_RXFLR_Msk                       (0x0000000FUL)               /*!< RXFLR (Bitfield-Mask: 0x0F)                    */
/* ==================================  SDAHOLD  ============================= */
#define I2C_SDAHOLD_IC_SDA_RX_HOLD_Pos            (16UL)                       /*!< IC_SDA_RX_HOLD (Bit 16)                        */
#define I2C_SDAHOLD_IC_SDA_RX_HOLD_Msk            (0x00FF0000UL)               /*!< IC_SDA_RX_HOLD (Bitfield-Mask: 0xff)           */
#define I2C_SDAHOLD_IC_SDA_TX_HOLD_Pos            (0UL)                        /*!< IC_SDA_TX_HOLD (Bit 0)                         */
#define I2C_SDAHOLD_IC_SDA_TX_HOLD_Msk            (0x0000FFFFUL)               /*!< IC_SDA_TX_HOLD (Bitfield-Mask: 0xffff)         */
/* ================================  TXABRTSOURCE  ========================== */
#define I2C_TXABRTSOURCE_TX_FLUSH_CNT_Pos         (23UL)                       /*!< TX_FLUSH_CNT (Bit 23)                          */
#define I2C_TXABRTSOURCE_TX_FLUSH_CNT_Msk         (0xFF800000UL)               /*!< TX_FLUSH_CNT (Bitfield-Mask: 0x1FF)            */
#define I2C_TXABRTSOURCE_ABRT_USER_ABRT_Pos       (16UL)                       /*!< ABRT_USER_ABRT (Bit 16)                        */
#define I2C_TXABRTSOURCE_ABRT_USER_ABRT_Msk       (0x00010000UL)               /*!< ABRT_USER_ABRT (Bitfield-Mask: 0x01)           */
#define I2C_TXABRTSOURCE_ABRT_SLVRD_INTX_Pos      (15UL)                       /*!< ABRT_SLVRD_INTX (Bit 15)                       */
#define I2C_TXABRTSOURCE_ABRT_SLVRD_INTX_Msk      (0x00008000UL)               /*!< ABRT_SLVRD_INTX (Bitfield-Mask: 0x01)          */
#define I2C_TXABRTSOURCE_ABRT_SLV_ARBLOST_Pos     (14UL)                       /*!< ABRT_SLV_ARBLOST (Bit 14)                      */
#define I2C_TXABRTSOURCE_ABRT_SLV_ARBLOST_Msk     (0x00004000UL)               /*!< ABRT_SLV_ARBLOST (Bitfield-Mask: 0x01)         */
#define I2C_TXABRTSOURCE_ABRT_SLVFLUSH_TXFIFO_Pos (13UL)                       /*!< ABRT_SLVFLUSH_TXFIFO (Bit 13)                  */
#define I2C_TXABRTSOURCE_ABRT_SLVFLUSH_TXFIFO_Msk (0x00002000UL)               /*!< ABRT_SLVFLUSH_TXFIFO (Bitfield-Mask: 0x01)     */
#define I2C_TXABRTSOURCE_ARB_LOST_Pos             (12UL)                       /*!< ARB_LOST (Bit 12)                              */
#define I2C_TXABRTSOURCE_ARB_LOST_Msk             (0x00001000UL)               /*!< ARB_LOST (Bitfield-Mask: 0x01)                 */
#define I2C_TXABRTSOURCE_ABRT_MASTER_DIS_Pos      (11UL)                       /*!< ABRT_MASTER_DIS (Bit 11)                       */
#define I2C_TXABRTSOURCE_ABRT_MASTER_DIS_Msk      (0x00000800UL)               /*!< ABRT_MASTER_DIS (Bitfield-Mask: 0x01)          */
#define I2C_TXABRTSOURCE_ABRT_10B_RD_NORSTRT_Pos  (10UL)                       /*!< ABRT_10B_RD_NORSTRT (Bit 10)                   */
#define I2C_TXABRTSOURCE_ABRT_10B_RD_NORSTRT_Msk  (0x00000400UL)               /*!< ABRT_10B_RD_NORSTRT (Bitfield-Mask: 0x01)      */
#define I2C_TXABRTSOURCE_ABRT_SBYTE_NORSTRT_Pos   (9UL)                        /*!< ABRT_SBYTE_NORSTRT (Bit 9)                     */
#define I2C_TXABRTSOURCE_ABRT_SBYTE_NORSTRT_Msk   (0x00000200UL)               /*!< ABRT_SBYTE_NORSTRT (Bitfield-Mask: 0x01)       */
#define I2C_TXABRTSOURCE_ABRT_HS_NORSTRT_Pos      (8UL)                        /*!< ABRT_HS_NORSTRT (Bit 8)                        */
#define I2C_TXABRTSOURCE_ABRT_HS_NORSTRT_Msk      (0x00000100UL)               /*!< ABRT_HS_NORSTRT (Bitfield-Mask: 0x01)          */
#define I2C_TXABRTSOURCE_ABRT_SBYTE_ACKDET_Pos    (7UL)                        /*!< ABRT_SBYTE_ACKDET (Bit 7)                      */
#define I2C_TXABRTSOURCE_ABRT_SBYTE_ACKDET_Msk    (0x00000080UL)               /*!< ABRT_SBYTE_ACKDET (Bitfield-Mask: 0x01)        */
#define I2C_TXABRTSOURCE_ABRT_HS_ACKDET_Pos       (6UL)                        /*!< ABRT_HS_ACKDET (Bit 6)                         */
#define I2C_TXABRTSOURCE_ABRT_HS_ACKDET_Msk       (0x00000040UL)               /*!< ABRT_HS_ACKDET (Bitfield-Mask: 0x01)           */
#define I2C_TXABRTSOURCE_ABRT_GCALL_READ_Pos      (5UL)                        /*!< ABRT_GCALL_READ (Bit 5)                        */
#define I2C_TXABRTSOURCE_ABRT_GCALL_READ_Msk      (0x00000020UL)               /*!< ABRT_GCALL_READ (Bitfield-Mask: 0x01)          */
#define I2C_TXABRTSOURCE_ABRT_GCALL_NOACK_Pos     (4UL)                        /*!< ABRT_GCALL_NOACK (Bit 4)                       */
#define I2C_TXABRTSOURCE_ABRT_GCALL_NOACK_Msk     (0x00000010UL)               /*!< ABRT_GCALL_NOACK (Bitfield-Mask: 0x01)         */
#define I2C_TXABRTSOURCE_ABRT_TXDATA_NOACK_Pos    (3UL)                        /*!< ABRT_TXDATA_NOACK (Bit 3)                      */
#define I2C_TXABRTSOURCE_ABRT_TXDATA_NOACK_Msk    (0x00000008UL)               /*!< ABRT_TXDATA_NOACK (Bitfield-Mask: 0x01)        */
#define I2C_TXABRTSOURCE_ABRT_10ADDR2_NOACK_Pos   (2UL)                        /*!< ABRT_10ADDR2_NOACK (Bit 2)                     */
#define I2C_TXABRTSOURCE_ABRT_10ADDR2_NOACK_Msk   (0x00000004UL)               /*!< ABRT_10ADDR2_NOACK (Bitfield-Mask: 0x01)       */
#define I2C_TXABRTSOURCE_ABRT_10ADDR1_NOACK_Pos   (1UL)                        /*!< ABRT_10ADDR1_NOACK (Bit 1)                     */
#define I2C_TXABRTSOURCE_ABRT_10ADDR1_NOACK_Msk   (0x00000002UL)               /*!< ABRT_10ADDR1_NOACK (Bitfield-Mask: 0x01)       */
#define I2C_TXABRTSOURCE_ABRT_7B_ADDR_NOACK_Pos   (0UL)                        /*!< ABRT_7B_ADDR_NOACK (Bit 0)                     */
#define I2C_TXABRTSOURCE_ABRT_7B_ADDR_NOACK_Msk   (0x00000001UL)               /*!< ABRT_7B_ADDR_NOACK (Bitfield-Mask: 0x01)       */
/* ================================  SLVDATANACKONLY  ======================= */
#define I2C_SLVDATANACKONLY_NACK_Pos              (0UL)                        /*!< NACK (Bit 0)                                   */
#define I2C_SLVDATANACKONLY_NACK_Msk              (0x00000001UL)               /*!< NACK (Bitfield-Mask: 0x01)                     */
/* ====================================  DMACR  ============================= */
#define I2C_DMACR_TDMAE_Pos                       (1UL)                        /*!< TDMAE (Bit 1)                                  */
#define I2C_DMACR_TDMAE_Msk                       (0x00000002UL)               /*!< TDMAE (Bitfield-Mask: 0x01)                    */
#define I2C_DMACR_RDMAE_Pos                       (0UL)                        /*!< RDMAE (Bit 0)                                  */
#define I2C_DMACR_RDMAE_Msk                       (0x00000001UL)               /*!< RDMAE (Bitfield-Mask: 0x01)                    */
/* ===================================  DMATDLR  ============================ */
#define I2C_DMATDLR_DMATDL_Pos                    (0UL)                        /*!< DMATDL (Bit 0)                                 */
#define I2C_DMATDLR_DMATDL_Msk                    (0x00000007UL)               /*!< DMATDL (Bitfield-Mask: 0x07)                   */
/* ===================================  DMARDLR  ============================ */
#define I2C_DMARDLR_DMARDL_Pos                    (0UL)                        /*!< DMARDL (Bit 0)                                 */
#define I2C_DMARDLR_DMARDL_Msk                    (0x00000007UL)               /*!< DMARDL (Bitfield-Mask: 0x07)                   */
/* ===================================  SDASETUP  =========================== */
#define I2C_SDASETUP_SDA_SETUP_Pos                (0UL)                        /*!< SDA_SETUP (Bit 0)                              */
#define I2C_SDASETUP_SDA_SETUP_Msk                (0x000000FFUL)               /*!< SDA_SETUP (Bitfield-Mask: 0xFF)                */
/* ================================  ACKGENERALCALL  ======================== */
#define I2C_ACKGENERALCALL_ACK_GEN_CALL_Pos       (0UL)                        /*!< ACK_GEN_CALL (Bit 0)                           */
#define I2C_ACKGENERALCALL_ACK_GEN_CALL_Msk       (0x00000001UL)               /*!< ACK_GEN_CALL (Bitfield-Mask: 0x01)             */
/* ================================  ENABLESTATUS  ========================== */
#define I2C_ENABLESTATUS_SLV_RX_DATA_LOST_Pos     (2UL)                        /*!< SLV_RX_DATA_LOST (Bit 2)                       */
#define I2C_ENABLESTATUS_SLV_RX_DATA_LOST_Msk     (0x00000004UL)               /*!< SLV_RX_DATA_LOST (Bitfield-Mask: 0x01)         */
#define I2C_ENABLESTATUS_SLV_DIS_WHILE_BUSY_Pos   (1UL)                        /*!< SLV_DISABLED_WHILE_BUSY (Bit 1)                */
#define I2C_ENABLESTATUS_SLV_DIS_WHILE_BUSY_Msk   (0x00000002UL)               /*!< SLV_DISABLED_WHILE_BUSY (Bitfield-Mask: 0x01)  */
#define I2C_ENABLESTATUS_IC_EN_Pos                (0UL)                        /*!< IC_EN (Bit 0)                                  */
#define I2C_ENABLESTATUS_IC_EN_Msk                (0x00000001UL)               /*!< IC_EN (Bitfield-Mask: 0x01)                    */
/* ==================================  FSSPKLEN  ============================ */
#define I2C_FSSPKLEN_IC_FS_SPKLEN_Pos             (0UL)                        /*!< IC_FS_SPKLEN (Bit 0)                           */
#define I2C_FSSPKLEN_IC_FS_SPKLEN_Msk             (0x00000001UL)               /*!< IC_FS_SPKLEN (Bitfield-Mask: 0x01)             */
/* ================================  CLRRESTARTDET  ========================= */
#define I2C_CLRRESTARTDET_CLR_RESTART_DET_Pos     (0UL)                        /*!< CLR_RESTART_DET (Bit 0)                        */
#define I2C_CLRRESTARTDET_CLR_RESTART_DET_Msk     (0x00000001UL)               /*!< CLR_RESTART_DET (Bitfield-Mask: 0x01)          */
/* ==============================    I2C_CLR_TIMEOUT    ===================== */
#define I2C_CLR_TIMEOUT_Pos                       (0UL)                        /*!< CLR_TIMEOUT (Bit 0)                            */
#define I2C_CLR_TIMEOUT_Msk                       (0x00000001UL)               /*!< CLR_TIMEOUT (Bitfield-Ma                       */
/* ===============================  I2C_CLR_SMBUS_ALERT  ==================== */
#define I2C_CLR_SMBUS_ALERT_Pos                   (0UL)                        /*!< SMBUS_ALERT (Bit 0)                            */
#define I2C_CLR_SMBUS_ALERT_Msk                   (0x00000001UL)               /*!< SMBUS_ALERT (Bitfield-Mask: 0x1)               */
/* ==============================   I2C_CLR_PEC_ERROR   ===================== */
#define I2C_CLR_PEC_ERROR_Pos                     (0UL)                        /*!< CLR_PEC_ERROR (Bit 0)                          */
#define I2C_CLR_PEC_ERROR_Msk                     (0x00000001UL)               /*!< CLR_PEC_ERROR (Bitfield-Mask: 0x1)             */
/* =============================    I2C_TIMEOUT_DATA   ====================== */
#define I2C_TIMEOUT_DATA_Pos                      (0UL)                        /*!< TIMEOUT_DATA (Bit 0)                           */
#define I2C_TIMEOUT_DATA_Msk                      (0x0000FFFFUL)               /*!< TIMEOUT_DATA (Bitfield-Mask: 0xFFFF)           */
/* =============================      I2C_PEC_DATA     ====================== */
#define I2C_PEC_DATA_Pos                          (0UL)                        /*!< PEC_DATA (Bit 0)                               */
#define I2C_PEC_DATA_Msk                          (0x000000FFUL)               /*!< PEC_DATA (Bitfield-Mask: 0xFF)                 */
/* =============================  I2C_CLR_ADDR0_MATCH  ====================== */
#define I2C_CLR_ADDR0_MATCH_Pos                   (0UL)                        /*!< ADDR0_MATCH (Bit 0)                            */
#define I2C_CLR_ADDR0_MATCH_Msk                   (0x00000001UL)               /*!< ADDR0_MATCH (Bitfield-Mask: 0x1)               */
/* =============================  I2C_CLR_ADDR1_MATCH  ====================== */
#define I2C_CLR_ADDR1_MATCH_Pos                   (0UL)                        /*!< ADDR1_MATCH (Bit 0)                            */
#define I2C_CLR_ADDR1_MATCH_Msk                   (0x00000001UL)               /*!< ADDR1_MATCH (Bitfield-Mask: 0x1)               */

/* ========================================================================== */
/* ================                  END                      =============== */
/* ========================================================================== */

void I2C_DeInit(I2C_TypeDef *I2Cx);
void SlaveAckGeneralCall(I2C_TypeDef *I2Cx, bool status);
void I2C_GenerateGeneralCall(I2C_TypeDef *I2Cx);
void I2C_ClearGeneralCall(I2C_TypeDef *I2Cx);
void I2C_GenerateStartByte(I2C_TypeDef *I2Cx);
void I2C_SetFrequent(I2C_TypeDef *I2Cx, int sclClock);
void I2C_TxFifoITthreshold(uint32_t TxFifoValue);
void I2C_RxFifoITthreshold(uint32_t RxFifoValue);
void I2C_TransferHandling(I2C_TypeDef *I2Cx, uint32_t Address);
void I2C_SlaveAddressConfig(I2C_TypeDef *I2Cx, uint32_t Address);
void I2C_Cmd(I2C_TypeDef *I2Cx, FunctionalState NewState);
void I2C_StructInit(I2C_InitTypeDef *I2C_InitStruct);
void I2C_Init(I2C_TypeDef *I2Cx, I2C_InitTypeDef *I2C_InitStruct);
uint32_t I2C_ReadRegister(I2C_TypeDef *I2Cx, uint8_t I2C_Register);
uint8_t I2C_ReadData(I2C_TypeDef *I2Cx);
uint8_t I2C_ReadData_Stop(I2C_TypeDef *I2Cx);
void I2C_WriteData(I2C_TypeDef *I2Cx, uint8_t Data);
void I2C_WriteData_Stop(I2C_TypeDef *I2Cx, uint8_t Data);
uint32_t I2C_ReceiveData(I2C_TypeDef *I2Cx);
void I2C_SendData(I2C_TypeDef *I2Cx, uint8_t Data);
void I2C_DMACmd(I2C_TypeDef *I2Cx, uint32_t I2C_DMAReq, FunctionalState NewState);
void I2C_TxDMAITthreshold(uint32_t TxDMAValue);
void I2C_RxDMAITthreshold(uint32_t RxDMAValue);
FlagStatus I2C_GetFlagStatus(I2C_TypeDef *I2Cx, uint32_t I2C_FLAG);
FlagStatus I2C_GetITStatus(I2C_TypeDef *I2Cx, uint32_t I2C_STATUS);
void I2C_ClearITPendingBit(I2C_TypeDef *I2Cx, uint32_t I2C_IT);
void I2C_ITConfig(I2C_TypeDef *I2Cx, uint32_t I2C_IT, FunctionalState NewState);

/*software IIC fuction*/
void IIC_Init(void);
uint8_t AT24CXX_ReadOneByte(uint16_t ReadAddr);                                   //ַָȡһֽ /Read one byte from the specified address
void AT24CXX_WriteOneByte(uint16_t WriteAddr, uint8_t DataToWrite);               //ַָдһֽ /Write one byte to the address specified
void AT24CXX_WriteLenByte(uint16_t WriteAddr, uint32_t DataToWrite, uint8_t Len); //ַָʼдָȵ /The specified address starts writing data of the specified length
uint32_t AT24CXX_ReadLenByte(uint16_t ReadAddr, uint8_t Len);                     //ַָʼȡָ /The specified address starts to read data of the specified length
void AT24CXX_Write(uint16_t WriteAddr, uint8_t *pBuffer, uint16_t NumToWrite);    //ַָʼдָȵ /Write the specified length of data from the specified address
void AT24CXX_Read(uint16_t ReadAddr, uint8_t *pBuffer, uint16_t NumToRead);       //ַָʼָȵ /Read the specified length of data from the specified address

#ifdef __cplusplus
}
#endif

#endif // __LCM32F06X_I2C_H__
