/**
  ******************************************************************************
  * @file    lcm32f06x_flash.h 
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   Header for flash.c module
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __LCM32F06X_FLASH_H__
#define __LCM32F06X_FLASH_H__

/* Includes ------------------------------------------------------------------*/
#include "lcm32f06x.h"

#ifdef __cplusplus
  extern "C" {
#endif


#define __REG(addr) (*(volatile uint32_t *)(addr))
#define __REGH(addr) (*(volatile uint16_t *)(addr))
#define __REGB(addr) (*(volatile uint8_t *)(addr))
#define WR_REG(addr, data) (__REG((addr)) = (data))
#define WR_REGH(addr, data) (__REGH((addr)) = (data))
#define WR_REGB(addr, data) (__REGB((addr)) = (data))
#define RD_REG(addr) (__REG((addr)))
#define RD_REGH(addr) (__REGH((addr)))
#define RD_REGB(addr) (__REGB((addr)))

/** 
  * @brief  FLASH Status
  */
typedef enum
{
  FLASH_BUSY = 1,
  FLASH_ERROR_WRP,
  FLASH_ERROR_PROGRAM,
  FLASH_COMPLETE,
  FLASH_TIMEOUT
} FLASH_Status;

/* @defgroup FLASH_Timeout_definition */
#define FLASH_ER_PRG_TIMEOUT              ((uint32_t)0x000B0000)
/* @end */

#define FLASH_FKEY1                       ((uint32_t)0x45670123) /*!< Flash program erase key1 */
#define FLASH_FKEY2                       ((uint32_t)0xCDEF89AB) /*!< Flash program erase key2: used with FLASH_PEKEY1 
                                                                      to unlock the write access to the FPEC. */
#define FLASH_SMKEYR1                     ((uint32_t)0x4567FEDC)
#define FLASH_SMKEYR2                     ((uint32_t)0xBA980123)

#define FLASH_OPTKEY1                     ((uint32_t)0x45670123)
#define FLASH_OPTKEY2                     ((uint32_t)0xCDEF89AB)



/** @defgroup FLASH_Timeout_definition 
  * @{
  */
#define FLASH_ER_PRG_TIMEOUT              ((uint32_t)0x000B0000)

/** @defgroup FLASH_Address 
  * @{
  */
#define IS_FLASH_PROGRAM_ADDRESS(ADDRESS) (((ADDRESS) >= 0x08000000) && ((ADDRESS) <= 0x0800FFFF)) /*main flash����*/


#define RDPRT2                            ((uint32_t)0x00000100)
#define RDPRT                             ((uint32_t)0x000000FF)

/* @defgroup FLASH_Latency */
#define FLASH_Latency_0                   ((uint32_t)0x00000000) /*@EN:FLASH Zero Latency cycle @CN:FLASH���ӳ�����*/
#define FLASH_Latency_1                   ((uint32_t)0x00000011) /*@EN:FLASH 1 Latency cycle @CN:FLASH 1�� �ӳ�����*/
#define FLASH_Latency_2                   ((uint32_t)0x00000012) /*@EN:FLASH 2 Latency cycle @CN:FLASH 2�� ���ӳ�����*/
#define FLASH_Latency_3                   ((uint32_t)0x00000013) /*@EN:FLASH 3 Latency cycle @CN:FLASH 3�� ���ӳ�����*/
#define FLASH_Latency_4                   ((uint32_t)0x00000014) /*@EN:FLASH 4 Latency cycle @CN:FLASH 4�� ���ӳ�����*/
#define IS_FLASH_LATENCY(LATENCY)         (((LATENCY) == FLASH_Latency_0) || \
                                           ((LATENCY) == FLASH_Latency_1) || \
                                           ((LATENCY) == FLASH_Latency_2)|| \
                                           ((LATENCY) == FLASH_Latency_3)|| \
                                           ((LATENCY) == FLASH_Latency_4))
/* @end */

/*sectorд����*/
/* @defgroup OB_WRP */
#define OB_WRP_Pages0to3                  ((uint32_t)0x00000001) /*@EN: Write protection of page 0 to 3 @CN: ��0ҳ����3ҳ��д����*/
#define OB_WRP_Pages4to7                  ((uint32_t)0x00000002) /*@EN: Write protection of page 4 to 7 @CN: ��4ҳ����7ҳ��д����*/
#define OB_WRP_Pages8to11                 ((uint32_t)0x00000004) /*@EN: Write protection of page 8 to 11  @CN: ��8ҳ����11ҳ��д����*/
#define OB_WRP_Pages12to15                ((uint32_t)0x00000008) /*@EN: Write protection of page 12 to 15 @CN: ��12ҳ����15ҳ��д����*/
#define OB_WRP_Pages16to19                ((uint32_t)0x00000010) /*@EN: Write protection of page 16 to 19 @CN: ��16ҳ����19ҳ��д����*/
#define OB_WRP_Pages20to23                ((uint32_t)0x00000020) /*@EN: Write protection of page 20 to 23 @CN: ��20ҳ����23ҳ��д����*/
#define OB_WRP_Pages24to27                ((uint32_t)0x00000040) /*@EN: Write protection of page 24 to 27 @CN: ��24ҳ����27ҳ��д����*/
#define OB_WRP_Pages28to31                ((uint32_t)0x00000080) /*@EN: Write protection of page 28 to 31 @CN: ��28ҳ����31ҳ��д����*/
#define OB_WRP_Pages32to35                ((uint32_t)0x00000100) /*@EN: Write protection of page 32 to 35 @CN: ��32ҳ����35ҳ��д����*/
#define OB_WRP_Pages36to39                ((uint32_t)0x00000200) /*@EN: Write protection of page 36 to 39 @CN: ��36ҳ����39ҳ��д����*/
#define OB_WRP_Pages40to43                ((uint32_t)0x00000400) /*@EN: Write protection of page 40 to 43 @CN: ��40ҳ����43ҳ��д����*/
#define OB_WRP_Pages44to47                ((uint32_t)0x00000800) /*@EN: Write protection of page 44 to 47 @CN: ��44ҳ����47ҳ��д����*/
#define OB_WRP_Pages48to51                ((uint32_t)0x00001000) /*@EN: Write protection of page 48 to 51 @CN: ��48ҳ����51ҳ��д����*/
#define OB_WRP_Pages52to55                ((uint32_t)0x00002000) /*@EN: Write protection of page 52 to 55 @CN: ��52ҳ����55ҳ��д����*/
#define OB_WRP_Pages56to59                ((uint32_t)0x00004000) /*@EN: Write protection of page 56 to 59 @CN: ��56ҳ����59ҳ��д����*/
#define OB_WRP_Pages60to63                ((uint32_t)0x00008000) /*@EN: Write protection of page 60 to 63 @CN: ��60ҳ����63ҳ��д����*/
#define OB_WRP_AllPages                   ((uint32_t)0x0000FFFF) /*@EN: Write protection of all Sectors @CN: ����������д����*/
#define IS_OB_WRP(PAGE)                   (((PAGE) != 0x0000000))
/* @end */

/* @defgroup OB_RDP */
#define OB_RDP_Level_0                    ((uint8_t)0xAA) /*@EN: No protection @CN: ������*/
#define OB_RDP_Level_1                    ((uint8_t)0xBB) /*@EN: Read protection of the memory @CN: �洢���Ķ�ȡ����*/
#define IS_OB_RDP(LEVEL)                  (((LEVEL) == OB_RDP_Level_0) ||    \
                                           ((LEVEL) == OB_RDP_Level_1))
/* @end */

/* @defgroup IWDG_SEL */
#define OB_IWDG_HARD                      ((uint_8)0xF7) /*@EN: Hardware configuration watchdog @CN: Ӳ�����ÿ��Ź� */
#define OB_IWDG_SOFT                      ((uint_8)0xEF) /*@EN: Software configuration watchdog @CN: �������ÿ��Ź� */
#define IS_OB_IWDG(CFG)                   ((CFG == OB_IWDG_HARD) || (CFG == IWDG_SOFT))
/* @end */

/* @defgroup OB_BOOT_SEL */
#define OB_BOOT0_HARD                     ((uint_8)0xFB) /*@EN: BOOT0 option byte determination @CN: BOOT0ѡ���ֽھ��� */
#define OB_BOOT0_SOFT                     ((uint_8)0xFF) /*@EN: BOOT0 Pin decision @CN: BOOT0�ܽž��� */
#define IS_OB_BOOT_SEL(sources)           ((sources == OB_BOOT_HARD) || (sources == OB_BOOT_SOFT))
/* @end */

/* @defgroup OB_BOOT1 */
#define OB_BOOT1_L                        ((uint_8)0xFD) /*@EN: BOOT1 low @CN: BOOT1 �͵�ƽ */
#define OB_BOOT1_H                        ((uint_8)0xFF) /*@EN: BOOT1 high @CN: BOOT1 �ߵ�ƽ */
#define IS_OB_BOOT1(BOOT1)                ((BOOT1 == OB_BOOT1_L) || (BOOT1 == OB_BOOT1_H))
/* @end */

/* @defgroup OB_BOOT0 */
#define OB_BOOT0_L                        ((uint_8)0xFE) /*@EN: BOOT0 low @CN: BOOT0 �͵�ƽ */
#define OB_BOOT0_H                        ((uint_8)0xFF) /*@EN: BOOT0 high @CN: BOOT0 �ߵ�ƽ */
#define IS_OB_BOOT0(BOOT0)                ((BOOT0 == OB_BOOT0_L) || (BOOT0 == OB_BOOT0_H))
/* @end */

/* @defgroup DATAx */
#define OB_USER_DATA0                     ((uint8_t)0x00) /*@EN: user data 0 @CN: �û�������0 */
#define OB_USER_DATA1                     ((uint8_t)0x01) /*@EN: user data 1 @CN: �û�������1 */
#define OB_USER_DATA2                     ((uint8_t)0x02) /*@EN: user data 2 @CN: �û�������2 */
#define IS_OB_USER_DATA(DATAx)            ((DATAx == OB_USER_DATA0) || \
                                           (DATAx == OB_USER_DATA1) || \
                                           (DATAx == OB_USER_DATA2))
/* @end */

/* @defgroup FLASH_IT */
#define FLASH_IT_EOPIE                    ((uint32_t)0x00001000) /*@EN: FLASH end of programming Interrupt @CN: FLASH��̽����ж� */
#define FLASH_IT_ERRIE                    ((uint32_t)0x00000400) /*@EN: FLASH Error Interrupt @CN:  FLASH�����ж� */
#define FLASH_IT_OPTWRE                   ((uint32_t)0x00000200) /*@EN: FLASH option byte write enable @CN: FLASHѡ���ֽ�дʹ�� */
#define IS_FLASH_IT(IT)                   ((IT == FLASH_IT_EOPIE) || \
                                           (IT == FLASH_IT_ERRIE) || \
                                           (IT == FLASH_IT_OPTWRE))
/* @end */

/* @defgroup FLASH_IT */
#define FLASH_FLAG_BSY                    FLASH_SR_BSY_Msk      /*@EN: FLASH Busy flag @CN: FLASHæ��־ */
#define FLASH_FLAG_PGERR                  FLASH_SR_PGERR_Msk    /*@EN: FLASH Programming error flag @CN: FLASH��̴����־ */
#define FLASH_FLAG_WRPERR                 FLASH_SR_WRPRTERR_Msk /*@EN: FLASH Write protected error flag @CN: FLASHд���������־ */
#define FLASH_FLAG_EOP                    FLASH_SR_EOP_Msk      /*@EN: FLASH End of Programming flag @CN: FLASH��̽�����־ */
#define IS_FLASH_CLEAR_FLAG(FLAG)         ((((FLAG) & (uint32_t)0xFFFFFFC3) == 0x00000000) && ((FLAG) != 0x00000000))
#define IS_FLASH_GET_FLAG(FLASH_FLAG)     ((FLASH_FLAG == FLASH_FLAG_BSY)    || \
                                           (FLASH_FLAG == FLASH_FLAG_PGERR   || \
                                           (FLASH_FLAG == FLASH_FLAG_WRPERR) || \
                                           (FLASH_FLAG == FLASH_FLAG_EOP)))
/* @end */



/* ========================================================================== */
/* ================            FLASH REGISTER BIT               ============= */
/* ========================================================================== */

/* ================================== ACR  ================================== */
#define FLASH_ACR_R_LATENCY_Pos                   (8UL)                        /*!< R_LATENCY (Bit 8)                    */
#define FLASH_ACR_R_LATENCY_Msk                   (0x00000100UL)               /*!< TOSEL (Bitfield-Mask: 0x01)          */
#define FLASH_ACR_PRFTBS_Pos                      (5UL)                        /*!< PRFTBS (Bit 5)                       */
#define FLASH_ACR_PRFTBS_Msk                      (0x00000020UL)               /*!< PRFTBS (Bitfield-Mask: 0x01)         */
#define FLASH_ACR_PRFTBE_Pos                      (4UL)                        /*!< PRFTBE (Bit 4)                       */
#define FLASH_ACR_PRFTBE_Msk                      (0x00000010UL)               /*!< PRFTBE (Bitfield-Mask: 0x01)         */
#define FLASH_ACR_LATENCY_Pos                     (0UL)                        /*!< LATENCY (Bit 0)                      */
#define FLASH_ACR_LATENCY_Msk                     (0x00000007UL)               /*!< LATENCY (Bitfield-Mask: 0x07)        */
/* ================================  KEYR  ================================== */
#define FLASH_KEYR_FKEY_Pos                       (0UL)                        /*!< FKEY (Bit 0)                         */
#define FLASH_KEYR_FKEY_Msk                       (0xFFFFFFFFUL)               /*!< FKEY (Bitfield-Mask: 0xFFFFFFFF)     */
/* ===============================  OPTKEYR  ================================ */
#define FLASH_OPTKEYR_OPTKEY_Pos                  (0UL)                        /*!< OPTKEY (Bit 0)                       */
#define FLASH_OPTKEYR_OPTKEY_Msk                  (0xFFFFFFFFUL)               /*!< OPTKEY (Bitfield-Mask: 0xFFFFFFFF)   */
/* =================================  SR  =================================== */
#define FLASH_SR_OBERR_Pos                        (20UL)                       /*!< OBERR (Bit 20)                       */
#define FLASH_SR_OBERR_Msk                        (0x00100000UL)               /*!< OBERR (Bitfield-Mask: 0x01)          */
#define FLASH_SR_OB_EMPTY_FLAG_Pos                (19UL)                       /*!< OB_EMPTY_FLAG (Bit 19)               */
#define FLASH_SR_OB_EMPTY_FLAG_Msk                (0x00080000UL)               /*!< OB_EMPTY_FLAG (Bitfield-Mask: 0x01)  */
#define FLASH_SR_OB_BOOT0_PIN_Pos                 (18UL)                       /*!< OB_BOOT0_PIN (Bit 18)                */
#define FLASH_SR_OB_BOOT0_PIN_Msk                 (0x00040000UL)               /*!< OB_BOOT0_PIN (Bitfield-Mask: 0x01)   */
#define FLASH_SR_OB_BOOT_MODE_Pos                 (16UL)                       /*!< OB_BOOT_MODE (Bit 16)                */
#define FLASH_SR_OB_BOOT_MODE_Msk                 (0x00030000UL)               /*!< OB_BOOT_MODE (Bitfield-Mask: 0x03)   */
#define FLASH_SR_PG_TIMEOUT_Pos                   (9UL)                        /*!< PG_TIMEOUT (Bit 9)                   */
#define FLASH_SR_PG_TIMEOUT_Msk                   (0x00000200UL)               /*!< PG_TIMEOUT (Bitfield-Mask: 0x01)     */
#define FLASH_SR_PG_OVRANGE_Pos                   (8UL)                        /*!< PG_OVRANGE (Bit 8)                   */
#define FLASH_SR_PG_OVRANGE_Msk                   (0x00000100UL)               /*!< PG_OVRANGE (Bitfield-Mask: 0x01)     */
#define FLASH_SR_CONEOP_Pos                       (6UL)                        /*!< CONEOP (Bit 6)                       */
#define FLASH_SR_CONEOP_Msk                       (0x00000040UL)               /*!< CONEOP (Bitfield-Mask: 0x01)         */
#define FLASH_SR_EOP_Pos                          (5UL)                        /*!< EOP (Bit 5)                          */
#define FLASH_SR_EOP_Msk                          (0x00000020UL)               /*!< EOP (Bitfield-Mask: 0x01)            */
#define FLASH_SR_WRPRTERR_Pos                     (4UL)                        /*!< WRPRTERR (Bit 4)                     */
#define FLASH_SR_WRPRTERR_Msk                     (0x00000010UL)               /*!< WRPRTERR (Bitfield-Mask: 0x01)       */
#define FLASH_SR_PGERR_Pos                        (2UL)                        /*!< PGERR (Bit 2)                        */
#define FLASH_SR_PGERR_Msk                        (0x00000004UL)               /*!< PGERR (Bitfield-Mask: 0x01)          */
#define FLASH_SR_BSY_Pos                          (0UL)                        /*!< BSY (Bit 0)                          */
#define FLASH_SR_BSY_Msk                          (0x00000001UL)               /*!< BSY (Bitfield-Mask: 0x01)            */
/* =================================  CR  =================================== */
#define FLASH_CR_RDONLY_Pos                       (28UL)                       /*!< RDONLY (Bit 28)                      */
#define FLASH_CR_RDONLY_Msk                       (0x10000000UL)               /*!< RDONLY (Bitfield-Mask: 0x01)         */
#define FLASH_CR_BOOT_MODE_Pos                    (24UL)                       /*!< BOOT_MODE (Bit 24)                   */
#define FLASH_CR_BOOT_MODE_Msk                    (0x03000000UL)               /*!< BOOT_MODE (Bitfield-Mask: 0x03)      */
#define FLASH_CR_PG_STOP_Pos                      (21UL)                       /*!< PG_STOP (Bit 21)                     */
#define FLASH_CR_PG_STOP_Msk                      (0x00200000UL)               /*!< PG_STOP (Bitfield-Mask: 0x01)        */
#define FLASH_CR_CONPG_Pos                        (20UL)                       /*!< CONPG (Bit 20)                       */
#define FLASH_CR_CONPG_Msk                        (0x00100000UL)               /*!< CONPG (Bitfield-Mask: 0x01)          */
#define FLASH_CR_SMLOCK_Pos                       (16UL)                       /*!< SMLOCK (Bit 16)                      */
#define FLASH_CR_SMLOCK_Msk                       (0x00010000UL)               /*!< SMLOCK (Bitfield-Mask: 0x01)         */
#define FLASH_CR_EOPIE_Pos                        (12UL)                       /*!< EOPIE (Bit 12)                       */
#define FLASH_CR_EOPIE_Msk                        (0x00001000UL)               /*!< EOPIE (Bitfield-Mask: 0x01)          */
#define FLASH_CR_ERRIE_Pos                        (10UL)                       /*!< ERRIE (Bit 10)                       */
#define FLASH_CR_ERRIE_Msk                        (0x00000400UL)               /*!< ERRIE (Bitfield-Mask: 0x01)          */
#define FLASH_CR_OPTWRE_Pos                       (9UL)                        /*!< OPTWRE (Bit 9)                       */
#define FLASH_CR_OPTWRE_Msk                       (0x00000200UL)               /*!< OPTWRE (Bitfield-Mask: 0x01)         */
#define FLASH_CR_CONPG_LOCK_Pos                   (8UL)                        /*!< CONPG_LOCK (Bit 8)                   */
#define FLASH_CR_CONPG_LOCK_Msk                   (0x00000100UL)               /*!< CONPG_LOCK (Bitfield-Mask: 0x01)     */
#define FLASH_CR_LOCK_Pos                         (7UL)                        /*!< LOCK (Bit 7)                         */
#define FLASH_CR_LOCK_Msk                         (0x00000080UL)               /*!< LOCK (Bitfield-Mask: 0x01)           */
#define FLASH_CR_STRT_Pos                         (6UL)                        /*!< STRT (Bit 6)                         */
#define FLASH_CR_STRT_Msk                         (0x00000040UL)               /*!< STRT (Bitfield-Mask: 0x01)           */
#define FLASH_CR_OPTER_Pos                        (5UL)                        /*!< OPTER (Bit 5)                        */
#define FLASH_CR_OPTER_Msk                        (0x00000020UL)               /*!< OPTER (Bitfield-Mask: 0x01)          */
#define FLASH_CR_OPTPG_Pos                        (4UL)                        /*!< OPTPG (Bit 4)                        */
#define FLASH_CR_OPTPG_Msk                        (0x00000010UL)               /*!< OPTPG (Bitfield-Mask: 0x01)          */
#define FLASH_CR_HALFER_Pos                       (3UL)                        /*!< HALFER (Bit 3)                       */
#define FLASH_CR_HALFER_Msk                       (0x00000008UL)               /*!< HALFER (Bitfield-Mask: 0x01)         */
#define FLASH_CR_MER_Pos                          (2UL)                        /*!< MER (Bit 2)                          */
#define FLASH_CR_MER_Msk                          (0x00000004UL)               /*!< MER (Bitfield-Mask: 0x01)            */
#define FLASH_CR_PER_Pos                          (1UL)                        /*!< PER (Bit 1)                          */
#define FLASH_CR_PER_Msk                          (0x00000002UL)               /*!< PER (Bitfield-Mask: 0x01)            */
#define FLASH_CR_PG_Pos                           (0UL)                        /*!< PG (Bit 0)                           */
#define FLASH_CR_PG_Msk                           (0x00000001UL)               /*!< PG (Bitfield-Mask: 0x01)             */
/* ==================================  AR  ================================== */
#define FLASH_AR_FAR_Pos                          (0UL)                        /*!< FAR (Bit 0)                          */
#define FLASH_AR_FAR_Msk                          (0xFFFFFFFFUL)               /*!< FAR (Bitfield-Mask: 0xFFFFFFFF)      */
/* ==================================  OBR  ================================= */
#define FLASH_OBR_DATA2_Pos                       (24UL)                       /*!< DATA2 (Bit 24)                       */
#define FLASH_OBR_DATA2_Msk                       (0xFF000000UL)               /*!< DATA2 (Bitfield-Mask: 0xFF)          */
#define FLASH_OBR_DATA1_Pos                       (16UL)                       /*!< DATA1 (Bit 16)                       */
#define FLASH_OBR_DATA1_Msk                       (0x00FF0000UL)               /*!< DATA1 (Bitfield-Mask: 0xFF)          */
#define FLASH_OBR_DATA0_Pos                       (8UL)                        /*!< DATA0 (Bit 8)                        */
#define FLASH_OBR_DATA0_Msk                       (0x0000FF00UL)               /*!< DATA0 (Bitfield-Mask: 0xFF)          */
#define FLASH_OBR_WDG_SW_Pos                      (3UL)                        /*!< WDG_SW (Bit 3)                       */
#define FLASH_OBR_WDG_SW_Msk                      (0x00000008UL)               /*!< WDG_SW (Bitfield-Mask: 0x01)         */
#define FLASH_OBR_BOOT_SEL_Pos                    (2UL)                        /*!< BOOT_SEL (Bit 2)                     */
#define FLASH_OBR_BOOT_SEL_Msk                    (0x00000004UL)               /*!< BOOT_SEL (Bitfield-Mask: 0x01)       */
#define FLASH_OBR_nBOOT1_Pos                      (1UL)                        /*!< nBOOT1 (Bit 1)                       */
#define FLASH_OBR_nBOOT1_Msk                      (0x00000002UL)               /*!< nBOOT1 (Bitfield-Mask: 0x01)         */
#define FLASH_OBR_nBOOT0_Pos                      (0UL)                        /*!< nBOOT0 (Bit 0)                       */
#define FLASH_OBR_nBOOT0_Msk                      (0x00000001UL)               /*!< nBOOT0 (Bitfield-Mask: 0x01)         */
/* =================================  CAL1  ================================= */
#define FLASH_CAL1_CAL1_Pos                       (0UL)                        /*!< CAL1 (Bit 0)                         */
#define FLASH_CAL1_CAL1_Msk                       (0xFFFFFFFFUL)               /*!< CAL1 (Bitfield-Mask: 0xFFFFFFFF)     */
/* =================================  RDPR  ================================= */
#define FLASH_RDPR_RDPRT2_Pos                     (8UL)                        /*!< RDPRT2 (Bit 8)                       */
#define FLASH_RDPR_RDPRT2_Msk                     (0x00000100UL)               /*!< RDPRT2 (Bitfield-Mask: 0x01)         */
#define FLASH_RDPR_RDPRT_Pos                      (0UL)                        /*!< RDPRT (Bit 0)                        */
#define FLASH_RDPR_RDPRT_Msk                      (0x000000FFUL)               /*!< RDPRT (Bitfield-Mask: 0xFF)          */
/* =================================  WRPR  ================================= */
#define FLASH_WRPR_WRP_Pos                        (0UL)                        /*!< WRP (Bit 0)                          */
#define FLASH_WRPR_WRP_Msk                        (0xFFFFFFFFUL)               /*!< WRP (Bitfield-Mask: 0xFFFFFFFF)      */
/* =================================  CAL2  ================================= */
#define FLASH_CAL2_CAL2_Pos                       (0UL)                        /*!< CAL2 (Bit 0)                         */
#define FLASH_CAL2_CAL2_Msk                       (0xFFFFFFFFUL)               /*!< CAL2 (Bitfield-Mask: 0xFFFFFFFF)     */
/* ================================  SMKEYR  ================================ */
#define FLASH_SMKEYR_SMKEY_Pos                    (0UL)                        /*!< SMKEY (Bit 0)                        */
#define FLASH_SMKEYR_SMKEY_Msk                    (0xFFFFFFFFUL)               /*!< SMKEY (Bitfield-Mask: 0xFFFFFFFF)    */


/******************  Bit definition for OB_WRP0 register  *********************/
#define OB_WRP0_WRP0 ((uint32_t)0x000000FF)  /*!< Flash memory write protection option bytes */
#define OB_WRP0_nWRP0 ((uint32_t)0x0000FF00) /*!< Flash memory write protection complemented option bytes */

/******************  Bit definition for OB_WRP1 register  *********************/
#define OB_WRP1_WRP1 ((uint32_t)0x00FF0000)  /*!< Flash memory write protection option bytes */
#define OB_WRP1_nWRP1 ((uint32_t)0xFF000000) /*!< Flash memory write protection complemented option bytes */

/******************  Bit definition for OB_WRP0 register  *********************/
#define OB_WRP0_WRP2 ((uint32_t)0x000000FF)  /*!< Flash memory write protection option bytes */
#define OB_WRP0_nWRP2 ((uint32_t)0x0000FF00) /*!< Flash memory write protection complemented option bytes */

/******************  Bit definition for OB_WRP1 register  *********************/
#define OB_WRP1_WRP3 ((uint32_t)0x00FF0000)  /*!< Flash memory write protection option bytes */
#define OB_WRP1_nWRP3 ((uint32_t)0xFF000000) /*!< Flash memory write protection complemented option bytes */

/* ========================================================================== */
/* ================                  END                      =============== */
/* ========================================================================== */



FLASH_Status FLASH_WaitForLastOperation(uint32_t Timeout);
FLASH_Status FLASH_GetStatus(void);
FlagStatus FLASH_OB_GetRDP(void);
void FLASH_SetLatency(uint32_t FLASH_Latency);
FlagStatus FLASH_GetPrefetchBufferStatus(void);
void FLASH_PrefetchBufferCmd(FunctionalState NewState);
void FLASH_Unlock(void);
void FLASH_Lock(void);
void FLASH_UnOnlyReadMode(void);
void FLASH_OnlyReadMode(void);
FLASH_Status FLASH_ErasePage(uint32_t Page_Address);
FLASH_Status FLASH_EraseAllPages(void);
FLASH_Status FLASH_ProgramWord(uint32_t Address, uint32_t Data);
FLASH_Status FLASH_ProgramHalfWord(uint32_t Address, uint16_t Data);
void FLASH_OB_Unlock(void);
void FLASH_OB_Lock(void);
FLASH_Status FLASH_DM_ErasePage(uint32_t Page_Address);
FLASH_Status FLASH_DM_ProgramWord(uint32_t Address, uint32_t Data);
FLASH_Status FLASH_DM_ProgramHalfWord(uint32_t Address, uint16_t Data);
FLASH_Status FLASH_OB_Erase(void);
FLASH_Status FLASH_OB_EnableWRP(uint32_t OB_WRP);
FLASH_Status FLASH_OB_RDPConfig(uint8_t OB_RDP);
FLASH_Status FLASH_OB_USERConfig(uint8_t IWDG_SEL, uint8_t OB_BOOT_SEL, uint8_t OB_BOOT1, uint8_t OB_BOOT0);
FLASH_Status FLASH_OB_ProgramData(uint8_t DATAx, uint8_t OB_DATA);
void FLASH_ITConfig(uint32_t FLASH_IT, FunctionalState NewState);
FlagStatus FLASH_GetFlagStatus(uint32_t FLASH_FLAG);
uint8_t FLASH_OB_GetUser(void);
uint32_t FLASH_OB_GetWRP(void);
void FLASH_ClearFlag(uint32_t FLASH_FLAG);
void ERASE_OPT(void);
void write_opt_flash_n(unsigned int addr, unsigned int data);
void ERASE_DATA_MERMORY(void);
void write_sm_flash_n(unsigned int addr, unsigned int data);
void ERASE_Calibration_MERMORY(void);
void write_DataMemory_n(unsigned int addr, unsigned int data);
void rcl_trim(uint32_t trim);
void rch_trim(uint32_t trim);
void bgr_trim(uint32_t trim);
void adc_trim(uint32_t trim);
void mco_set(uint32_t sel, uint32_t div);
void lpbgr_trim(uint32_t trim);
#ifdef __cplusplus
}
#endif

#endif // __LCM32F06X_FLASH_H__
