/**
  ******************************************************************************
  * @file    lcm32f06x_exti.h 
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   Header for exti.c module
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __LCM32F06X_EXTI_H__
#define __LCM32F06X_EXTI_H__

/* Includes ------------------------------------------------------------------*/
#include "lcm32f06x.h"

#ifdef __cplusplus
  extern "C" {
#endif

/**
 * @brief  NVIC Init Structure definition
 */

typedef struct
{
  uint8_t NVIC_IRQChannel;            /*!< Specifies the IRQ channel to be enabled or disabled.
                                           This parameter can be a value of @ref IRQn_Type 
                                           (For the complete LCM32 Devices IRQ Channels list, 
                                            note: SVCall,PendSv,SysTick Configure using kernel interface) */

  uint8_t NVIC_IRQChannelPriority;    /*!< Specifies the priority level for the IRQ channel specified
                                           in NVIC_IRQChannel. This parameter can be a value
                                           between 0 and 3.  */

  FunctionalState NVIC_IRQChannelCmd; /*!< Specifies whether the IRQ channel defined in NVIC_IRQChannel
                                           will be enabled or disabled.
                                           This parameter can be set either to ENABLE or DISABLE */
} NVIC_InitTypeDef;

/**
 * @}
 */

  typedef enum
  {
    EXTI_Mode_Interrupt = 0x00,               /*@EN: External interrupt mode interrupt @CN:ⲿжģʽж*/
    EXTI_Mode_Event = 0x04                    /*@EN: External interrupt mode event @CN:ⲿжģʽ¼*/
  } EXTIMode_TypeDef;

#define IS_EXTI_MODE(MODE) (((MODE) == EXTI_Mode_Interrupt) || ((MODE) == EXTI_Mode_Event))

  typedef enum
  {
    EXTI_Trigger_Rising = 0x08,               /*@EN: External interrupt triggers rising edge @CN:ⲿжϴ*/
    EXTI_Trigger_Falling = 0x0C,              /*@EN: External interrupt triggers falling edge @CN:ⲿжϴ½*/
    EXTI_Trigger_Rising_Falling = 0x10        /*@EN: External interrupt triggers bilateral edges @CN:ⲿжϴ˫*/
  } EXTITrigger_TypeDef;

#define IS_EXTI_TRIGGER(TRIGGER) (((TRIGGER) == EXTI_Trigger_Rising) ||  \
                                  ((TRIGGER) == EXTI_Trigger_Falling) || \
                                  ((TRIGGER) == EXTI_Trigger_Rising_Falling))

typedef struct
{
  uint32_t EXTI_Line;               /*!< EN: Specifies the EXTI lines to be enabled or disabled.
                                         CN: ָҪûõEXTIС
                                         @default EXTI_Line0 */

  EXTIMode_TypeDef EXTI_Mode;       /*!< EN: Specifies the mode for the EXTI lines.
                                         CN: ָEXTIߵģʽ
                                         @default EXTI_Mode_Interrupt */

  EXTITrigger_TypeDef EXTI_Trigger; /*!< EN: Specifies the trigger signal active edge for the EXTI lines.
                                         CN: ָEXTIߵĴźŴء
                                         @Register: EXTI_RTSR EXTI_FTSR @Bit:[23:0]RTx FTx
                                         @default EXTI_Trigger_Rising */

  FunctionalState EXTI_LineCmd;     /*!< EN: Specifies the new state of the selected EXTI lines.
                                         CN: ָѡEXTIߵ״̬
                                         @Register: EXTI_IMR EXTI_EMR @Bit:[23:0]IMx EMx
                                         @default DISABLE */
} EXTI_InitTypeDef;


#define EXTI_Line0             ((uint32_t)0x00001)  /*@EN: External interrupt line 0 @CN:ⲿж·0*/
#define EXTI_Line1             ((uint32_t)0x00002)  /*@EN: External interrupt line 1 @CN:ⲿж·1*/
#define EXTI_Line2             ((uint32_t)0x00004)  /*@EN: External interrupt line 2 @CN:ⲿж·2*/
#define EXTI_Line3             ((uint32_t)0x00008)  /*@EN: External interrupt line 3 @CN:ⲿж·3*/
#define EXTI_Line4             ((uint32_t)0x00010)  /*@EN: External interrupt line 4 @CN:ⲿж·4*/
#define EXTI_Line5             ((uint32_t)0x00020)  /*@EN: External interrupt line 5 @CN:ⲿж·5*/
#define EXTI_Line6             ((uint32_t)0x00040)  /*@EN: External interrupt line 6 @CN:ⲿж·6*/
#define EXTI_Line7             ((uint32_t)0x00080)  /*@EN: External interrupt line 7 @CN:ⲿж·7*/
#define EXTI_Line8             ((uint32_t)0x00100)  /*@EN: External interrupt line 8 @CN:ⲿж·8*/
#define EXTI_Line9             ((uint32_t)0x00200)  /*@EN: External interrupt line 9 @CN:ⲿж·9*/
#define EXTI_Line10            ((uint32_t)0x00400)  /*@EN: External interrupt line 10 @CN:ⲿж·10*/
#define EXTI_Line11            ((uint32_t)0x00800)  /*@EN: External interrupt line 11 @CN:ⲿж·11*/
#define EXTI_Line12            ((uint32_t)0x01000)  /*@EN: External interrupt line 12 @CN:ⲿж·12*/
#define EXTI_Line13            ((uint32_t)0x02000)  /*@EN: External interrupt line 13 @CN:ⲿж·13*/
#define EXTI_Line14            ((uint32_t)0x04000)  /*@EN: External interrupt line 14 @CN:ⲿж·14*/
#define EXTI_Line15            ((uint32_t)0x08000)  /*@EN: External interrupt line 15 @CN:ⲿж·15*/
#define EXTI_Line16            ((uint32_t)0x10000)  /*@EN: External interrupt line 16 @CN:ⲿж·16*/
#define EXTI_Line17            ((uint32_t)0x20000)  /*@EN: External interrupt line 17 @CN:ⲿж·17*/
#define EXTI_Line18            ((uint32_t)0x40000)  /*@EN: External interrupt line 18 @CN:ⲿж·18*/
#define EXTI_Line19            ((uint32_t)0x80000)  /*@EN: External interrupt line 19 @CN:ⲿж·19*/
#define EXTI_Line20            ((uint32_t)0x100000) /*@EN: External interrupt line 20 @CN:ⲿж·20*/
#define EXTI_Line21            ((uint32_t)0x200000) /*@EN: External interrupt line 21 @CN:ⲿж·21*/
#define EXTI_Line22            ((uint32_t)0x400000) /*@EN: External interrupt line 22 @CN:ⲿж·22*/
#define EXTI_Line23            ((uint32_t)0x800000) /*@EN: External interrupt line 23 @CN:ⲿж·23*/

#define IS_EXTI_LINE(LINE)     ((((LINE) & (uint32_t)0xFFF00000) == 0x00) && ((LINE) != (uint16_t)0x00))
#define IS_GET_EXTI_LINE(LINE) (((LINE) == EXTI_Line0) || ((LINE) == EXTI_Line1) ||   \
                                ((LINE) == EXTI_Line2) || ((LINE) == EXTI_Line3) ||   \
                                ((LINE) == EXTI_Line4) || ((LINE) == EXTI_Line5) ||   \
                                ((LINE) == EXTI_Line6) || ((LINE) == EXTI_Line7) ||   \
                                ((LINE) == EXTI_Line8) || ((LINE) == EXTI_Line9) ||   \
                                ((LINE) == EXTI_Line10) || ((LINE) == EXTI_Line11) || \
                                ((LINE) == EXTI_Line12) || ((LINE) == EXTI_Line13) || \
                                ((LINE) == EXTI_Line14) || ((LINE) == EXTI_Line15) || \
                                ((LINE) == EXTI_Line16) || ((LINE) == EXTI_Line17) || \
                                ((LINE) == EXTI_Line18) || ((LINE) == EXTI_Line19) || \
                                ((LINE) == EXTI_Line20) || ((LINE) == EXTI_Line21) || \
                                ((LINE) == EXTI_Line22) || ((LINE) == EXTI_Line23))


/* ========================================================================== */
/* ================              EXTI REGISTER BIT              ============= */
/* ========================================================================== */

/* ==================================== IMR  ================================ */
#define EXTI_IMR_MR24_Pos                         (24UL)                       /*!< MR24 (Bit 24)                 */
#define EXTI_IMR_MR24_Msk                         (0x01000000UL)               /*!< MR24 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR23_Pos                         (23UL)                       /*!< MR23 (Bit 23)                 */
#define EXTI_IMR_MR23_Msk                         (0x00800000UL)               /*!< MR23 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR22_Pos                         (22UL)                       /*!< MR22 (Bit 22)                 */
#define EXTI_IMR_MR22_Msk                         (0x00400000UL)               /*!< MR22 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR21_Pos                         (21UL)                       /*!< MR21 (Bit 21)                 */
#define EXTI_IMR_MR21_Msk                         (0x00200000UL)               /*!< MR21 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR20_Pos                         (20UL)                       /*!< MR20 (Bit 20)                 */
#define EXTI_IMR_MR20_Msk                         (0x00100000UL)               /*!< MR20 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR19_Pos                         (19UL)                       /*!< MR19 (Bit 19)                 */
#define EXTI_IMR_MR19_Msk                         (0x00080000UL)               /*!< MR19 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR18_Pos                         (18UL)                       /*!< MR18 (Bit 18)                 */
#define EXTI_IMR_MR18_Msk                         (0x00040000UL)               /*!< MR18 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR17_Pos                         (17UL)                       /*!< MR17 (Bit 17)                 */
#define EXTI_IMR_MR17_Msk                         (0x00020000UL)               /*!< MR17 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR16_Pos                         (16UL)                       /*!< MR16 (Bit 16)                 */
#define EXTI_IMR_MR16_Msk                         (0x00010000UL)               /*!< MR16 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR15_Pos                         (15UL)                       /*!< MR15 (Bit 15)                 */
#define EXTI_IMR_MR15_Msk                         (0x00008000UL)               /*!< MR15 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR14_Pos                         (14UL)                       /*!< MR14 (Bit 14)                 */
#define EXTI_IMR_MR14_Msk                         (0x00004000UL)               /*!< MR14 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR13_Pos                         (13UL)                       /*!< MR13 (Bit 13)                 */
#define EXTI_IMR_MR13_Msk                         (0x00002000UL)               /*!< MR13 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR12_Pos                         (12UL)                       /*!< MR12 (Bit 12)                 */
#define EXTI_IMR_MR12_Msk                         (0x00001000UL)               /*!< MR12 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR11_Pos                         (11UL)                       /*!< MR11 (Bit 11)                 */
#define EXTI_IMR_MR11_Msk                         (0x00000800UL)               /*!< MR11 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR10_Pos                         (10UL)                       /*!< MR10 (Bit 10)                 */
#define EXTI_IMR_MR10_Msk                         (0x00000400UL)               /*!< MR10 (Bitfield-Mask: 0x01)    */
#define EXTI_IMR_MR9_Pos                          (9UL)                        /*!< MR9 (Bit 9)                   */
#define EXTI_IMR_MR9_Msk                          (0x00000200UL)               /*!< MR9 (Bitfield-Mask: 0x01)     */
#define EXTI_IMR_MR8_Pos                          (8UL)                        /*!< MR8 (Bit 8)                   */
#define EXTI_IMR_MR8_Msk                          (0x00000100UL)               /*!< MR8 (Bitfield-Mask: 0x01)     */
#define EXTI_IMR_MR7_Pos                          (7UL)                        /*!< MR7 (Bit 7)                   */
#define EXTI_IMR_MR7_Msk                          (0x00000080UL)               /*!< MR7 (Bitfield-Mask: 0x01)     */
#define EXTI_IMR_MR6_Pos                          (6UL)                        /*!< MR6 (Bit 6)                   */
#define EXTI_IMR_MR6_Msk                          (0x00000040UL)               /*!< MR6 (Bitfield-Mask: 0x01)     */
#define EXTI_IMR_MR5_Pos                          (5UL)                        /*!< MR5 (Bit 5)                   */
#define EXTI_IMR_MR5_Msk                          (0x00000020UL)               /*!< MR5 (Bitfield-Mask: 0x01)     */
#define EXTI_IMR_MR4_Pos                          (4UL)                        /*!< MR4 (Bit 4)                   */
#define EXTI_IMR_MR4_Msk                          (0x00000010UL)               /*!< MR4 (Bitfield-Mask: 0x01)     */
#define EXTI_IMR_MR3_Pos                          (3UL)                        /*!< MR3 (Bit 3)                   */
#define EXTI_IMR_MR3_Msk                          (0x00000008UL)               /*!< MR3 (Bitfield-Mask: 0x01)     */
#define EXTI_IMR_MR2_Pos                          (2UL)                        /*!< MR2 (Bit 2)                   */
#define EXTI_IMR_MR2_Msk                          (0x00000004UL)               /*!< MR2 (Bitfield-Mask: 0x01)     */
#define EXTI_IMR_MR1_Pos                          (1UL)                        /*!< MR1 (Bit 1)                   */
#define EXTI_IMR_MR1_Msk                          (0x00000002UL)               /*!< MR1 (Bitfield-Mask: 0x01)     */
#define EXTI_IMR_MR0_Pos                          (0UL)                        /*!< MR0 (Bit 0)                   */
#define EXTI_IMR_MR0_Msk                          (0x00000001UL)               /*!< MR0 (Bitfield-Mask: 0x01)     */
/* ===================================  EMR  ================================ */
#define EXTI_EMR_MR24_Pos                         (24UL)                       /*!< MR24 (Bit 24)                 */
#define EXTI_EMR_MR24_Msk                         (0x01000000UL)               /*!< MR24 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR23_Pos                         (23UL)                       /*!< MR23 (Bit 23)                 */
#define EXTI_EMR_MR23_Msk                         (0x00800000UL)               /*!< MR23 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR22_Pos                         (22UL)                       /*!< MR22 (Bit 22)                 */
#define EXTI_EMR_MR22_Msk                         (0x00400000UL)               /*!< MR22 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR21_Pos                         (21UL)                       /*!< MR21 (Bit 21)                 */
#define EXTI_EMR_MR21_Msk                         (0x00200000UL)               /*!< MR21 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR20_Pos                         (20UL)                       /*!< MR20 (Bit 20)                 */
#define EXTI_EMR_MR20_Msk                         (0x00100000UL)               /*!< MR20 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR19_Pos                         (19UL)                       /*!< MR19 (Bit 19)                 */
#define EXTI_EMR_MR19_Msk                         (0x00080000UL)               /*!< MR19 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR18_Pos                         (18UL)                       /*!< MR18 (Bit 18)                 */
#define EXTI_EMR_MR18_Msk                         (0x00040000UL)               /*!< MR18 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR17_Pos                         (17UL)                       /*!< MR17 (Bit 17)                 */
#define EXTI_EMR_MR17_Msk                         (0x00020000UL)               /*!< MR17 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR16_Pos                         (16UL)                       /*!< MR16 (Bit 16)                 */
#define EXTI_EMR_MR16_Msk                         (0x00010000UL)               /*!< MR16 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR15_Pos                         (15UL)                       /*!< MR15 (Bit 15)                 */
#define EXTI_EMR_MR15_Msk                         (0x00008000UL)               /*!< MR15 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR14_Pos                         (14UL)                       /*!< MR14 (Bit 14)                 */
#define EXTI_EMR_MR14_Msk                         (0x00004000UL)               /*!< MR14 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR13_Pos                         (13UL)                       /*!< MR13 (Bit 13)                 */
#define EXTI_EMR_MR13_Msk                         (0x00002000UL)               /*!< MR13 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR12_Pos                         (12UL)                       /*!< MR12 (Bit 12)                 */
#define EXTI_EMR_MR12_Msk                         (0x00001000UL)               /*!< MR12 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR11_Pos                         (11UL)                       /*!< MR11 (Bit 11)                 */
#define EXTI_EMR_MR11_Msk                         (0x00000800UL)               /*!< MR11 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR10_Pos                         (10UL)                       /*!< MR10 (Bit 10)                 */
#define EXTI_EMR_MR10_Msk                         (0x00000400UL)               /*!< MR10 (Bitfield-Mask: 0x01)    */
#define EXTI_EMR_MR9_Pos                          (9UL)                        /*!< MR9 (Bit 9)                   */
#define EXTI_EMR_MR9_Msk                          (0x00000200UL)               /*!< MR9 (Bitfield-Mask: 0x01)     */
#define EXTI_EMR_MR8_Pos                          (8UL)                        /*!< MR8 (Bit 8)                   */
#define EXTI_EMR_MR8_Msk                          (0x00000100UL)               /*!< MR8 (Bitfield-Mask: 0x01)     */
#define EXTI_EMR_MR7_Pos                          (7UL)                        /*!< MR7 (Bit 7)                   */
#define EXTI_EMR_MR7_Msk                          (0x00000080UL)               /*!< MR7 (Bitfield-Mask: 0x01)     */
#define EXTI_EMR_MR6_Pos                          (6UL)                        /*!< MR6 (Bit 6)                   */
#define EXTI_EMR_MR6_Msk                          (0x00000040UL)               /*!< MR6 (Bitfield-Mask: 0x01)     */
#define EXTI_EMR_MR5_Pos                          (5UL)                        /*!< MR5 (Bit 5)                   */
#define EXTI_EMR_MR5_Msk                          (0x00000020UL)               /*!< MR5 (Bitfield-Mask: 0x01)     */
#define EXTI_EMR_MR4_Pos                          (4UL)                        /*!< MR4 (Bit 4)                   */
#define EXTI_EMR_MR4_Msk                          (0x00000010UL)               /*!< MR4 (Bitfield-Mask: 0x01)     */
#define EXTI_EMR_MR3_Pos                          (3UL)                        /*!< MR3 (Bit 3)                   */
#define EXTI_EMR_MR3_Msk                          (0x00000008UL)               /*!< MR3 (Bitfield-Mask: 0x01)     */
#define EXTI_EMR_MR2_Pos                          (2UL)                        /*!< MR2 (Bit 2)                   */
#define EXTI_EMR_MR2_Msk                          (0x00000004UL)               /*!< MR2 (Bitfield-Mask: 0x01)     */
#define EXTI_EMR_MR1_Pos                          (1UL)                        /*!< MR1 (Bit 1)                   */
#define EXTI_EMR_MR1_Msk                          (0x00000002UL)               /*!< MR1 (Bitfield-Mask: 0x01)     */
#define EXTI_EMR_MR0_Pos                          (0UL)                        /*!< MR0 (Bit 0)                   */
#define EXTI_EMR_MR0_Msk                          (0x00000001UL)               /*!< MR0 (Bitfield-Mask: 0x01)     */
/* ===================================  RTSR  =============================== */
#define EXTI_RTSR_TR24_Pos                        (24UL)                       /*!< TR24 (Bit 24)                 */
#define EXTI_RTSR_TR24_Msk                        (0x01000000UL)               /*!< TR24 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR23_Pos                        (23UL)                       /*!< TR23 (Bit 23)                 */
#define EXTI_RTSR_TR23_Msk                        (0x00800000UL)               /*!< TR23 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR22_Pos                        (22UL)                       /*!< TR22 (Bit 22)                 */
#define EXTI_RTSR_TR22_Msk                        (0x00400000UL)               /*!< TR22 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR21_Pos                        (21UL)                       /*!< TR21 (Bit 21)                 */
#define EXTI_RTSR_TR21_Msk                        (0x00200000UL)               /*!< TR21 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR20_Pos                        (20UL)                       /*!< TR20 (Bit 20)                 */
#define EXTI_RTSR_TR20_Msk                        (0x00100000UL)               /*!< TR20 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR19_Pos                        (19UL)                       /*!< TR19 (Bit 19)                 */
#define EXTI_RTSR_TR19_Msk                        (0x00080000UL)               /*!< TR19 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR18_Pos                        (18UL)                       /*!< TR18 (Bit 18)                 */
#define EXTI_RTSR_TR18_Msk                        (0x00040000UL)               /*!< TR18 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR17_Pos                        (17UL)                       /*!< TR17 (Bit 17)                 */
#define EXTI_RTSR_TR17_Msk                        (0x00020000UL)               /*!< TR17 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR16_Pos                        (16UL)                       /*!< TR16 (Bit 16)                 */
#define EXTI_RTSR_TR16_Msk                        (0x00010000UL)               /*!< TR16 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR15_Pos                        (15UL)                       /*!< TR15 (Bit 15)                 */
#define EXTI_RTSR_TR15_Msk                        (0x00008000UL)               /*!< TR15 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR14_Pos                        (14UL)                       /*!< TR14 (Bit 14)                 */
#define EXTI_RTSR_TR14_Msk                        (0x00004000UL)               /*!< TR14 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR13_Pos                        (13UL)                       /*!< TR13 (Bit 13)                 */
#define EXTI_RTSR_TR13_Msk                        (0x00002000UL)               /*!< TR13 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR12_Pos                        (12UL)                       /*!< TR12 (Bit 12)                 */
#define EXTI_RTSR_TR12_Msk                        (0x00001000UL)               /*!< TR12 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR11_Pos                        (11UL)                       /*!< TR11 (Bit 11)                 */
#define EXTI_RTSR_TR11_Msk                        (0x00000800UL)               /*!< TR11 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR10_Pos                        (10UL)                       /*!< TR10 (Bit 10)                 */
#define EXTI_RTSR_TR10_Msk                        (0x00000400UL)               /*!< TR10 (Bitfield-Mask: 0x01)    */
#define EXTI_RTSR_TR9_Pos                         (9UL)                        /*!< TR9 (Bit 9)                   */
#define EXTI_RTSR_TR9_Msk                         (0x00000200UL)               /*!< TR9 (Bitfield-Mask: 0x01)     */
#define EXTI_RTSR_TR8_Pos                         (8UL)                        /*!< TR8 (Bit 8)                   */
#define EXTI_RTSR_TR8_Msk                         (0x00000100UL)               /*!< TR8 (Bitfield-Mask: 0x01)     */
#define EXTI_RTSR_TR7_Pos                         (7UL)                        /*!< TR7 (Bit 7)                   */
#define EXTI_RTSR_TR7_Msk                         (0x00000080UL)               /*!< TR7 (Bitfield-Mask: 0x01)     */
#define EXTI_RTSR_TR6_Pos                         (6UL)                        /*!< TR6 (Bit 6)                   */
#define EXTI_RTSR_TR6_Msk                         (0x00000040UL)               /*!< TR6 (Bitfield-Mask: 0x01)     */
#define EXTI_RTSR_TR5_Pos                         (5UL)                        /*!< TR5 (Bit 5)                   */
#define EXTI_RTSR_TR5_Msk                         (0x00000020UL)               /*!< TR5 (Bitfield-Mask: 0x01)     */
#define EXTI_RTSR_TR4_Pos                         (4UL)                        /*!< TR4 (Bit 4)                   */
#define EXTI_RTSR_TR4_Msk                         (0x00000010UL)               /*!< TR4 (Bitfield-Mask: 0x01)     */
#define EXTI_RTSR_TR3_Pos                         (3UL)                        /*!< TR3 (Bit 3)                   */
#define EXTI_RTSR_TR3_Msk                         (0x00000008UL)               /*!< TR3 (Bitfield-Mask: 0x01)     */
#define EXTI_RTSR_TR2_Pos                         (2UL)                        /*!< TR2 (Bit 2)                   */
#define EXTI_RTSR_TR2_Msk                         (0x00000004UL)               /*!< TR2 (Bitfield-Mask: 0x01)     */
#define EXTI_RTSR_TR1_Pos                         (1UL)                        /*!< TR1 (Bit 1)                   */
#define EXTI_RTSR_TR1_Msk                         (0x00000002UL)               /*!< TR1 (Bitfield-Mask: 0x01)     */
#define EXTI_RTSR_TR0_Pos                         (0UL)                        /*!< TR0 (Bit 0)                   */
#define EXTI_RTSR_TR0_Msk                         (0x00000001UL)               /*!< TR0 (Bitfield-Mask: 0x01)     */
/* ===================================  FTSR  =============================== */
#define EXTI_FTSR_TR24_Pos                        (24UL)                       /*!< TR24 (Bit 24)                 */
#define EXTI_FTSR_TR24_Msk                        (0x01000000UL)               /*!< TR24 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR23_Pos                        (23UL)                       /*!< TR23 (Bit 23)                 */
#define EXTI_FTSR_TR23_Msk                        (0x00800000UL)               /*!< TR23 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR22_Pos                        (22UL)                       /*!< TR22 (Bit 22)                 */
#define EXTI_FTSR_TR22_Msk                        (0x00400000UL)               /*!< TR22 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR21_Pos                        (21UL)                       /*!< TR21 (Bit 21)                 */
#define EXTI_FTSR_TR21_Msk                        (0x00200000UL)               /*!< TR21 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR20_Pos                        (20UL)                       /*!< TR20 (Bit 20)                 */
#define EXTI_FTSR_TR20_Msk                        (0x00100000UL)               /*!< TR20 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR19_Pos                        (19UL)                       /*!< TR19 (Bit 19)                 */
#define EXTI_FTSR_TR19_Msk                        (0x00080000UL)               /*!< TR19 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR18_Pos                        (18UL)                       /*!< TR18 (Bit 18)                 */
#define EXTI_FTSR_TR18_Msk                        (0x00040000UL)               /*!< TR18 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR17_Pos                        (17UL)                       /*!< TR17 (Bit 17)                 */
#define EXTI_FTSR_TR17_Msk                        (0x00020000UL)               /*!< TR17 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR16_Pos                        (16UL)                       /*!< TR16 (Bit 16)                 */
#define EXTI_FTSR_TR16_Msk                        (0x00010000UL)               /*!< TR16 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR15_Pos                        (15UL)                       /*!< TR15 (Bit 15)                 */
#define EXTI_FTSR_TR15_Msk                        (0x00008000UL)               /*!< TR15 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR14_Pos                        (14UL)                       /*!< TR14 (Bit 14)                 */
#define EXTI_FTSR_TR14_Msk                        (0x00004000UL)               /*!< TR14 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR13_Pos                        (13UL)                       /*!< TR13 (Bit 13)                 */
#define EXTI_FTSR_TR13_Msk                        (0x00002000UL)               /*!< TR13 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR12_Pos                        (12UL)                       /*!< TR12 (Bit 12)                 */
#define EXTI_FTSR_TR12_Msk                        (0x00001000UL)               /*!< TR12 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR11_Pos                        (11UL)                       /*!< TR11 (Bit 11)                 */
#define EXTI_FTSR_TR11_Msk                        (0x00000800UL)               /*!< TR11 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR10_Pos                        (10UL)                       /*!< TR10 (Bit 10)                 */
#define EXTI_FTSR_TR10_Msk                        (0x00000400UL)               /*!< TR10 (Bitfield-Mask: 0x01)    */
#define EXTI_FTSR_TR9_Pos                         (9UL)                        /*!< TR9 (Bit 9)                   */
#define EXTI_FTSR_TR9_Msk                         (0x00000200UL)               /*!< TR9 (Bitfield-Mask: 0x01)     */
#define EXTI_FTSR_TR8_Pos                         (8UL)                        /*!< TR8 (Bit 8)                   */
#define EXTI_FTSR_TR8_Msk                         (0x00000100UL)               /*!< TR8 (Bitfield-Mask: 0x01)     */
#define EXTI_FTSR_TR7_Pos                         (7UL)                        /*!< TR7 (Bit 7)                   */
#define EXTI_FTSR_TR7_Msk                         (0x00000080UL)               /*!< TR7 (Bitfield-Mask: 0x01)     */
#define EXTI_FTSR_TR6_Pos                         (6UL)                        /*!< TR6 (Bit 6)                   */
#define EXTI_FTSR_TR6_Msk                         (0x00000040UL)               /*!< TR6 (Bitfield-Mask: 0x01)     */
#define EXTI_FTSR_TR5_Pos                         (5UL)                        /*!< TR5 (Bit 5)                   */
#define EXTI_FTSR_TR5_Msk                         (0x00000020UL)               /*!< TR5 (Bitfield-Mask: 0x01)     */
#define EXTI_FTSR_TR4_Pos                         (4UL)                        /*!< TR4 (Bit 4)                   */
#define EXTI_FTSR_TR4_Msk                         (0x00000010UL)               /*!< TR4 (Bitfield-Mask: 0x01)     */
#define EXTI_FTSR_TR3_Pos                         (3UL)                        /*!< TR3 (Bit 3)                   */
#define EXTI_FTSR_TR3_Msk                         (0x00000008UL)               /*!< TR3 (Bitfield-Mask: 0x01)     */
#define EXTI_FTSR_TR2_Pos                         (2UL)                        /*!< TR2 (Bit 2)                   */
#define EXTI_FTSR_TR2_Msk                         (0x00000004UL)               /*!< TR2 (Bitfield-Mask: 0x01)     */
#define EXTI_FTSR_TR1_Pos                         (1UL)                        /*!< TR1 (Bit 1)                   */
#define EXTI_FTSR_TR1_Msk                         (0x00000002UL)               /*!< TR1 (Bitfield-Mask: 0x01)     */
#define EXTI_FTSR_TR0_Pos                         (0UL)                        /*!< TR0 (Bit 0)                   */
#define EXTI_FTSR_TR0_Msk                         (0x00000001UL)               /*!< TR0 (Bitfield-Mask: 0x01)     */
/* =================================  SWIER  ================================ */
#define EXTI_SWIER_SWIER24_Pos                    (24UL)                       /*!< SWIER24 (Bit 24)              */
#define EXTI_SWIER_SWIER24_Msk                    (0x01000000UL)               /*!< SWIER24 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER23_Pos                    (23UL)                       /*!< SWIER23 (Bit 23)              */
#define EXTI_SWIER_SWIER23_Msk                    (0x00800000UL)               /*!< SWIER23 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER22_Pos                    (22UL)                       /*!< SWIER22 (Bit 22)              */
#define EXTI_SWIER_SWIER22_Msk                    (0x00400000UL)               /*!< SWIER22 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER21_Pos                    (21UL)                       /*!< SWIER21 (Bit 21)              */
#define EXTI_SWIER_SWIER21_Msk                    (0x00200000UL)               /*!< SWIER21 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER20_Pos                    (20UL)                       /*!< SWIER20 (Bit 20)              */
#define EXTI_SWIER_SWIER20_Msk                    (0x00100000UL)               /*!< SWIER20 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER19_Pos                    (19UL)                       /*!< SWIER19 (Bit 19)              */
#define EXTI_SWIER_SWIER19_Msk                    (0x00080000UL)               /*!< SWIER19 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER18_Pos                    (18UL)                       /*!< SWIER18 (Bit 18)              */
#define EXTI_SWIER_SWIER18_Msk                    (0x00040000UL)               /*!< SWIER18 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER17_Pos                    (17UL)                       /*!< SWIER17 (Bit 17)              */
#define EXTI_SWIER_SWIER17_Msk                    (0x00020000UL)               /*!< SWIER17 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER16_Pos                    (16UL)                       /*!< SWIER16 (Bit 16)              */
#define EXTI_SWIER_SWIER16_Msk                    (0x00010000UL)               /*!< SWIER16 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER15_Pos                    (15UL)                       /*!< SWIER15 (Bit 15)              */
#define EXTI_SWIER_SWIER15_Msk                    (0x00008000UL)               /*!< SWIER15 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER14_Pos                    (14UL)                       /*!< SWIER14 (Bit 14)              */
#define EXTI_SWIER_SWIER14_Msk                    (0x00004000UL)               /*!< SWIER14 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER13_Pos                    (13UL)                       /*!< SWIER13 (Bit 13)              */
#define EXTI_SWIER_SWIER13_Msk                    (0x00002000UL)               /*!< SWIER13 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER12_Pos                    (12UL)                       /*!< SWIER12 (Bit 12)              */
#define EXTI_SWIER_SWIER12_Msk                    (0x00001000UL)               /*!< SWIER12 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER11_Pos                    (11UL)                       /*!< SWIER11 (Bit 11)              */
#define EXTI_SWIER_SWIER11_Msk                    (0x00000800UL)               /*!< SWIER11 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER10_Pos                    (10UL)                       /*!< SWIER10 (Bit 10)              */
#define EXTI_SWIER_SWIER10_Msk                    (0x00000400UL)               /*!< SWIER10 (Bitfield-Mask: 0x01) */
#define EXTI_SWIER_SWIER9_Pos                     (9UL)                        /*!< SWIER9 (Bit 9)                */
#define EXTI_SWIER_SWIER9_Msk                     (0x00000200UL)               /*!< SWIER9 (Bitfield-Mask: 0x01)  */
#define EXTI_SWIER_SWIER8_Pos                     (8UL)                        /*!< SWIER8 (Bit 8)                */
#define EXTI_SWIER_SWIER8_Msk                     (0x00000100UL)               /*!< SWIER8 (Bitfield-Mask: 0x01)  */
#define EXTI_SWIER_SWIER7_Pos                     (7UL)                        /*!< SWIER7 (Bit 7)                */
#define EXTI_SWIER_SWIER7_Msk                     (0x00000080UL)               /*!< SWIER7 (Bitfield-Mask: 0x01)  */
#define EXTI_SWIER_SWIER6_Pos                     (6UL)                        /*!< SWIER6 (Bit 6)                */
#define EXTI_SWIER_SWIER6_Msk                     (0x00000040UL)               /*!< SWIER6 (Bitfield-Mask: 0x01)  */
#define EXTI_SWIER_SWIER5_Pos                     (5UL)                        /*!< SWIER5 (Bit 5)                */
#define EXTI_SWIER_SWIER5_Msk                     (0x00000020UL)               /*!< SWIER5 (Bitfield-Mask: 0x01)  */
#define EXTI_SWIER_SWIER4_Pos                     (4UL)                        /*!< SWIER4 (Bit 4)                */
#define EXTI_SWIER_SWIER4_Msk                     (0x00000010UL)               /*!< SWIER4 (Bitfield-Mask: 0x01)  */
#define EXTI_SWIER_SWIER3_Pos                     (3UL)                        /*!< SWIER3 (Bit 3)                */
#define EXTI_SWIER_SWIER3_Msk                     (0x00000008UL)               /*!< SWIER3 (Bitfield-Mask: 0x01)  */
#define EXTI_SWIER_SWIER2_Pos                     (2UL)                        /*!< SWIER2 (Bit 2)                */
#define EXTI_SWIER_SWIER2_Msk                     (0x00000004UL)               /*!< SWIER2 (Bitfield-Mask: 0x01)  */
#define EXTI_SWIER_SWIER1_Pos                     (1UL)                        /*!< SWIER1 (Bit 1)                */
#define EXTI_SWIER_SWIER1_Msk                     (0x00000002UL)               /*!< SWIER1 (Bitfield-Mask: 0x01)  */
#define EXTI_SWIER_SWIER0_Pos                     (0UL)                        /*!< SWIER0 (Bit 0)                */
#define EXTI_SWIER_SWIER0_Msk                     (0x00000001UL)               /*!< SWIER0 (Bitfield-Mask: 0x01)  */
/* ====================================  PR  ================================ */
#define EXTI_PR_PR24_Pos                          (24UL)                       /*!< PR24 (Bit 24)                 */
#define EXTI_PR_PR24_Msk                          (0x01000000UL)               /*!< PR24 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR23_Pos                          (23UL)                       /*!< PR23 (Bit 23)                 */
#define EXTI_PR_PR23_Msk                          (0x00800000UL)               /*!< PR23 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR22_Pos                          (22UL)                       /*!< PR22 (Bit 22)                 */
#define EXTI_PR_PR22_Msk                          (0x00400000UL)               /*!< PR22 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR21_Pos                          (21UL)                       /*!< PR21 (Bit 21)                 */
#define EXTI_PR_PR21_Msk                          (0x00200000UL)               /*!< PR21 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR20_Pos                          (20UL)                       /*!< PR20 (Bit 20)                 */
#define EXTI_PR_PR20_Msk                          (0x00100000UL)               /*!< PR20 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR19_Pos                          (19UL)                       /*!< PR19 (Bit 19)                 */
#define EXTI_PR_PR19_Msk                          (0x00080000UL)               /*!< PR19 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR18_Pos                          (18UL)                       /*!< PR18 (Bit 18)                 */
#define EXTI_PR_PR18_Msk                          (0x00040000UL)               /*!< PR18 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR17_Pos                          (17UL)                       /*!< PR17 (Bit 17)                 */
#define EXTI_PR_PR17_Msk                          (0x00020000UL)               /*!< PR17 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR16_Pos                          (16UL)                       /*!< PR16 (Bit 16)                 */
#define EXTI_PR_PR16_Msk                          (0x00010000UL)               /*!< PR16 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR15_Pos                          (15UL)                       /*!< PR15 (Bit 15)                 */
#define EXTI_PR_PR15_Msk                          (0x00008000UL)               /*!< PR15 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR14_Pos                          (14UL)                       /*!< PR14 (Bit 14)                 */
#define EXTI_PR_PR14_Msk                          (0x00004000UL)               /*!< PR14 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR13_Pos                          (13UL)                       /*!< PR13 (Bit 13)                 */
#define EXTI_PR_PR13_Msk                          (0x00002000UL)               /*!< PR13 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR12_Pos                          (12UL)                       /*!< PR12 (Bit 12)                 */
#define EXTI_PR_PR12_Msk                          (0x00001000UL)               /*!< PR12 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR11_Pos                          (11UL)                       /*!< PR11 (Bit 11)                 */
#define EXTI_PR_PR11_Msk                          (0x00000800UL)               /*!< PR11 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR10_Pos                          (10UL)                       /*!< PR10 (Bit 10)                 */
#define EXTI_PR_PR10_Msk                          (0x00000400UL)               /*!< PR10 (Bitfield-Mask: 0x01)    */
#define EXTI_PR_PR9_Pos                           (9UL)                        /*!< PR9 (Bit 9)                   */
#define EXTI_PR_PR9_Msk                           (0x00000200UL)               /*!< PR9 (Bitfield-Mask: 0x01)     */
#define EXTI_PR_PR8_Pos                           (8UL)                        /*!< PR8 (Bit 8)                   */
#define EXTI_PR_PR8_Msk                           (0x00000100UL)               /*!< PR8 (Bitfield-Mask: 0x01)     */
#define EXTI_PR_PR7_Pos                           (7UL)                        /*!< PR7 (Bit 7)                   */
#define EXTI_PR_PR7_Msk                           (0x00000080UL)               /*!< PR7 (Bitfield-Mask: 0x01)     */
#define EXTI_PR_PR6_Pos                           (6UL)                        /*!< PR6 (Bit 6)                   */
#define EXTI_PR_PR6_Msk                           (0x00000040UL)               /*!< PR6 (Bitfield-Mask: 0x01)     */
#define EXTI_PR_PR5_Pos                           (5UL)                        /*!< PR5 (Bit 5)                   */
#define EXTI_PR_PR5_Msk                           (0x00000020UL)               /*!< PR5 (Bitfield-Mask: 0x01)     */
#define EXTI_PR_PR4_Pos                           (4UL)                        /*!< PR4 (Bit 4)                   */
#define EXTI_PR_PR4_Msk                           (0x00000010UL)               /*!< PR4 (Bitfield-Mask: 0x01)     */
#define EXTI_PR_PR3_Pos                           (3UL)                        /*!< PR3 (Bit 3)                   */
#define EXTI_PR_PR3_Msk                           (0x00000008UL)               /*!< PR3 (Bitfield-Mask: 0x01)     */
#define EXTI_PR_PR2_Pos                           (2UL)                        /*!< PR2 (Bit 2)                   */
#define EXTI_PR_PR2_Msk                           (0x00000004UL)               /*!< PR2 (Bitfield-Mask: 0x01)     */
#define EXTI_PR_PR1_Pos                           (1UL)                        /*!< PR1 (Bit 1)                   */
#define EXTI_PR_PR1_Msk                           (0x00000002UL)               /*!< PR1 (Bitfield-Mask: 0x01)     */
#define EXTI_PR_PR0_Pos                           (0UL)                        /*!< PR0 (Bit 0)                   */
#define EXTI_PR_PR0_Msk                           (0x00000001UL)               /*!< PR0 (Bitfield-Mask: 0x01)     */
/* =====================================  FR  =============================== */
#define EXTI_FR_FR15_Pos                          (30UL)                       /*!< FR15 (Bit 30)                 */
#define EXTI_FR_FR15_Msk                          (0xC0000000UL)               /*!< FR15 (Bitfield-Mask: 0x03)    */
#define EXTI_FR_FR14_Pos                          (28UL)                       /*!< FR14 (Bit 28)                 */
#define EXTI_FR_FR14_Msk                          (0x30000000UL)               /*!< FR14 (Bitfield-Mask: 0x03)    */
#define EXTI_FR_FR13_Pos                          (26UL)                       /*!< FR13 (Bit 26)                 */
#define EXTI_FR_FR13_Msk                          (0xC0000000UL)               /*!< FR13 (Bitfield-Mask: 0x03)    */
#define EXTI_FR_FR12_Pos                          (24UL)                       /*!< FR12 (Bit 24)                 */
#define EXTI_FR_FR12_Msk                          (0x03000000UL)               /*!< FR12 (Bitfield-Mask: 0x03)    */
#define EXTI_FR_FR11_Pos                          (22UL)                       /*!< FR11 (Bit 22)                 */
#define EXTI_FR_FR11_Msk                          (0x00C00000UL)               /*!< FR11 (Bitfield-Mask: 0x03)    */
#define EXTI_FR_FR10_Pos                          (20UL)                       /*!< FR10 (Bit 20)                 */
#define EXTI_FR_FR10_Msk                          (0x00300000UL)               /*!< FR10 (Bitfield-Mask: 0x03)    */
#define EXTI_FR_FR9_Pos                           (18UL)                       /*!< FR9 (Bit 18)                  */
#define EXTI_FR_FR9_Msk                           (0x000C0000UL)               /*!< FR9 (Bitfield-Mask: 0x03)     */
#define EXTI_FR_FR8_Pos                           (16UL)                       /*!< FR8 (Bit 16)                  */
#define EXTI_FR_FR8_Msk                           (0x00300000UL)               /*!< FR8 (Bitfield-Mask: 0x03)     */
#define EXTI_FR_FR7_Pos                           (14UL)                       /*!< FR7 (Bit 14)                  */
#define EXTI_FR_FR7_Msk                           (0x0000c000UL)               /*!< FR7 (Bitfield-Mask: 0x03)     */
#define EXTI_FR_FR6_Pos                           (12UL)                       /*!< FR6 (Bit 12)                  */
#define EXTI_FR_FR6_Msk                           (0x00003000UL)               /*!< FR6 (Bitfield-Mask: 0x03)     */
#define EXTI_FR_FR5_Pos                           (10UL)                       /*!< FR5 (Bit 10)                  */
#define EXTI_FR_FR5_Msk                           (0x00000C00UL)               /*!< FR5 (Bitfield-Mask: 0x03)     */
#define EXTI_FR_FR4_Pos                           (8UL)                        /*!< FR4 (Bit 8)                   */
#define EXTI_FR_FR4_Msk                           (0x00000300UL)               /*!< FR4 (Bitfield-Mask: 0x03)     */
#define EXTI_FR_FR3_Pos                           (6UL)                        /*!< FR3 (Bit 6)                   */
#define EXTI_FR_FR3_Msk                           (0x000000C0UL)               /*!< FR3 (Bitfield-Mask: 0x03)     */
#define EXTI_FR_FR2_Pos                           (4UL)                        /*!< FR2 (Bit 4)                   */
#define EXTI_FR_FR2_Msk                           (0x00000030UL)               /*!< FR2 (Bitfield-Mask: 0x03)     */
#define EXTI_FR_FR1_Pos                           (2UL)                        /*!< FR1 (Bit 2)                   */
#define EXTI_FR_FR1_Msk                           (0x0000000CUL)               /*!< FR1 (Bitfield-Mask: 0x03)     */
#define EXTI_FR_FR0_Pos                           (0UL)                        /*!< FR0 (Bit 0)                   */
#define EXTI_FR_FR0_Msk                           (0x00000003UL)               /*!< FR0 (Bitfield-Mask: 0x03)     */
/* =====================================  CR  =============================== */
#define EXTI_CR_PSC_Pos                           (28UL)                       /*!< PSC (Bit 28)                  */
#define EXTI_CR_PSC_Msk                           (0x70000000UL)               /*!< PSC (Bitfield-Mask: 0x07)     */
#define EXTI_CR_LR24_Pos                          (24UL)                       /*!< LR24 (Bit 24)                 */
#define EXTI_CR_LR24_Msk                          (0x01000000UL)               /*!< LR24 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR23_Pos                          (23UL)                       /*!< LR23 (Bit 23)                 */
#define EXTI_CR_LR23_Msk                          (0x00800000UL)               /*!< LR23 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR22_Pos                          (22UL)                       /*!< LR22 (Bit 22)                 */
#define EXTI_CR_LR22_Msk                          (0x00400000UL)               /*!< LR22 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR21_Pos                          (21UL)                       /*!< LR21 (Bit 21)                 */
#define EXTI_CR_LR21_Msk                          (0x00200000UL)               /*!< LR21 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR20_Pos                          (20UL)                       /*!< LR20 (Bit 20)                 */
#define EXTI_CR_LR20_Msk                          (0x00100000UL)               /*!< LR20 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR19_Pos                          (19UL)                       /*!< LR19 (Bit 19)                 */
#define EXTI_CR_LR19_Msk                          (0x00080000UL)               /*!< LR19 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR18_Pos                          (18UL)                       /*!< LR18 (Bit 18)                 */
#define EXTI_CR_LR18_Msk                          (0x00040000UL)               /*!< LR18 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR17_Pos                          (17UL)                       /*!< LR17 (Bit 17)                 */
#define EXTI_CR_LR17_Msk                          (0x00020000UL)               /*!< LR17 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR16_Pos                          (16UL)                       /*!< LR16 (Bit 16)                 */
#define EXTI_CR_LR16_Msk                          (0x00010000UL)               /*!< LR16 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR15_Pos                          (15UL)                       /*!< LR15 (Bit 15)                 */
#define EXTI_CR_LR15_Msk                          (0x00008000UL)               /*!< LR15 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR14_Pos                          (14UL)                       /*!< LR14 (Bit 14)                 */
#define EXTI_CR_LR14_Msk                          (0x00004000UL)               /*!< LR14 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR13_Pos                          (13UL)                       /*!< LR13 (Bit 13)                 */
#define EXTI_CR_LR13_Msk                          (0x00002000UL)               /*!< LR13 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR12_Pos                          (12UL)                       /*!< LR12 (Bit 12)                 */
#define EXTI_CR_LR12_Msk                          (0x00001000UL)               /*!< LR12 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR11_Pos                          (11UL)                       /*!< LR11 (Bit 11)                 */
#define EXTI_CR_LR11_Msk                          (0x00000800UL)               /*!< LR11 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR10_Pos                          (10UL)                       /*!< LR10 (Bit 10)                 */
#define EXTI_CR_LR10_Msk                          (0x00000400UL)               /*!< LR10 (Bitfield-Mask: 0x01)    */
#define EXTI_CR_LR9_Pos                           (9UL)                        /*!< LR9 (Bit 9)                   */
#define EXTI_CR_LR9_Msk                           (0x00000200UL)               /*!< LR9 (Bitfield-Mask: 0x01)     */
#define EXTI_CR_LR8_Pos                           (8UL)                        /*!< LR8 (Bit 8)                   */
#define EXTI_CR_LR8_Msk                           (0x00000100UL)               /*!< LR8 (Bitfield-Mask: 0x01)     */
#define EXTI_CR_LR7_Pos                           (7UL)                        /*!< LR7 (Bit 7)                   */
#define EXTI_CR_LR7_Msk                           (0x00000080UL)               /*!< LR7 (Bitfield-Mask: 0x01)     */
#define EXTI_CR_LR6_Pos                           (6UL)                        /*!< LR6 (Bit 6)                   */
#define EXTI_CR_LR6_Msk                           (0x00000040UL)               /*!< LR6 (Bitfield-Mask: 0x01)     */
#define EXTI_CR_LR5_Pos                           (5UL)                        /*!< LR5 (Bit 5)                   */
#define EXTI_CR_LR5_Msk                           (0x00000020UL)               /*!< LR5 (Bitfield-Mask: 0x01)     */
#define EXTI_CR_LR4_Pos                           (4UL)                        /*!< LR4 (Bit 4)                   */
#define EXTI_CR_LR4_Msk                           (0x00000010UL)               /*!< LR4 (Bitfield-Mask: 0x01)     */
#define EXTI_CR_LR3_Pos                           (3UL)                        /*!< LR3 (Bit 3)                   */
#define EXTI_CR_LR3_Msk                           (0x00000008UL)               /*!< LR3 (Bitfield-Mask: 0x01)     */
#define EXTI_CR_LR2_Pos                           (2UL)                        /*!< LR2 (Bit 2)                   */
#define EXTI_CR_LR2_Msk                           (0x00000004UL)               /*!< LR2 (Bitfield-Mask: 0x01)     */
#define EXTI_CR_LR1_Pos                           (1UL)                        /*!< LR1 (Bit 1)                   */
#define EXTI_CR_LR1_Msk                           (0x00000002UL)               /*!< LR1 (Bitfield-Mask: 0x01)     */
#define EXTI_CR_LR0_Pos                           (0UL)                        /*!< LR0 (Bit 0)                   */
#define EXTI_CR_LR0_Msk                           (0x00000001UL)               /*!< LR0 (Bitfield-Mask: 0x01)     */

/* ========================================================================== */
/* ================                  END                      =============== */
/* ========================================================================== */


  void EXTI_DeInit(void);
  void EXTI_Init(EXTI_InitTypeDef *EXTI_InitStruct);
  void EXTI_StructInit(EXTI_InitTypeDef *EXTI_InitStruct);
  void EXTI_GenerateSWInterrupt(uint32_t EXTI_Line);
  FlagStatus EXTI_GetFlagStatus(uint32_t EXTI_Line);
  void EXTI_ClearFlag(uint32_t EXTI_Line);
  ITStatus EXTI_GetITStatus(uint32_t EXTI_Line);
  void EXTI_ClearITPendingBit(uint32_t EXTI_Line);

  //connect exti interrupt line to GPIOA,GPIOB,GPIOC or GPIOD
  void EXTI_ConnectTo(GPIO_TypeDef *GPIOx, uint32_t EXTI_Line);

void NVIC_Init(NVIC_InitTypeDef *NVIC_InitStruct);
#ifdef __cplusplus
}
#endif

#endif // __LCM32F06X_EXTI_H__
