/**
  ******************************************************************************
  * @file    lcm32f06x_adc.h 
  * @author  System R&D Team
  * @version V2.0.2
  * @date    10-April-2025
  * @brief   Header for adc.c module
  ******************************************************************************
  * @attention
  *
  * Copyright (c) Hangzhou Lingxin Microelectronics Co.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __LCM32F06X_ADC_H__
#define __LCM32F06X_ADC_H__

/* Includes ------------------------------------------------------------------*/
#include "lcm32f06x.h"

#ifdef __cplusplus
  extern "C" {
#endif

/** 
  * @brief ADC Init structure definition
  */
typedef struct
{
  uint32_t ADC_ClkMode;               /*!< EN: clock division.
                                           CN: ʱӷƵ
                                           @Register: ADC_CFGR @Bits:[31:28]CKMODE.
                                           @default: ADC_ClockMode_SynClkDiv1 */

  uint32_t ADC_ReferenceVoltage;      /*!< EN: 
                                           CN: οѹѡ
                                           @Register: ADC_CFGR @Bits:[18:17]VFS,VREF.
                                           @default: ADC_ReferenceVoltage_3V2 */

  uint32_t ADC_VCCChanelMapping;      /*!< EN: Internal voltage monitoring mapping SOC[x].
                                           CN: ڲѹӳ SOC[x]
                                           @Register: ADC_CFGR @Bits:[15:14]SV,SVS.
                                           @default: ADC_VCCChanel_DISABLE */

  uint32_t ADC_TempChanelMapping;     /*!< EN: Temperature monitoring mapping SOC[x].
                                           CN: ¶ȼӳ SOC[x]
                                           @Register: ADC_CFGR @Bits:[13:12]ST,STS.
                                           @default: ADC_TempChanel_DISABLE */

  uint32_t ADC_WorkMode;              /*!< EN: ADC working mode.
                                           CN: ADC ģʽ
                                           @Register: ADC_CFGR @Bits:[19]MD2.
                                           @default: ADC_WorkMode_Normal */

  uint32_t ADC_OverFlowMode;          /*!< EN: Data overflow mode.
                                           CN: ģʽ
                                           @Register: ADC_CFGR @Bits:[1]OVF.
                                           @default: ADC_OverFlow_New */

  FunctionalState ADC_LowPowerMode;   /*!< EN: low power mode.
                                           CN: ͹ģʽ
                                           @Register: ADC_CFGR @Bits:[16]LP.
                                           @default: DISABLE */

  uint32_t ADC_DataAlign;             /*!< EN: Data alignment direction.
                                           CN: ݶ뷽
                                           @Register: ADC_CFGR @Bits:[0]ALIGN.
                                           @default: ADC_DataAlign_Right */

  uint32_t ADC_TempPulseTime;         /*!< EN: ST sample pulse width.
                                           CN: ¶Ȳȡ
                                           @Register: ADC_SMPR @Bits:[21:20]STSPW.
                                           @range 0x0 - 0x3
                                           @default: 0x0 */

  uint32_t ADC_ConvertTime;           /*!< EN: Conversion cycle.
                                           CN: ڡ
                                           @Register: ADC_SMPR @Bits:[21:20]CC.
                                           @range 0x0 - 0xF
                                           @default: 0x0 */

  uint32_t ADC_ConvertPulseTime;      /*!< EN: Convert pulse width.
                                           CN: תȡ
                                           @Register: ADC_SMPR @Bits:[21:20]CPW.
                                           @range 0x0 - 0x3
                                           @default: 0x2 */

  uint32_t ADC_SampleTime;            /*!< EN: sample pulse width.
                                           CN: ADȡ
                                           @Register: ADC_SMPR @Bits:[21:20]SPW.
                                           @range 0x0 - 0xF
                                           @default: 0x4 */

  uint32_t ADC_AdjustTime;            /*!< EN: adjust pulse width.
                                           CN: ϵʱȡ
                                           @Register: ADC_SMPR @Bits:[21:20]APW.
                                           @range 0x0 - 0x7
                                           @default: 0x7 */
} ADC_InitTypeDef;

/* @defgroup SOC */
#define SOC0                            ((uint8_t)0x00) /*@EN: Single sampling conversion 0  @CN: βת0  */
#define SOC1                            ((uint8_t)0x01) /*@EN: Single sampling conversion 1  @CN: βת1  */
#define SOC2                            ((uint8_t)0x02) /*@EN: Single sampling conversion 2  @CN: βת2  */
#define SOC3                            ((uint8_t)0x03) /*@EN: Single sampling conversion 3  @CN: βת3  */
#define SOC4                            ((uint8_t)0x04) /*@EN: Single sampling conversion 4  @CN: βת4  */
#define SOC5                            ((uint8_t)0x05) /*@EN: Single sampling conversion 5  @CN: βת5  */
#define SOC6                            ((uint8_t)0x06) /*@EN: Single sampling conversion 6  @CN: βת6  */
#define SOC7                            ((uint8_t)0x07) /*@EN: Single sampling conversion 7  @CN: βת7  */
#define SOC8                            ((uint8_t)0x08) /*@EN: Single sampling conversion 8  @CN: βת8  */
#define SOC9                            ((uint8_t)0x09) /*@EN: Single sampling conversion 9  @CN: βת9  */
#define SOC10                           ((uint8_t)0x0A) /*@EN: Single sampling conversion 10 @CN: βת10 */
#define SOC11                           ((uint8_t)0x0B) /*@EN: Single sampling conversion 11 @CN: βת11 */
#define SOC12                           ((uint8_t)0x0C) /*@EN: Single sampling conversion 12 @CN: βת12 */
#define SOC13                           ((uint8_t)0x0D) /*@EN: Single sampling conversion 13 @CN: βת13 */
#define SOC14                           ((uint8_t)0x0E) /*@EN: Single sampling conversion 14 @CN: βת14 */
#define SOC15                           ((uint8_t)0x0F) /*@EN: Single sampling conversion 15 @CN: βת15 */
/* @end */

#define DR(x)                           *(unsigned int *)(&(ADC->DR0)+0X01 *x)

#define ADCSOC(ADCx, SOCx)              (*(unsigned int *)(&(ADCx->ADCSOC0) + (0x01 * SOCx)))

/* @defgroup ADC_ClkMode */
/* ͬʱ pclk */
#define ADC_ClockMode_SynClkDiv1        ((uint32_t)0x00000000) /*@EN: Synchronous clock mode               @CN: ͬʱģʽ         */
#define ADC_ClockMode_SynClkDiv2        ((uint32_t)0x08000000) /*@EN: Synchronous clock mode divided by 2  @CN: ͬʱģʽ 2 Ƶ  */
#define ADC_ClockMode_SynClkDiv3        ((uint32_t)0x10000000) /*@EN: Synchronous clock mode divided by 3  @CN: ͬʱģʽ 3 Ƶ  */
#define ADC_ClockMode_SynClkDiv4        ((uint32_t)0x18000000) /*@EN: Synchronous clock mode divided by 4  @CN: ͬʱģʽ 4 Ƶ  */
#define ADC_ClockMode_SynClkDiv5        ((uint32_t)0x20000000) /*@EN: Synchronous clock mode divided by 5  @CN: ͬʱģʽ 5 Ƶ  */
#define ADC_ClockMode_SynClkDiv6        ((uint32_t)0x28000000) /*@EN: Synchronous clock mode divided by 6  @CN: ͬʱģʽ 6 Ƶ  */
#define ADC_ClockMode_SynClkDiv8        ((uint32_t)0x30000000) /*@EN: Synchronous clock mode divided by 8  @CN: ͬʱģʽ 8 Ƶ  */
#define ADC_ClockMode_SynClkDiv10       ((uint32_t)0x40000000) /*@EN: Synchronous clock mode divided by 10 @CN: ͬʱģʽ 10 Ƶ */
#define ADC_ClockMode_SynClkDiv12       ((uint32_t)0x48000000) /*@EN: Synchronous clock mode divided by 12 @CN: ͬʱģʽ 12 Ƶ */
#define ADC_ClockMode_SynClkDiv14       ((uint32_t)0x50000000) /*@EN: Synchronous clock mode divided by 14 @CN: ͬʱģʽ 14 Ƶ */
#define ADC_ClockMode_SynClkDiv16       ((uint32_t)0x58000000) /*@EN: Synchronous clock mode divided by 16 @CN: ͬʱģʽ 16 Ƶ */
/* 첽ʱpllclk */
#define ADC_ClockMode_PLLClkDiv1        ((uint32_t)0x80000000) /*@EN: PLL clock mode               @CN: 첽ʱģʽ         */
#define ADC_ClockMode_PLLClkDiv2        ((uint32_t)0x88000000) /*@EN: PLL clock mode divided by 2  @CN: 첽ʱģʽ 2 Ƶ  */
#define ADC_ClockMode_PLLClkDiv3        ((uint32_t)0x90000000) /*@EN: PLL clock mode divided by 3  @CN: 첽ʱģʽ 3 Ƶ  */
#define ADC_ClockMode_PLLClkDiv4        ((uint32_t)0x98000000) /*@EN: PLL clock mode divided by 4  @CN: 첽ʱģʽ 4 Ƶ  */
#define ADC_ClockMode_PLLClkDiv5        ((uint32_t)0xA0000000) /*@EN: PLL clock mode divided by 5  @CN: 첽ʱģʽ 5 Ƶ  */
#define ADC_ClockMode_PLLClkDiv6        ((uint32_t)0xA8000000) /*@EN: PLL clock mode divided by 6  @CN: 첽ʱģʽ 6 Ƶ  */
#define ADC_ClockMode_PLLClkDiv8        ((uint32_t)0xB0000000) /*@EN: PLL clock mode divided by 8  @CN: 첽ʱģʽ 8 Ƶ  */
#define ADC_ClockMode_PLLClkDiv10       ((uint32_t)0xC0000000) /*@EN: PLL clock mode divided by 10 @CN: 첽ʱģʽ 10 Ƶ */
#define ADC_ClockMode_PLLClkDiv12       ((uint32_t)0xC8000000) /*@EN: PLL clock mode divided by 12 @CN: 첽ʱģʽ 12 Ƶ */
#define ADC_ClockMode_PLLClkDiv14       ((uint32_t)0xD0000000) /*@EN: PLL clock mode divided by 14 @CN: 첽ʱģʽ 14 Ƶ */
#define ADC_ClockMode_PLLClkDiv16       ((uint32_t)0xD8000000) /*@EN: PLL clock mode divided by 16 @CN: 첽ʱģʽ 16 Ƶ */
/* @end */

/* @defgroup ADC_ReferenceVoltage */
#define ADC_ReferenceVoltage_3V2        ((uint32_t)0x00000000) /*@EN: Reference voltage 3.2V @CN: οѹ 3.2V */
#define ADC_ReferenceVoltage_4V9        ((uint32_t)0x00020000) /*@EN: Reference voltage 4.9V @CN: οѹ 4.9V */
#define ADC_ReferenceVoltage_VDDA       ((uint32_t)0x00040000) /*@EN: Reference voltage VDDA @CN: οѹ VDDA */
/* @end */

/* @defgroup ADC_VCCChanelMapping */
#define ADC_VCCChanel_DISABLE           ((uint32_t)0x00000000) /*@EN: Internal voltage monitoring mapping Disable @CN: ڲѹӳر*/
#define ADC_VCCChanel_SOC15             ((uint32_t)0x00004000) /*@EN: Internal voltage monitoring mapping SOC [15] @CN: ڲѹӳ SOC[15] */
#define ADC_VCCChanel_SOC7              ((uint32_t)0x0000C000) /*@EN: Internal voltage monitoring mapping SOC [7] @CN: ڲѹӳ SOC[7] */
/* @end */

/* @defgroup ADC_TempChanelMapping */
#define ADC_TempChanel_DISABLE          ((uint32_t)0x00000000) /*@EN: Temperature monitoring mapping Disable @CN: ¶ȼӳر */
#define ADC_TempChanel_SOC15            ((uint32_t)0x00001000) /*@EN: Temperature monitoring mapping SOC [15] @CN: ¶ȼӳ SOC[15] */
#define ADC_TempChanel_SOC7             ((uint32_t)0x00003000) /*@EN: Temperature monitoring mapping SOC [7] @CN: ¶ȼӳ SOC[7] */
/* @end */

/* @defgroup ADC_WorkMode */
#define ADC_WorkMode_Normal             ((uint32_t)0x00000000) /*@EN: Normal Working Mode @CN: ģʽ */
#define ADC_WorkMode_Stable             ((uint32_t)0x00080000) /*@EN: Stable Mode @CN: ȶģʽ */
/* @end */

/* @defgroup ADC_OverFlowMode */
#define ADC_OverFlow_New                ((uint32_t)0x00000000) /*@EN: DOF events retain new data @CN: DOF¼ */
#define ADC_OverFlow_Old                ((uint32_t)0x00000002) /*@EN: DOF events retain old data @CN: DOF¼ */
/* @end */

/* @defgroup ADC_OverFlowMode */
#define ADC_DataAlign_Right             ((uint32_t)0x00000000) /*@EN: Right aligned 16 bits of data, padding high zeros @CN: Ҷ 16 λλ */
#define ADC_DataAlign_Left              ((uint32_t)0x00000001) /*@EN: Left aligned 16 bits of data, padding low zeros @CN:  16 λλ */
/* @end */

/* @defgroup ExternTigger */
#define ADC_ExternTrigeer0              ((uint8_t)0x00)        /*@EN: external trigger 0 @CN: ⲿ 0 */
#define ADC_ExternTrigeer1              ((uint8_t)0x01)        /*@EN: external trigger 1 @CN: ⲿ 1 */
#define ADC_ExternTrigeer2              ((uint8_t)0x02)        /*@EN: external trigger 2 @CN: ⲿ 2 */
#define ADC_ExternTrigeer3              ((uint8_t)0x03)        /*@EN: external trigger 3 @CN: ⲿ3 */
#define IS_ADC_EXTERNT_TRIGGER(TRIGGER) ((TRIGGER == ADC_ExternTrigeer0) || \
                                         (TRIGGER == ADC_ExternTrigeer1) || \
                                         (TRIGGER == ADC_ExternTrigeer2) || \
                                         (TRIGGER == ADC_ExternTrigeer3))
/* @end */

/* @defgroup TriggerSelect */
#define TriggerSelect0                  ((uint32_t)0x00000000) /*@EN: External trigger source 0 @CN: ⲿԴ0 POF¼SOCԱ*/
#define TriggerSelect1                  ((uint32_t)0x00000010) /*@EN: External trigger source 1 @CN: ⲿԴ1 POF¼SOCԱ*/
#define TriggerSelect2                  ((uint32_t)0x00000020) /*@EN: External trigger source 2 @CN: ⲿԴ2 POF¼SOCԱ*/
#define TriggerSelect3                  ((uint32_t)0x00000030) /*@EN: External trigger source 3 @CN: ⲿԴ3 POF¼SOCԱ*/
#define TriggerSelect0_FP               ((uint32_t)0x00000080) /*@EN: External trigger source 0 @CN: ⲿԴ0 POF¼SOCԱ */
#define TriggerSelect1_FP               ((uint32_t)0x00000090) /*@EN: External trigger source 1 @CN: ⲿԴ1 POF¼SOCԱ*/
#define TriggerSelect2_FP               ((uint32_t)0x000000A0) /*@EN: External trigger source 2 @CN: ⲿԴ2 POF¼SOCԱ*/
#define TriggerSelect3_FP               ((uint32_t)0x000000B0) /*@EN: External trigger source 3 @CN: ⲿԴ3 POF¼SOCԱ*/
#define TriggerSelectNone               ((uint32_t)0x000000F0) /*@EN: trigger source software @CN:  */
/* @end */

/* @defgroup TriggerSelect0 */
#define TriggerSelect0_TIM1_TRIG_OC1    ((uint32_t)0x00000001) /*@EN: Hardware trigger 0: TIM1 TRIG OC1 @CN: Ӳ0TIM1 TRIG OC1*/
#define TriggerSelect0_TIM2_TRIG_OC1    ((uint32_t)0x00000002) /*@EN: Hardware trigger 0: TIM2 TRIG OC1 @CN: Ӳ0TIM2 TRIG OC1 */
#define TriggerSelect0_TIM15_TRIG_OC1   ((uint32_t)0x00000004) /*@EN: Hardware trigger 0: TIM15 TRIG OC1 @CN: Ӳ0TIM15 TRIG OC1 */
#define TriggerSelect0_EXTI1            ((uint32_t)0x00000008) /*@EN: Hardware trigger 0: EXTI1 @CN: Ӳ0EXTI1 */
#define TriggerSelect0_TIM1_TRIG_OC2    ((uint32_t)0x00000010) /*@EN: Hardware trigger 0: TIM1 TRIG OC2 @CN: Ӳ0TIM1 TRIG OC2 */
#define TriggerSelect0_TIM1_TRIG_OC3    ((uint32_t)0x00000020) /*@EN: Hardware trigger 0: TIM1 TRIG OC3 @CN: Ӳ0TIM1 TRIG OC3 */
#define TriggerSelect0_TIM1_TRIG_OC4    ((uint32_t)0x00000040) /*@EN: Hardware trigger 0: TIM1 TRIG OC4 @CN: Ӳ0TIM1 TRIG OC4 */
#define TriggerSelect0_TIM1_TRIG_OC5    ((uint32_t)0x00000080) /*@EN: Hardware trigger 0: TIM1 TRIG OC5 @CN: Ӳ0TIM1 TRIG OC5 */
/* @end */

/* @defgroup TriggerSelect1 */
#define TriggerSelect1_TIM1_TRIG_OC2    ((uint32_t)0x00000001) /*@EN: Hardware trigger 1: TIM1 TRIG OC2 @CN: Ӳ1TIM1 TRIG OC2 */
#define TriggerSelect1_TIM2_TRIG_OC2    ((uint32_t)0x00000002) /*@EN: Hardware trigger 1: TIM2 TRIG OC2 @CN: Ӳ1TIM2 TRIG OC2 */
#define TriggerSelect1_TIM15_TRIG_OC2   ((uint32_t)0x00000004) /*@EN: Hardware trigger 1: TIM15 TRIG OC2 @CN: Ӳ1TIM15 TRIG OC2 */
#define TriggerSelect1_EXTI3            ((uint32_t)0x00000008) /*@EN: Hardware trigger 1: EXTI3 @CN: Ӳ1EXTI3 */
#define TriggerSelect1_TIM2_TRIG_OC1    ((uint32_t)0x00000010) /*@EN: Hardware trigger 1: TIM2 TRIG OC2 @CN: Ӳ1TIM2 TRIG OC2 */
#define TriggerSelect1_TIM2_TRIG_OC3    ((uint32_t)0x00000020) /*@EN: Hardware trigger 1: TIM2 TRIG OC2 @CN: Ӳ1TIM2 TRIG OC2 */
#define TriggerSelect1_TIM2_TRIG_OC4    ((uint32_t)0x00000040) /*@EN: Hardware trigger 1: TIM2 TRIG OC2 @CN: Ӳ1TIM2 TRIG OC2 */
#define TriggerSelect1_ACMP0            ((uint32_t)0x00000080) /*@EN: Hardware trigger 1: ACMP0 @CN: Ӳ1ACMP0 */
/* @end */

/* @defgroup TriggerSelect2 */
#define TriggerSelect2_TIM1_TRIG_OC3    ((uint32_t)0x00000001) /*@EN: Hardware trigger 2: TIM1 TRIG OC3 @CN: Ӳ2TIM1 TRIG OC3 */
#define TriggerSelect2_TIM2_TRIG_OC3    ((uint32_t)0x00000002) /*@EN: Hardware trigger 2: TIM2 TRIG OC3 @CN: Ӳ2TIM2 TRIG OC3 */
#define TriggerSelect2_TIM16_TRIG_OC1   ((uint32_t)0x00000004) /*@EN: Hardware trigger 2: TIM16 TRIG OC1 @CN: Ӳ2TIM16 TRIG OC1 */
#define TriggerSelect2_EXTI9            ((uint32_t)0x00000008) /*@EN: Hardware trigger 2: EXTI9 @CN: Ӳ2EXTI9 */
#define TriggerSelect2_TIM17_TRIG_OC1   ((uint32_t)0x00000020) /*@EN: Hardware trigger 2: TIM17_TRIG_OC1  @CN: Ӳ2TIM17_TRIG_OC1  */
#define TriggerSelect2_ACMP1            ((uint32_t)0x00000040) /*@EN: Hardware trigger 2: ACMP1 @CN: Ӳ2ACMP1*/
/* @end */

/* @defgroup TriggerSelect3 */
#define TriggerSelect3_TIM1_TRIG_OC4    ((uint32_t)0x00000001) /*@EN: Hardware trigger 3: TIM1 TRIG OC4 @CN: Ӳ3TIM1 TRIG OC4 */
#define TriggerSelect3_TIM2_TRIG_OC4    ((uint32_t)0x00000002) /*@EN: Hardware trigger 3: TIM2 TRIG OC4 @CN: Ӳ3TIM2 TRIG OC4 */
#define TriggerSelect3_TIM17_TRIG_OC1   ((uint32_t)0x00000004) /*@EN: Hardware trigger 3: TIM17 TRIG OC1 @CN: Ӳ3TIM17 TRIG OC1 */
#define TriggerSelect3_EXTI11           ((uint32_t)0x00000008) /*@EN: Hardware trigger 3: EXTI11 @CN: Ӳ3EXTI11 */
#define TriggerSelect3_TIM1_TRGO        ((uint32_t)0x00000010) /*@EN: Hardware trigger 3: TIM1 TRGO @CN: Ӳ3TIM1 TRGO */
#define TriggerSelect3_TIM2_TRGO        ((uint32_t)0x00000020) /*@EN: Hardware trigger 3: TIM2 TRGO @CN: Ӳ3TIM2 TRGO */
#define TriggerSelect3_TIM15_TRGO       ((uint32_t)0x00000040) /*@EN: Hardware trigger 3: TIM15 TRGO @CN: Ӳ3TIM15 TRGO */
#define TriggerSelect3_TIM1_TRIG_OC5    ((uint32_t)0x00000080) /*@EN: Hardware trigger 3: TIM1 TRIG OC5 @CN: Ӳ3TIM1 TRIG OC5 */
/* @end */

/* @defgroup Triggermode */
#define Triggermode_rising              ((uint32_t)0x00000000) /*@EN: Trigger mode: Rising edge trigger @CN: ģʽش */
#define Triggermode_falling             ((uint32_t)0x00000001) /*@EN: Trigger mode: Falling edge trigger @CN: ģʽ½ش */
#define Triggermode_rising_falling      ((uint32_t)0x00000002) /*@EN: Trigger mode: Double edge trigger @CN: ģʽ˫ش */
#define Triggermode_Dis                 ((uint32_t)0x00000003) /*@EN: Trigger mode: hardware triggering is prohibited, only software triggering is supported @CN: ģʽֹӲ֧ */
/* @end */

/* @defgroup ADC_IT */
#define ADC_IT_DOF                      ((uint32_t)0x00008000) /*@EN: DOF interrupt @CN: DOFж */
#define ADC_IT_AWD0                     ((uint32_t)0x00000001) /*@EN: AWD0 interrupt @CN: AWD0ж*/
#define ADC_IT_AWD1                     ((uint32_t)0x00000002) /*@EN: AWD1 interrupt @CN: AWD1ж*/
#define IS_ADC_IT(ADC_IT)               ((ADC_IT == ADC_IT_DOF)  || \
                                         (ADC_IT == ADC_IT_AWD0) || \
                                         (ADC_IT == ADC_IT_AWD1))
/* @end */

/* @defgroup ADC_SOC_IT */
#define ADC_IT_EOC0                 ((uint32_t)0x00000001) /*@EN: EOC0  interrupt @CN: EOC0 ж */
#define ADC_IT_EOC1                 ((uint32_t)0x00000002) /*@EN: EOC1  interrupt @CN: EOC1 ж */
#define ADC_IT_EOC2                 ((uint32_t)0x00000004) /*@EN: EOC2  interrupt @CN: EOC2 ж */
#define ADC_IT_EOC3                 ((uint32_t)0x00000008) /*@EN: EOC3  interrupt @CN: EOC3 ж */
#define ADC_IT_EOC4                 ((uint32_t)0x00000010) /*@EN: EOC4  interrupt @CN: EOC4 ж */
#define ADC_IT_EOC5                 ((uint32_t)0x00000020) /*@EN: EOC5  interrupt @CN: EOC5 ж */
#define ADC_IT_EOC6                 ((uint32_t)0x00000040) /*@EN: EOC6  interrupt @CN: EOC6 ж */
#define ADC_IT_EOC7                 ((uint32_t)0x00000080) /*@EN: EOC7  interrupt @CN: EOC7 ж */
#define ADC_IT_EOC8                 ((uint32_t)0x00000100) /*@EN: EOC8  interrupt @CN: EOC8 ж */
#define ADC_IT_EOC9                 ((uint32_t)0x00000200) /*@EN: EOC9  interrupt @CN: EOC9 ж */
#define ADC_IT_EOC10                ((uint32_t)0x00000400) /*@EN: EOC10 interrupt @CN: EOC10ж */
#define ADC_IT_EOC11                ((uint32_t)0x00000800) /*@EN: EOC11 interrupt @CN: EOC11ж */
#define ADC_IT_EOC12                ((uint32_t)0x00001000) /*@EN: EOC12 interrupt @CN: EOC12ж */
#define ADC_IT_EOC13                ((uint32_t)0x00002000) /*@EN: EOC13 interrupt @CN: EOC13ж */
#define ADC_IT_EOC14                ((uint32_t)0x00004000) /*@EN: EOC14 interrupt @CN: EOC14ж */
#define ADC_IT_EOC15                ((uint32_t)0x00008000) /*@EN: EOC15 interrupt @CN: EOC15ж */
#define ADC_IT_POF0                 ((uint32_t)0x00010000) /*@EN: POF0  interrupt @CN: POF0 ж */
#define ADC_IT_POF1                 ((uint32_t)0x00020000) /*@EN: POF1  interrupt @CN: POF1 ж */
#define ADC_IT_POF2                 ((uint32_t)0x00040000) /*@EN: POF2  interrupt @CN: POF2 ж */
#define ADC_IT_POF3                 ((uint32_t)0x00080000) /*@EN: POF3  interrupt @CN: POF3 ж */
#define ADC_IT_POF4                 ((uint32_t)0x00100000) /*@EN: POF4  interrupt @CN: POF4 ж */
#define ADC_IT_POF5                 ((uint32_t)0x00200000) /*@EN: POF5  interrupt @CN: POF5 ж */
#define ADC_IT_POF6                 ((uint32_t)0x00400000) /*@EN: POF6  interrupt @CN: POF6 ж */
#define ADC_IT_POF7                 ((uint32_t)0x00800000) /*@EN: POF7  interrupt @CN: POF7 ж */
#define ADC_IT_POF8                 ((uint32_t)0x01000000) /*@EN: POF8  interrupt @CN: POF8 ж */
#define ADC_IT_POF9                 ((uint32_t)0x02000000) /*@EN: POF9  interrupt @CN: POF9 ж */
#define ADC_IT_POF10                ((uint32_t)0x04000000) /*@EN: POF10 interrupt @CN: POF10ж */
#define ADC_IT_POF11                ((uint32_t)0x08000000) /*@EN: POF11 interrupt @CN: POF11ж */
#define ADC_IT_POF12                ((uint32_t)0x10000000) /*@EN: POF12 interrupt @CN: POF12ж */
#define ADC_IT_POF13                ((uint32_t)0x20000000) /*@EN: POF13 interrupt @CN: POF13ж */
#define ADC_IT_POF14                ((uint32_t)0x40000000) /*@EN: POF14 interrupt @CN: POF14ж */
#define ADC_IT_POF15                ((uint32_t)0x80000000) /*@EN: POF15 interrupt @CN: POF15ж */

#define IS_ADC_SOC_IT(SOC_IT)       ((SOC_IT == ADC_IT_EOC0) || (SOC_IT == ADC_IT_EOC1) || \
                                     (SOC_IT == ADC_IT_EOC2) || (SOC_IT == ADC_IT_EOC3) || \
                                     (SOC_IT == ADC_IT_EOC4) || (SOC_IT == ADC_IT_EOC5) || \
                                     (SOC_IT == ADC_IT_EOC6) || (SOC_IT == ADC_IT_EOC7) || \
                                     (SOC_IT == ADC_IT_EOC8) || (SOC_IT == ADC_IT_EOC9) || \
                                     (SOC_IT == ADC_IT_EOC10) || (SOC_IT == ADC_IT_EOC11) || \
                                     (SOC_IT == ADC_IT_EOC12) || (SOC_IT == ADC_IT_EOC13) || \
                                     (SOC_IT == ADC_IT_EOC14) || (SOC_IT == ADC_IT_EOC15) || \
                                     (SOC_IT == ADC_IT_POF0) || (SOC_IT == ADC_IT_POF1) || \
                                     (SOC_IT == ADC_IT_POF2) || (SOC_IT == ADC_IT_POF3) || \
                                     (SOC_IT == ADC_IT_POF4) || (SOC_IT == ADC_IT_POF5) || \
                                     (SOC_IT == ADC_IT_POF6) || (SOC_IT == ADC_IT_POF7) || \
                                     (SOC_IT == ADC_IT_POF8) || (SOC_IT == ADC_IT_POF9) || \
                                     (SOC_IT == ADC_IT_POF10) || (SOC_IT == ADC_IT_POF11) || \
                                     (SOC_IT == ADC_IT_POF12) || (SOC_IT == ADC_IT_POF13) || \
                                     (SOC_IT == ADC_IT_POF14) || (SOC_IT == ADC_IT_POF15))
/* @end */

/* @defgroup ADC_FLAG */
#define ADC_FLAG_READY                  ((uint32_t)0x00000001) /*@EN: ADC Preparation completion flag @CN: ADC ׼ɱ־ */
#define ADC_FLAG_BUSY                   ((uint32_t)0x00000002) /*@EN: ADC busy status flag @CN: ADC æ״̬־*/
#define ADC_FLAG_PENDING0               ((uint32_t)0x00010000) /*@EN: ADC SOC 0 pending status @CN: ADC SOC 0 ״̬ */
#define ADC_FLAG_PENDING1               ((uint32_t)0x00020000) /*@EN: ADC SOC 1 pending status @CN: ADC SOC 1 ״̬ */
#define ADC_FLAG_PENDING2               ((uint32_t)0x00040000) /*@EN: ADC SOC 2 pending status @CN: ADC SOC 2 ״̬ */
#define ADC_FLAG_PENDING3               ((uint32_t)0x00080000) /*@EN: ADC SOC 3 pending status @CN: ADC SOC 3 ״̬ */
#define ADC_FLAG_PENDING4               ((uint32_t)0x00100000) /*@EN: ADC SOC 4 pending status @CN: ADC SOC 4 ״̬ */
#define ADC_FLAG_PENDING5               ((uint32_t)0x00200000) /*@EN: ADC SOC 5 pending status @CN: ADC SOC 5 ״̬ */
#define ADC_FLAG_PENDING6               ((uint32_t)0x00400000) /*@EN: ADC SOC 6 pending status @CN: ADC SOC 6 ״̬ */
#define ADC_FLAG_PENDING7               ((uint32_t)0x00800000) /*@EN: ADC SOC 7 pending status @CN: ADC SOC 7 ״̬ */
#define ADC_FLAG_PENDING8               ((uint32_t)0x01000000) /*@EN: ADC SOC 8 pending status @CN: ADC SOC 8 ״̬ */
#define ADC_FLAG_PENDING9               ((uint32_t)0x02000000) /*@EN: ADC SOC 9 pending status @CN: ADC SOC 9 ״̬ */
#define ADC_FLAG_PENDING10              ((uint32_t)0x04000000) /*@EN: ADC SOC 10 pending status @CN: ADC SOC 10 ״̬ */
#define ADC_FLAG_PENDING11              ((uint32_t)0x08000000) /*@EN: ADC SOC 11 pending status @CN: ADC SOC 11 ״̬ */
#define ADC_FLAG_PENDING12              ((uint32_t)0x10000000) /*@EN: ADC SOC 12 pending status @CN: ADC SOC 12 ״̬ */
#define ADC_FLAG_PENDING13              ((uint32_t)0x20000000) /*@EN: ADC SOC 13 pending status @CN: ADC SOC 13 ״̬ */
#define ADC_FLAG_PENDING14              ((uint32_t)0x40000000) /*@EN: ADC SOC 14 pending status @CN: ADC SOC 14 ״̬ */
#define ADC_FLAG_PENDING15              ((uint32_t)0x80000000) /*@EN: ADC SOC 15 pending status @CN: ADC SOC 15 ״̬ */
#define IS_ADC_FLAG(FLAG)               ((FLAG == ADC_FLAG_READY)     || (FLAG == ADC_FLAG_BUSY)      || \
                                         (FLAG == ADC_FLAG_PENDING0)  || (FLAG == ADC_FLAG_PENDING1)  || \
                                         (FLAG == ADC_FLAG_PENDING2)  || (FLAG == ADC_FLAG_PENDING3)  || \
                                         (FLAG == ADC_FLAG_PENDING4)  || (FLAG == ADC_FLAG_PENDING5)  || \
                                         (FLAG == ADC_FLAG_PENDING6)  || (FLAG == ADC_FLAG_PENDING7)  || \
                                         (FLAG == ADC_FLAG_PENDING8)  || (FLAG == ADC_FLAG_PENDING9)  || \
                                         (FLAG == ADC_FLAG_PENDING10) || (FLAG == ADC_FLAG_PENDING11) || \
                                         (FLAG == ADC_FLAG_PENDING12) || (FLAG == ADC_FLAG_PENDING13) || \
                                         (FLAG == ADC_FLAG_PENDING14) || (FLAG == ADC_FLAG_PENDING15))
/* @end */

/* @defgroup SOCx_TRIG */
#define SOC0_TRIG                       ((uint32_t)0x00010000) /*@EN: SOC 0 Software triggering @CN:SOC 0  */
#define SOC1_TRIG                       ((uint32_t)0x00020000) /*@EN: SOC 1 Software triggering @CN:SOC 1  */
#define SOC2_TRIG                       ((uint32_t)0x00040000) /*@EN: SOC 2 Software triggering @CN:SOC 2  */
#define SOC3_TRIG                       ((uint32_t)0x00080000) /*@EN: SOC 3 Software triggering @CN:SOC 3  */
#define SOC4_TRIG                       ((uint32_t)0x00100000) /*@EN: SOC 4 Software triggering @CN:SOC 4  */
#define SOC5_TRIG                       ((uint32_t)0x00200000) /*@EN: SOC 5 Software triggering @CN:SOC 5  */
#define SOC6_TRIG                       ((uint32_t)0x00400000) /*@EN: SOC 6 Software triggering @CN:SOC 6  */
#define SOC7_TRIG                       ((uint32_t)0x00800000) /*@EN: SOC 7 Software triggering @CN:SOC 7  */
#define SOC8_TRIG                       ((uint32_t)0x01000000) /*@EN: SOC 8 Software triggering @CN:SOC 8  */
#define SOC9_TRIG                       ((uint32_t)0x02000000) /*@EN: SOC 9 Software triggering @CN:SOC 9  */
#define SOC10_TRIG                      ((uint32_t)0x04000000) /*@EN: SOC 10 Software triggering @CN:SOC 10  */
#define SOC11_TRIG                      ((uint32_t)0x08000000) /*@EN: SOC 11 Software triggering @CN:SOC 11  */
#define SOC12_TRIG                      ((uint32_t)0x10000000) /*@EN: SOC 12 Software triggering @CN:SOC 12  */
#define SOC13_TRIG                      ((uint32_t)0x20000000) /*@EN: SOC 13 Software triggering @CN:SOC 13  */
#define SOC14_TRIG                      ((uint32_t)0x40000000) /*@EN: SOC 14 Software triggering @CN:SOC 14  */
#define SOC15_TRIG                      ((uint32_t)0x80000000) /*@EN: SOC 15 Software triggering @CN:SOC 15  */
#define IS_SOC_TRIG(SOC)                ((SOC == SOC0_TRIG)  || (SOC == SOC1_TRIG)  || \
                                         (SOC == SOC2_TRIG)  || (SOC == SOC3_TRIG)  || \
                                         (SOC == SOC4_TRIG)  || (SOC == SOC5_TRIG)  || \
                                         (SOC == SOC6_TRIG)  || (SOC == SOC7_TRIG)  || \
                                         (SOC == SOC8_TRIG)  || (SOC == SOC9_TRIG)  || \
                                         (SOC == SOC10_TRIG) || (SOC == SOC11_TRIG) || \
                                         (SOC == SOC12_TRIG) || (SOC == SOC13_TRIG) || \
                                         (SOC == SOC14_TRIG) || (SOC == SOC15_TRIG))
/* @end */

/* @defgroup SOCx_DMA */
#define SOC0_DMA                       ((uint32_t)0x00000001) /*@EN: SOC 0 DMA transmission enable @CN: SOC 0 DMA ʹ */
#define SOC1_DMA                       ((uint32_t)0x00000002) /*@EN: SOC 1 DMA transmission enable @CN: SOC 1 DMA ʹ */
#define SOC2_DMA                       ((uint32_t)0x00000004) /*@EN: SOC 2 DMA transmission enable @CN: SOC 2 DMA ʹ */
#define SOC3_DMA                       ((uint32_t)0x00000008) /*@EN: SOC 3 DMA transmission enable @CN: SOC 3 DMA ʹ */
#define SOC4_DMA                       ((uint32_t)0x00000010) /*@EN: SOC 4 DMA transmission enable @CN: SOC 4 DMA ʹ */
#define SOC5_DMA                       ((uint32_t)0x00000020) /*@EN: SOC 5 DMA transmission enable @CN: SOC 5 DMA ʹ */
#define SOC6_DMA                       ((uint32_t)0x00000040) /*@EN: SOC 6 DMA transmission enable @CN: SOC 6 DMA ʹ */
#define SOC7_DMA                       ((uint32_t)0x00000080) /*@EN: SOC 7 DMA transmission enable @CN: SOC 7 DMA ʹ */
#define SOC8_DMA                       ((uint32_t)0x00000100) /*@EN: SOC 8 DMA transmission enable @CN: SOC 8 DMA ʹ */
#define SOC9_DMA                       ((uint32_t)0x00000200) /*@EN: SOC 9 DMA transmission enable @CN: SOC 9 DMA ʹ */
#define SOC10_DMA                      ((uint32_t)0x00000400) /*@EN: SOC 10 DMA transmission enable @CN: SOC 10 DMA ʹ */
#define SOC11_DMA                      ((uint32_t)0x00000800) /*@EN: SOC 11 DMA transmission enable @CN: SOC 11 DMA ʹ */
#define SOC12_DMA                      ((uint32_t)0x00001000) /*@EN: SOC 12 DMA transmission enable @CN: SOC 12 DMA ʹ */
#define SOC13_DMA                      ((uint32_t)0x00002000) /*@EN: SOC 13 DMA transmission enable @CN: SOC 13 DMA ʹ */
#define SOC14_DMA                      ((uint32_t)0x00004000) /*@EN: SOC 14 DMA transmission enable @CN: SOC 14 DMA ʹ */
#define SOC15_DMA                      ((uint32_t)0x00008000) /*@EN: SOC 15 DMA transmission enable @CN: SOC 15 DMA ʹ */
#define IS_SOC_DMA(SOC)                ((SOC == SOC0_DMA)  || (SOC == SOC1_DMA)  || \
                                        (SOC == SOC2_DMA)  || (SOC == SOC3_DMA)  || \
                                        (SOC == SOC4_DMA)  || (SOC == SOC5_DMA)  || \
                                        (SOC == SOC6_DMA)  || (SOC == SOC7_DMA)  || \
                                        (SOC == SOC8_DMA)  || (SOC == SOC9_DMA)  || \
                                        (SOC == SOC10_DMA) || (SOC == SOC11_DMA) || \
                                        (SOC == SOC12_DMA) || (SOC == SOC13_DMA) || \
                                        (SOC == SOC14_DMA) || (SOC == SOC15_DMA))
/* @end */

/* @defgroup ADCIN */
#define ADCIN_ST                        ((uint32_t)0x00000000) /*@EN: Internal ST channel Only used when ST function is enabled@CN: ڲ ST ͨ STܿʱʹ*/
#define ADCIN_SV                        ((uint32_t)0x00000000) /*@EN: Internal SV channel Only used when SV function is enabled@CN: ڲ SV ͨ SVܿʱʹ*/
#define ADCIN_0_A                       ((uint32_t)0x00000000) /*@EN: External ADC_IN 0 @CN: ⲿADC_IN 0 */
#define ADCIN_1_A                       ((uint32_t)0x00000001) /*@EN: External ADC_IN 1 @CN: ⲿADC_IN 1 */
#define ADCIN_2_A                       ((uint32_t)0x00000002) /*@EN: External ADC_IN 2 @CN: ⲿADC_IN 2 */
#define ADCIN_3_A                       ((uint32_t)0x00000003) /*@EN: External ADC_IN 3 @CN: ⲿADC_IN 3 */
#define ADCIN_4_A                       ((uint32_t)0x00000004) /*@EN: External ADC_IN 4 @CN: ⲿADC_IN 4 */
#define ADCIN_5_A                       ((uint32_t)0x00000005) /*@EN: External ADC_IN 5 @CN: ⲿADC_IN 5 */
#define ADCIN_6_A                       ((uint32_t)0x00000006) /*@EN: External ADC_IN 6 @CN: ⲿADC_IN 6 */
#define ADCIN_7_A                       ((uint32_t)0x00000007) /*@EN: External ADC_IN 7 @CN: ⲿADC_IN 7 */
#define ADCIN_8_B                       ((uint32_t)0x00000008) /*@EN: External ADC_IN 8 @CN: ⲿADC_IN 8 */
#define ADCIN_9_B                       ((uint32_t)0x00000009) /*@EN: External ADC_IN 9 @CN: ⲿADC_IN 9 */
#define ADCIN_10_B                      ((uint32_t)0x0000000A) /*@EN: External ADC_IN 10 @CN: ⲿADC_IN 10 */
#define ADCIN_11_B                      ((uint32_t)0x0000000B) /*@EN: External ADC_IN 11 @CN: ⲿADC_IN 11 */
#define ADCIN_12_B                      ((uint32_t)0x0000000C) /*@EN: External ADC_IN 12 @CN: ⲿADC_IN 12 */
#define ADCIN_13_B                      ((uint32_t)0x0000000D) /*@EN: External ADC_IN 13 @CN: ⲿADC_IN 13 */
#define ADCIN_14_B                      ((uint32_t)0x0000000E) /*@EN: External ADC_IN 14 @CN: ⲿADC_IN 14 */
#define ADCIN_15_B                      ((uint32_t)0x0000000F) /*@EN: External ADC_IN 15 @CN: ⲿADC_IN 15 */
/* @end */

/* @defgroup SOC_SAMP */
#define SOC_SAMP_SINGLE                 ((uint32_t)0x00000000)  /* @EN: SOC [x] is single sampling @CN: SOC[x]Ϊ */
#define SOC_SAMP_DOUBLE                 ((uint32_t)0x00000100)  /* @EN: SOC [x] is double sampling @CN: SOC[x]Ϊ˫ */
/* @end */

/* @defgroup AWDx_SEL */
#define AWD0                            ((uint8_t)0x01)  /* @EN: Analog Watchdog 0 @CN: ģ⿴Ź0*/
#define AWD1                            ((uint8_t)0x02)  /* @EN: Analog Watchdog 1 @CN: ģ⿴Ź1 */
/* @end */

/* @defgroup ADC_AWD_Channel */
#define ADC_AWD_Channel_0               ((uint32_t)0x00000001) /*@EN: AWD monitoring channel 0 @CN: AWDͨ  0 */
#define ADC_AWD_Channel_1               ((uint32_t)0x00000002) /*@EN: AWD monitoring channel 1 @CN: AWDͨ  1 */
#define ADC_AWD_Channel_2               ((uint32_t)0x00000004) /*@EN: AWD monitoring channel 2 @CN: AWDͨ  2 */
#define ADC_AWD_Channel_3               ((uint32_t)0x00000008) /*@EN: AWD monitoring channel 3 @CN: AWDͨ  3 */
#define ADC_AWD_Channel_4               ((uint32_t)0x00000010) /*@EN: AWD monitoring channel 4 @CN: AWDͨ  4 */
#define ADC_AWD_Channel_5               ((uint32_t)0x00000020) /*@EN: AWD monitoring channel 5 @CN: AWDͨ  5 */
#define ADC_AWD_Channel_6               ((uint32_t)0x00000040) /*@EN: AWD monitoring channel 6 @CN: AWDͨ  6 */
#define ADC_AWD_Channel_7               ((uint32_t)0x00000080) /*@EN: AWD monitoring channel 7 @CN: AWDͨ  7 */
#define ADC_AWD_Channel_8               ((uint32_t)0x00000100) /*@EN: AWD monitoring channel 8 @CN: AWDͨ  8 */
#define ADC_AWD_Channel_9               ((uint32_t)0x00000200) /*@EN: AWD monitoring channel 9 @CN: AWDͨ  9 */
#define ADC_AWD_Channel_10              ((uint32_t)0x00000400) /*@EN: AWD monitoring channel 10 @CN: AWDͨ 10 */
#define ADC_AWD_Channel_11              ((uint32_t)0x00000800) /*@EN: AWD monitoring channel 11 @CN: AWDͨ 11 */
#define ADC_AWD_Channel_12              ((uint32_t)0x00001000) /*@EN: AWD monitoring channel 12 @CN: AWDͨ 12 */
#define ADC_AWD_Channel_13              ((uint32_t)0x00002000) /*@EN: AWD monitoring channel 13 @CN: AWDͨ 13 */
#define ADC_AWD_Channel_14              ((uint32_t)0x00004000) /*@EN: AWD monitoring channel 14 @CN: AWDͨ 14 */
#define ADC_AWD_Channel_15              ((uint32_t)0x00008000) /*@EN: AWD monitoring channel 15 @CN: AWDͨ 15 */
#define IS_ADC_AWD_Channel(ADC_AWD_Channel) ((ADC_AWD_Channel == ADC_AWD_Channel_0) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_1) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_2) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_3) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_4) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_5) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_6) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_7) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_8) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_9) ||  \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_10) || \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_11) || \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_12) || \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_13) || \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_14) || \
                                             (ADC_AWD_Channel == ADC_AWD_Channel_15))
/* ========================================================================== */
/* ================             ADC REGISTER BIT               ============== */
/* ========================================================================== */

/* =================================  SR  =================================== */
#define ADC_SR_PENDING_Pos                        (16UL)                       /*!< PENDING (Bit 16)                    */
#define ADC_SR_PENDING_Msk                        (0xFFFF0000UL)               /*!< PENDING (Bitfield-Mask: 0xFFFF)     */
#define ADC_SR_BUSY_Pos                           (1UL)                        /*!< BUSY    (Bit 1)                     */
#define ADC_SR_BUSY_Msk                           (0x00000002UL)               /*!< BUSY    (Bitfield-Mask: 0x1)        */
#define ADC_SR_READY_Pos                          (0UL)                        /*!< READY   (Bit 0)                     */
#define ADC_SR_READY_Msk                          (0x00000001UL)               /*!< READY   (Bitfield-Mask: 0x1)        */
/* =================================  ISR  ================================== */
#define ADC_ISR_SDOF_Pos                          (16UL)                       /*!< SDOF (Bit 16)                       */
#define ADC_ISR_SDOF_Msk                          (0xFFFF0000UL)               /*!< SDOF (Bitfield-Mask: 0xFFFF)        */
#define ADC_ISR_DOF_Pos                           (15UL)                       /*!< DOF  (Bit 15)                       */
#define ADC_ISR_DOF_Msk                           (0x00008000UL)               /*!< DOF  (Bitfield-Mask: 0x1)           */
#define ADC_ISR_AWD_Pos                           (0UL)                        /*!< AWD  (Bit 0)                        */
#define ADC_ISR_AWD_Msk                           (0x00000003UL)               /*!< AWD  (Bitfield-Mask: 0x3)           */
/* =================================  IER  ================================== */
#define ADC_IER_DOFIE_Pos                         (15UL)                       /*!< DOFIE (Bit 15)                      */
#define ADC_IER_DOFIE_Msk                         (0x00008000UL)               /*!< DOFIE (Bitfield-Mask: 0x1)          */
#define ADC_IER_AWDIE_Pos                         (0UL)                        /*!< AWDIE (Bit 0)                       */
#define ADC_IER_AWDIE_Msk                         (0x00000003UL)               /*!< AWDIE (Bitfield-Mask: 0x3)          */
/* ================================ SOC ISR  ================================ */
#define ADC_SOC_ISR_POF_Pos                       (16UL)                       /*!< POF (Bit 16)                        */
#define ADC_SOC_ISR_POF_Msk                       (0xFFFF0000UL)               /*!< POF (Bitfield-Mask: 0xFFFF)         */
#define ADC_SOC_ISR_EOC_Pos                       (0UL)                        /*!< EOC (Bit 0)                         */
#define ADC_SOC_ISR_EOC_Msk                       (0x0000FFFFUL)               /*!< EOC (Bitfield-Mask: 0xFFFF)         */
/* ================================ SOC IER  ================================ */
#define ADC_SOC_IER_POFIE_Pos                     (16UL)                       /*!< POFIE (Bit 16)                      */
#define ADC_SOC_IER_POFIE_Msk                     (0xFFFF0000UL)               /*!< POFIE (Bitfield-Mask: 0xFFFF)       */
#define ADC_SOC_IER_EOCIE_Pos                     (0UL)                        /*!< EOCIE (Bit 0)                       */
#define ADC_SOC_IER_EOCIE_Msk                     (0x0000FFFFUL)               /*!< EOCIE (Bitfield-Mask: 0xFFFF)       */
/* =================================   CR   ================================= */
#define ADC_CR_SW_Pos                             (16UL)                       /*!< SW (Bit 16)                         */
#define ADC_CR_SW_Msk                             (0xFFFF0000UL)               /*!< SW (Bitfield-Mask: 0xFFFF)          */
#define ADC_CR_STOP_Pos                           (3UL)                        /*!< STOP (Bit 3)                        */
#define ADC_CR_STOP_Msk                           (0x00000008UL)               /*!< STOP (Bitfield-Mask: 0x1)           */
#define ADC_CR_START_Pos                          (2UL)                        /*!< START (Bit 2)                       */
#define ADC_CR_START_Msk                          (0x00000004UL)               /*!< START (Bitfield-Mask: 0x1)          */
#define ADC_CR_DIS_Pos                            (1UL)                        /*!< DIS (Bit 1)                         */
#define ADC_CR_DIS_Msk                            (0x00000002UL)               /*!< DIS (Bitfield-Mask: 0x1)            */
#define ADC_CR_EN_Pos                             (0UL)                        /*!< EN (Bit 0)                          */
#define ADC_CR_EN_Msk                             (0x00000001UL)               /*!< EN (Bitfield-Mask: 0x1)             */
/* ==================================  CFGR  ================================ */
#define ADC_CFGR_CKMODE_Pos                       (27UL)                       /*!< CKMODE (Bit 27)                     */
#define ADC_CFGR_CKMODE_Msk                       (0xF8000000UL)               /*!< CKMODE (Bitfield-Mask: 0x1F)        */
#define ADC_CFGR_MD2_Pos                          (19UL)                       /*!< MD2 (Bit 19)                        */
#define ADC_CFGR_MD2_Msk                          (0x00080000UL)               /*!< MD2 (Bitfield-Mask: 0x1)            */
#define ADC_CFGR_VREF_Pos                         (18UL)                       /*!< VREF (Bit 18)                       */
#define ADC_CFGR_VREF_Msk                         (0x00040000UL)               /*!< VREF (Bitfield-Mask: 0x1)           */
#define ADC_CFGR_VFS_Pos                          (17UL)                       /*!< VFS (Bit 17)                        */
#define ADC_CFGR_VFS_Msk                          (0x00020000UL)               /*!< VFS (Bitfield-Mask: 0x1)            */
#define ADC_CFGR_LP_Pos                           (16UL)                       /*!< LP (Bit 16)                         */
#define ADC_CFGR_LP_Msk                           (0x00010000UL)               /*!< LP (Bitfield-Mask: 0x1)             */
#define ADC_CFGR_SVS_Pos                          (15UL)                       /*!< SVS (Bit 15)                        */
#define ADC_CFGR_SVS_Msk                          (0x00008000UL)               /*!< SVS (Bitfield-Mask: 0x1)            */
#define ADC_CFGR_SV_Pos                           (14UL)                       /*!< SV (Bit 14)                         */
#define ADC_CFGR_SV_Msk                           (0x00004000UL)               /*!< SV (Bitfield-Mask: 0x1)             */
#define ADC_CFGR_STS_Pos                          (13UL)                       /*!< STS (Bit 13)                        */
#define ADC_CFGR_STS_Msk                          (0x00002000UL)               /*!< STS (Bitfield-Mask: 0x1)            */
#define ADC_CFGR_ST_Pos                           (12UL)                       /*!< ST (Bit 12)                         */
#define ADC_CFGR_ST_Msk                           (0x00001000UL)               /*!< ST (Bitfield-Mask: 0x1f)            */
#define ADC_CFGR_OVF_Pos                          (1UL)                        /*!< OVF (Bit 1)                         */
#define ADC_CFGR_OVF_Msk                          (0x00000002UL)               /*!< OVF (Bitfield-Mask: 0x1)            */
#define ADC_CFGR_ALIGN_Pos                        (0UL)                        /*!< ALIGN (Bit 0)                       */
#define ADC_CFGR_ALIGN_Msk                        (0x00000001UL)               /*!< ALIGN (Bitfield-Mask: 0x1)          */
/* ===================================  SMPR  =============================== */
#define ADC_SMPR_STSPW_Pos                        (20UL)                       /*!< STSPW (Bit 20)                      */
#define ADC_SMPR_STSPW_Msk                        (0x00300000UL)               /*!< STSPW (Bitfield-Mask: 0x3)          */
#define ADC_SMPR_CC_Pos                           (12UL)                       /*!< CC (Bit 12)                         */
#define ADC_SMPR_CC_Msk                           (0x0000F000UL)               /*!< CC (Bitfield-Mask: 0xF)             */
#define ADC_SMPR_CPW_Pos                          (8UL)                        /*!< CPW (Bit 8)                         */
#define ADC_SMPR_CPW_Msk                          (0x00000300UL)               /*!< CPW (Bitfield-Mask: 0x3)            */
#define ADC_SMPR_SPW_Pos                          (4UL)                        /*!< SPW (Bit 4)                         */
#define ADC_SMPR_SPW_Msk                          (0x000000F0UL)               /*!< SPW (Bitfield-Mask: 0xF)            */
#define ADC_SMPR_APW_Pos                          (0UL)                        /*!< APW (Bit 0)                         */
#define ADC_SMPR_APW_Msk                          (0x00000007UL)               /*!< APW (Bitfield-Mask: 0x7)            */
/* ===================================  TRIM  =============================== */
#define ADC_TRIM_VOS_TRIM_Pos                     (16UL)                       /*!< VOS_TRIM (Bit 16)                   */
#define ADC_TRIM_VOS_TRIM_Msk                     (0x00FF0000UL)               /*!< VOS_TRIM (Bitfield-Mask: 0xFF)      */
#define ADC_TRIM_GAIN_TRIM_Pos                    (0UL)                        /*!< GAIN_TRIM (Bit 0)                   */
#define ADC_TRIM_GAIN_TRIM_Msk                    (0x000000FFUL)               /*!< GAIN_TRIM (Bitfield-Mask: 0xFF)     */
/* ===================================  ANA  ================================ */
#define ADC_ANA_PD_ADC_Pos                        (15UL)                       /*!< PD_ADC (Bit 15)                     */
#define ADC_ANA_PD_ADC_Msk                        (0x00008000UL)               /*!< PD_ADC (Bitfield-Mask: 0x1)         */
#define ADC_ANA_PD_REF_Pos                        (14UL)                       /*!< PD_REF (Bit 14)                     */
#define ADC_ANA_PD_REF_Msk                        (0x00004000UL)               /*!< PD_REF (Bitfield-Mask: 0x1)         */
#define ADC_ANA_PD_SHB_Pos                        (13UL)                       /*!< PD_SHB (Bit 13)                     */
#define ADC_ANA_PD_SHB_Msk                        (0x00002000UL)               /*!< PD_SHB (Bitfield-Mask: 0x1)         */
#define ADC_ANA_PD_SHA_Pos                        (12UL)                       /*!< PD_SHA (Bit 12)                     */
#define ADC_ANA_PD_SHA_Msk                        (0x00001000UL)               /*!< PD_SHA (Bitfield-Mask: 0x1)         */
/* ===================================  DMACFG  ============================= */
#define ADC_DMACFG_DEN_Pos                        (16UL)                       /*!< DEN (Bit 16)                        */
#define ADC_DMACFG_DEN_Msk                        (0xFFFF0000UL)               /*!< DEN (Bitfield-Mask: 0xFFFF)         */
#define ADC_DMACFG_DREQ_Pos                       (0UL)                        /*!< DREQ (Bit 0)                        */
#define ADC_DMACFG_DREQ_Msk                       (0x0000FFFFUL)               /*!< DREQ (Bitfield-Mask: 0xFFFF)        */
/* ====================================  DADDR  ============================= */
#define ADC_DADDR_DMA_DATA_Pos                    (0UL)                        /*!< DMA_DATA (Bit 0)                    */
#define ADC_DADDR_DMA_DATA_Msk                    (0x0000FFFFUL)               /*!< DMA_DATA (Bitfield-Mask: 0xFFFF)    */
/* =================================  AWD[x]CFGR  =========================== */
#define ADC_AWD_CFGR_AWDSEL_Pos                   (0UL)                        /*!< CFGR_AWDSEL (Bit 0)                 */
#define ADC_AWD_CFGR_AWDSEL_Msk                   (0x0000FFFFUL)               /*!< CFGR_AWDSEL (Bitfield-Mask: 0xFFFF) */
/* ==================================  AWD[x]TR  ============================ */
#define ADC_AWD_TR_HT_Pos                         (16UL)                       /*!< TR_HT (Bit 16)                      */
#define ADC_AWD_TR_HT_Msk                         (0x0FFF0000UL)               /*!< TR_HT (Bitfield-Mask: 0xFFF)        */
#define ADC_AWD_TR_LT_Pos                         (0UL)                        /*!< TR_LT (Bit 0)                       */
#define ADC_AWD_TR_LT_Msk                         (0x00000FFFUL)               /*!< TR_LT (Bitfield-Mask: 0xFFF)        */
/* ==================================  EXTCFGR0  ============================ */
#define ADC_EXTCFGR0_DLY3_Pos                     (24UL)                       /*!< DLY3 (Bit 24)                       */
#define ADC_EXTCFGR0_DLY3_Msk                     (0xFF000000UL)               /*!< DLY3 (Bitfield-Mask: 0xFF)          */
#define ADC_EXTCFGR0_DLY2_Pos                     (16UL)                       /*!< DLY2 (Bit 16)                       */
#define ADC_EXTCFGR0_DLY2_Msk                     (0x00FF0000UL)               /*!< DLY2 (Bitfield-Mask: 0xFF)          */
#define ADC_EXTCFGR0_DLY1_Pos                     (8UL)                        /*!< DLY1 (Bit 8)                        */
#define ADC_EXTCFGR0_DLY1_Msk                     (0x0000FF00UL)               /*!< DLY1 (Bitfield-Mask: 0xFF)          */
#define ADC_EXTCFGR0_DLY0_Pos                     (0UL)                        /*!< DLY0 (Bit 0)                        */
#define ADC_EXTCFGR0_DLY0_Msk                     (0x000000FFUL)               /*!< DLY0 (Bitfield-Mask: 0xFF)          */
/* ==================================  SOC[x]CFG  =========================== */
#define ADC_CFG_CON_Pos                           (8UL)                        /*!< CON (Bit 8)                         */
#define ADC_CFG_CON_Msk                           (0x00000100UL)               /*!< CON (Bitfield-Mask: 0x1)            */
#define ADC_CFG_FP_Pos                            (7UL)                        /*!< FP (Bit 7)                          */
#define ADC_CFG_FP_Msk                            (0x00000080UL)               /*!< FP (Bitfield-Mask: 0x1)             */
#define ADC_CFG_TRIGSEL_Pos                       (4UL)                        /*!< TRIGSEL (Bit 4)                     */
#define ADC_CFG_TRIGSEL_Msk                       (0x00000070UL)               /*!< TRIGSEL (Bitfield-Mask: 0x7)        */
#define ADC_CFG_CHSEL_Pos                         (0UL)                        /*!< CHSEL (Bit 0)                       */
#define ADC_CFG_CHSEL_Msk                         (0x0000000FUL)               /*!< CHSEL (Bitfield-Mask: 0xF)          */
/* ======================================= DR  ============================== */
#define ADC_DR_DATA_Pos                           (0UL)                        /*!< DATA (Bit 0)                        */
#define ADC_DR_DATA_Msk                           (0x0000FFFFUL)               /*!< DATA (Bitfield-Mask: 0xFFFF)        */

/* ========================================================================== */
/* ================                  END                      =============== */
/* ========================================================================== */

void ADC_DeInit(ADC_TypeDef *ADCx);
void ADC_Cmd(ADC_TypeDef *ADCx, FunctionalState State);
void ADC_StructInit(ADC_InitTypeDef *ADC_InitStruct);
void ADC_Init(ADC_TypeDef *ADCx, ADC_InitTypeDef *ADC_InitStruct);
void ADC_ExternTrigger_Config (ADC_TypeDef *ADCx, uint8_t ExternTigger, uint8_t Delay);
void ADC_StopOfConversion(ADC_TypeDef *ADCx);
void ADC_StartOfConversion(ADC_TypeDef *ADCx);
void ADC_CloseAdc(ADC_TypeDef *ADCx);
uint16_t ADC_GetConversionValue(ADC_TypeDef *ADCx ,uint32_t SOCx);
uint16_t ADC_GetDMAValue(ADC_TypeDef *ADCx );
void TrigSelect_Config(ADC_TypeDef *ADCx, uint32_t TriggerSelect ,uint32_t AdcExtsel, uint32_t Triggermode);
void ADC_DMA_Cmd(ADC_TypeDef *ADCx,uint32_t SOCx_DMA,FunctionalState NewState);
void ADC_DMA_TransReq(ADC_TypeDef *ADCx,uint32_t SOCx_DMA,FunctionalState NewState);
void ADC_ITConfig(ADC_TypeDef* ADCx, uint32_t ADC_IT, FunctionalState NewState);
void ADC_SOC_ITConfig(ADC_TypeDef* ADCx, uint32_t ADC_SOC_IT, FunctionalState NewState);
FlagStatus ADC_GetFlagStatus(ADC_TypeDef *ADCx, uint32_t ADC_FLAG);
ITStatus ADC_GetITStatus(ADC_TypeDef *ADCx, uint32_t ADC_IT);
ITStatus ADC_GetSOCITStatus(ADC_TypeDef *ADCx, uint32_t ADC_SOC_IT);
void ADC_ClearITPendingBit(ADC_TypeDef* ADCx, uint32_t ADC_IT);
void ADC_ClearSOCITPendingBit(ADC_TypeDef* ADCx, uint32_t ADC_SOC_IT);
void ADC_Trim(ADC_TypeDef *ADCx, uint8_t VosTrim, uint8_t GainTrim);
void ADC_SoftTrig(ADC_TypeDef *ADCx, uint32_t SofTrig);
void ADC_SOC_Config_New(ADC_TypeDef *ADCx, uint32_t SOCx, uint32_t TriggerSource, uint32_t ADCIN, uint32_t SOC_SAMP);
void ADC_SOC_Config(ADC_TypeDef *ADCx, uint32_t SOCx, uint32_t TriggerSource ,uint32_t ADCIN, uint32_t SOC_SAMP);
void ADC_AnalogWatchdogThresholdsConfg(ADC_TypeDef *ADCx, uint8_t AWDx_SEL, uint16_t HighThreshold, uint16_t LowThreshold);
void ADC_AnalogWatchdogSingleChannelConfig(ADC_TypeDef *ADCx, uint8_t AWDx_SEL, uint32_t ADC_AWD_Channel, FunctionalState NewState);

void ADC_Vref_Cmd(ADC_TypeDef *ADCx, FunctionalState State);
void ADC_ForExternVref_Cmd(ADC_TypeDef *ADCx);

int ADC_GetVTStemperature(int ADCValueNow, uint32_t Vref);

#ifdef __cplusplus
}
#endif

#endif // __LCM32F06X_ADC_H__
